package ru.yandex.direct.core.entity.projectparam.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamCondition;
import ru.yandex.direct.core.entity.pricepackage.model.ProjectParamConjunction;
import ru.yandex.direct.dbschema.ppcdict.tables.records.ProjectParamConditionsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppcdict.tables.ProjectParamConditions.PROJECT_PARAM_CONDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class ProjectParamConditionRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ProjectParamCondition> markupConditionMapper;

    @Autowired
    public ProjectParamConditionRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.markupConditionMapper = createProjectParamConditionMapper();
    }

    /**
     * Возвращает список всех условий наценок за таргеты
     */
    public List<ProjectParamCondition> getAllProjectParamConditions() {
        return dslContextProvider.ppcdict()
                .select(markupConditionMapper.getFieldsToRead())
                .from(PROJECT_PARAM_CONDITIONS)
                .fetch(markupConditionMapper::fromDb);
    }

    /**
     * Возвращает список условий наценок за таргеты по condition_id
     */
    public List<ProjectParamCondition> getProjectParamConditionsByConditionIds(List<Long> conditionIds) {
        return dslContextProvider.ppcdict()
                .select(markupConditionMapper.getFieldsToRead())
                .from(PROJECT_PARAM_CONDITIONS)
                .where(PROJECT_PARAM_CONDITIONS.CONDITION_ID.in(conditionIds))
                .fetch(markupConditionMapper::fromDb);
    }

    /**
     * Добавление новых и изменение существующих записей в таблице PROJECT_PARAM_CONDITIONS
     */
    public int updateProjectParamConditions(Collection<ProjectParamCondition> markupConditions) {
        if (markupConditions.isEmpty()) {
            return 0;
        }

        InsertHelper<ProjectParamConditionsRecord> helper = new InsertHelper<>(dslContextProvider.ppcdict(), PROJECT_PARAM_CONDITIONS);
        helper.addAll(markupConditionMapper, markupConditions);

        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(PROJECT_PARAM_CONDITIONS.CONDITION_ID, MySQLDSL.values(PROJECT_PARAM_CONDITIONS.CONDITION_ID))
                    .set(PROJECT_PARAM_CONDITIONS.IS_ARCHIVED, MySQLDSL.values(PROJECT_PARAM_CONDITIONS.IS_ARCHIVED))
                    .set(PROJECT_PARAM_CONDITIONS.CONDITION_JSON, MySQLDSL.values(PROJECT_PARAM_CONDITIONS.CONDITION_JSON))
                    .set(PROJECT_PARAM_CONDITIONS.DESCRIPTION, MySQLDSL.values(PROJECT_PARAM_CONDITIONS.DESCRIPTION));
        }
        return helper.executeIfRecordsAdded();
    }

    public List<Long> addProjectParamConditions(List<ProjectParamCondition> markupConditions) {
        DSLContext dslContext = dslContextProvider.ppcdict();

        return InsertHelper.addModelsAndReturnIds(dslContext, PROJECT_PARAM_CONDITIONS,
                markupConditionMapper, markupConditions, PROJECT_PARAM_CONDITIONS.CONDITION_ID);
    }

    /**
     * Поиск по части названия
     */
    public List<ProjectParamCondition> findProjectParamConditionsByNamePart(String markupConditionDescriptionPrefix,
            int limit, int offset) {
        return dslContextProvider.ppcdict()
                .select(markupConditionMapper.getFieldsToRead())
                .from(PROJECT_PARAM_CONDITIONS)
                .where(PROJECT_PARAM_CONDITIONS.DESCRIPTION.contains(markupConditionDescriptionPrefix))
                .orderBy(PROJECT_PARAM_CONDITIONS.DESCRIPTION)
                .limit(limit)
                .offset(offset)
                .fetch(markupConditionMapper::fromDb);
    }

    public void deleteProjectParamConditions(Collection<Long> conditionIds) {
        if (conditionIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppcdict()
                .deleteFrom(PROJECT_PARAM_CONDITIONS)
                .where(PROJECT_PARAM_CONDITIONS.CONDITION_ID.in(conditionIds))
                .execute();
    }

    private static JooqMapperWithSupplier<ProjectParamCondition> createProjectParamConditionMapper() {
        return JooqMapperWithSupplierBuilder.builder(ProjectParamCondition::new)
                .map(property(ProjectParamCondition.ID, PROJECT_PARAM_CONDITIONS.CONDITION_ID))
                .map(convertibleProperty(ProjectParamCondition.IS_ARCHIVED, PROJECT_PARAM_CONDITIONS.IS_ARCHIVED,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong))
                .map(property(ProjectParamCondition.DESCRIPTION, PROJECT_PARAM_CONDITIONS.DESCRIPTION))
                .map(convertibleProperty(ProjectParamCondition.CONJUNCTIONS, PROJECT_PARAM_CONDITIONS.CONDITION_JSON,
                        json -> List.of(JsonUtils.fromJson(json, ProjectParamConjunction[].class)),
                        JsonUtils::toJson))
                .build();
    }
}
