package ru.yandex.direct.core.entity.promocodes.repository;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JavaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.promocodes.model.CampPromocodes;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeInfo;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppc.tables.CampPromocodes.CAMP_PROMOCODES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class CampPromocodesRepository {

    private static final JavaType CAMP_PROMOCODES_TYPE = JsonUtils.getTypeFactory()
            .constructCollectionType(List.class, PromocodeInfo.class);

    private final JooqMapperWithSupplier<CampPromocodes> jooqMapper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampPromocodesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(CampPromocodes::new)
                .map(property(CampPromocodes.CAMPAIGN_ID, CAMP_PROMOCODES.CID))
                .map(convertibleProperty(CampPromocodes.PROMOCODES, CAMP_PROMOCODES.PROMOCODES_INFO,
                        CampPromocodesRepository::promocodesInfoFromDb, JsonUtils::toJsonNullable))
                .map(property(CampPromocodes.RESTRICTED_DOMAIN, CAMP_PROMOCODES.RESTRICTED_DOMAIN))
                .map(property(CampPromocodes.LAST_CHANGE, CAMP_PROMOCODES.LAST_CHANGE))
                .build();
    }

    private static List<PromocodeInfo> promocodesInfoFromDb(@Nullable String promocodeInfo) {
        return promocodeInfo == null ? null : JsonUtils.fromJson(promocodeInfo, CAMP_PROMOCODES_TYPE);
    }

    /**
     * Получить промокодные сведения по кампании.
     * <p>
     * Ограничивающий домен в промокодных сведения - в нижнем регистре, без незначащего префикса www, в юникоде
     * (не в xn-- нотации).
     * Это нужно учитывать при сравнении его с доменами из объявлений.
     *
     * @param shard      шард
     * @param campaignId id кампании
     * @return прокомодные сведения или {@code null}
     */
    @Nullable
    public CampPromocodes getCampaignPromocodes(int shard, long campaignId) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(CAMP_PROMOCODES)
                .where(CAMP_PROMOCODES.CID.eq(campaignId))
                .fetchOne(jooqMapper::fromDb);
    }

    /**
     * Создать в БД промокодные сведения по кампании
     *
     * @param shard          шард
     * @param campPromocodes промокодные сведения по кампании
     */
    public void addCampaignPromocodes(int shard, CampPromocodes campPromocodes) {
        new InsertHelper<>(dslContextProvider.ppc(shard), CAMP_PROMOCODES).add(jooqMapper, campPromocodes).execute();
    }

    /**
     * Обновить список действующих промокодов по кампании
     *
     * @param shard      шард
     * @param campaignId id кампании
     * @param promocodes новый список промокодов
     */
    public void updateCampaignPromocodesList(int shard, long campaignId, List<PromocodeInfo> promocodes) {
        dslContextProvider.ppc(shard)
                .update(CAMP_PROMOCODES)
                .set(CAMP_PROMOCODES.PROMOCODES_INFO, JsonUtils.toJsonNullable(promocodes))
                .set(CAMP_PROMOCODES.LAST_CHANGE, LocalDateTime.now())
                .where(CAMP_PROMOCODES.CID.eq(campaignId))
                .execute();
    }

    /**
     * Удалить промокодные сведния о кампании
     *
     * @param shard      шард
     * @param campaignId id кампании
     */
    public void deleteCampaignPromocodes(int shard, long campaignId) {
        dslContextProvider.ppc(shard)
                .delete(CAMP_PROMOCODES)
                .where(CAMP_PROMOCODES.CID.eq(campaignId))
                .execute();
    }
}
