package ru.yandex.direct.core.entity.promocodes.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.promocodes.model.PromocodeClientDomain;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.PromocodeDomains.PROMOCODE_DOMAINS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class PromocodeDomainsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<PromocodeClientDomain> mapper;

    @Autowired
    public PromocodeDomainsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = JooqMapperWithSupplierBuilder.builder(PromocodeClientDomain::new)
                .map(property(PromocodeClientDomain.PROMOCODE, PROMOCODE_DOMAINS.PROMOCODE))
                .map(property(PromocodeClientDomain.DOMAIN, PROMOCODE_DOMAINS.DOMAIN))
                .map(convertibleProperty(PromocodeClientDomain.CLIENT_ID, PROMOCODE_DOMAINS.CLIENT_ID,
                        ClientId::fromLong, ClientId::asLong))
                .build();
    }

    /**
     * Прочитать информацию о промокодах
     *
     * @param promocodes — колекция промокодов (их текстов) без дефисов
     * @return словарь из текста промокода в записи о промокоде без даты импорта
     */
    public Map<String, PromocodeClientDomain> getPromocodeDomains(Collection<String> promocodes) {
        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(PROMOCODE_DOMAINS)
                .where(PROMOCODE_DOMAINS.PROMOCODE.in(promocodes))
                .fetchMap(PROMOCODE_DOMAINS.PROMOCODE, mapper::fromDb);
    }

    /**
     * Удалить промокоды, записанные раньше переданного времени
     *
     * @param time — время, с которого промокоды нужно оставить
     * @return число удалённых записей
     */
    public int deleteOldPromocodeDomains(LocalDateTime time) {
        return dslContextProvider.ppcdict()
                .deleteFrom(PROMOCODE_DOMAINS)
                .where(PROMOCODE_DOMAINS.IMPORT_TIME.lt(time))
                .execute();
    }

    public void addPromocodeDomain(PromocodeClientDomain record) {
        new InsertHelper<>(dslContextProvider.ppcdict(), PROMOCODE_DOMAINS)
                .add(mapper, record)
                    .onDuplicateKeyUpdate()
                    .set(PROMOCODE_DOMAINS.CLIENT_ID, record.getClientId().asLong())
                    .set(PROMOCODE_DOMAINS.DOMAIN, record.getDomain())
                    .set(PROMOCODE_DOMAINS.IMPORT_TIME, LocalDateTime.now())
                .execute();
    }
}
