package ru.yandex.direct.core.entity.promocodes.service;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.ANTIFRAUD_PROMOCODES;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.CPM;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.TEXT_CAMPAIGN_IN_BALANCE;

/**
 * Код для вычисления возможности предоставления клиенту введения промокода в Балансе
 */
@ParametersAreNonnullByDefault
@Service
public class PromocodeEntryAvailableService {
    private static final Logger LOGGER = LoggerFactory.getLogger(PromocodeEntryAvailableService.class);
    private static final Set<CampaignType> CAMPAIGN_TYPES_TO_CHECK_MONEY_PRESENCE =
            Sets.union(TEXT_CAMPAIGN_IN_BALANCE, CPM);

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final PromocodesAntiFraudService antiFraudService;

    @Autowired
    public PromocodeEntryAvailableService(ShardHelper shardHelper,
                                          CampaignRepository campaignRepository,
                                          PromocodesAntiFraudService antiFraudService) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.antiFraudService = antiFraudService;
    }

    /**
     * Проверить и записать в контейнер, может ли клиент {@param clientId} вводить промокод:
     * - На логине не было кампаний с деньгами
     * - На логине нет РМП кампаний
     * - На логине есть ровно один домен второго уровня (различие в www. - не учитывается)
     * - Домен, а так же его зеркала отсутствуют в статистике показов за год.
     * <p>
     * Аналог перлового Promocodes::is_promocode_entry_available
     * @return контейнер с подходящим доменом, если таковой есть,
     * а также c признаком того, что клиента можно считать новым.
     */

    public PromocodeValidationContainer getPromocodeValidationContainer(ClientId clientId) {

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        boolean isClientConsideredNew = true;

        if (campaignRepository.didClientHaveAnyCampaignWithMoney(shard, clientId,
                CAMPAIGN_TYPES_TO_CHECK_MONEY_PRESENCE)) {
            LOGGER.info("OH MONEY HERE");
            isClientConsideredNew = false;
        }

        Map<CampaignType, Set<String>> campaignTypeToDomains = campaignRepository.getCampaignTypeToDomains(shard,
                clientId, Sets.union(ANTIFRAUD_PROMOCODES, Set.of(CampaignType.MOBILE_CONTENT)));

        if (campaignTypeToDomains.containsKey(CampaignType.MOBILE_CONTENT)) {
            LOGGER.info("MOBILE_CONTENT HERE");
            return new PromocodeValidationContainer(clientId, null, isClientConsideredNew, 0);
        }
        Set<String> domains = EntryStream.of(campaignTypeToDomains)
                .filterKeys(ANTIFRAUD_PROMOCODES::contains)
                .values()
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
        Set<String> strippedDomains = antiFraudService.getStrippedDomains(domains);

        if (strippedDomains.size() == 1
                && strippedDomains.stream().anyMatch(antiFraudService::isRestrictedDomainAcceptable)) {
            var domain = strippedDomains.iterator().next();
            if (antiFraudService.anyDomainHasStat(Sets.union(strippedDomains, domains))) {
                isClientConsideredNew = false;
                LOGGER.info("HAS STAT");
            }
            LOGGER.info("DOMAIN IS {}", domain);
            return new PromocodeValidationContainer(clientId, domain, isClientConsideredNew, strippedDomains.size());
        }
        LOGGER.info("NO DOMAIN");
        return new PromocodeValidationContainer(clientId, null, isClientConsideredNew, strippedDomains.size());
    }

    public boolean isPromocodeEntryAvailable(ClientId clientId) {
        var container = getPromocodeValidationContainer(clientId);
        return container.hasDomain() && container.isClientConsideredNew();
    }
}
