package ru.yandex.direct.core.entity.promocodes.service;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.promocodes.model.PromocodeDomainsCheckResult;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.Defect;

@ParametersAreNonnullByDefault
@Service
public class PromocodeHelper {

    private final PromocodeEntryAvailableService promocodeEntryAvailableService;
    private final PromocodesAntiFraudService promocodesAntiFraudService;

    @Autowired
    public PromocodeHelper(PromocodeEntryAvailableService promocodeEntryAvailableService,
                           PromocodesAntiFraudService promocodesAntiFraudService) {
        this.promocodeEntryAvailableService = promocodeEntryAvailableService;
        this.promocodesAntiFraudService = promocodesAntiFraudService;
    }


    public PromocodeValidationContainer preparePromocodeValidationContainer(ClientId clientId) {
         return promocodeEntryAvailableService.getPromocodeValidationContainer(clientId);
    }

    public boolean doDenyPromocode(PromocodeValidationContainer promocodeValidationContainer) {
        return !(promocodeValidationContainer.isClientConsideredNew() && promocodeValidationContainer.hasDomain());
    }

    public boolean isApplicablePromocode(String promocode, PromocodeValidationContainer promocodeValidationContainer) {
        var checkResult = promocodesAntiFraudService.checkPromocodeDomains(
                List.of(promocode),
                promocodeValidationContainer.getClientId(),
                promocodeValidationContainer.getDomain()
        ).get(promocode);
        return checkResult == null || !checkResult.isMismatch();
    }

    @Nullable
    public Defect<Void> checkPromocode(String promoCode, PromocodeValidationContainer promocodeValidationContainer) {
        if (promoCode == null) {
            return null;
        }
        var checkResult = promocodesAntiFraudService.checkPromocodeDomains(
                List.of(promoCode),
                promocodeValidationContainer.getClientId(),
                promocodeValidationContainer.getDomain()
        ).get(promoCode);
        if (checkResult == PromocodeDomainsCheckResult.DOMAIN_MISMATCH) {
            if (promocodeValidationContainer.getDomainsCount() > 1) {
                return Defects.promocodeClientHasMultipleDomains();
            } else {
                return Defects.promocodeDomainDoesNotMatch();
            }
        }
        if (checkResult == PromocodeDomainsCheckResult.CLIENT_MISMATCH) {
            return Defects.promocodeClientDoesNotMatch();
        }
        return null;
    }
}
