package ru.yandex.direct.core.entity.promocodes.service;

import java.net.IDN;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.balance.client.model.request.TearOffPromocodeRequest;
import ru.yandex.direct.balance.client.model.response.ProcessedInvoiceItem;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeClientDomain;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeDomainsCheckResult;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeInfo;
import ru.yandex.direct.core.entity.promocodes.model.TearOffReason;
import ru.yandex.direct.core.entity.promocodes.repository.PromocodeDomainsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;

import static java.util.Collections.singleton;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.getDomainLevel;

/**
 * Код про противодействие злоупотреблением промокодами.
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-79100">DIRECT-79100: Новая механика для промокодов</a>
 */
@ParametersAreNonnullByDefault
@Service
public class PromocodesAntiFraudService {
    private static final Logger logger = LoggerFactory.getLogger(PromocodesAntiFraudService.class);

    private static final long SYSTEM_OPERATOR_UID = 0L;

    /**
     * Умолчательная дата (активации), с которой учитываем промокоды.
     * По содержимому - дата включения антифрода с отрывом промокодов.
     */
    public static final LocalDate DEFAULT_ANTI_FRAUD_START_DATE = LocalDate.of(2019, 1, 10);
    private static final String SKIP_TEAR_OFF_MESSAGE = "Skip tearing off {} from order {}-{}, reason: {} - disabled " +
            "by property";

    private final BalanceClient balanceClient;
    private final ShardHelper shardHelper;
    private final HostingsHandler hostingsHandler;
    private final CampaignRepository campaignRepository;
    private final BannerDomainRepository bannerDomainRepository;
    private final DomainRepository domainRepository;
    private final PromocodeDomainsRepository promocodeDomainsRepository;
    private final PromocodesTearOffMailSenderService mailSenderService;

    private final PpcProperty<Boolean> tearOffMismatchedPromocodeDomainsCached;
    private final Supplier<LocalDateTime> antiFraudStartDateCached;
    private final Supplier<Boolean> tearOffPromocodeEnabledCached;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public PromocodesAntiFraudService(BalanceClient balanceClient,
                                      CampaignRepository campaignRepository,
                                      BannerDomainRepository bannerDomainRepository,
                                      DomainRepository domainRepository,
                                      ShardHelper shardHelper,
                                      HostingsHandler hostingsHandler,
                                      PpcPropertiesSupport propertiesSupport,
                                      PromocodeDomainsRepository promocodeDomainsRepository,
                                      PromocodesTearOffMailSenderService mailSenderService) {
        this.balanceClient = balanceClient;
        this.shardHelper = shardHelper;
        this.hostingsHandler = hostingsHandler;
        this.campaignRepository = campaignRepository;
        this.bannerDomainRepository = bannerDomainRepository;
        this.domainRepository = domainRepository;
        this.mailSenderService = mailSenderService;
        this.promocodeDomainsRepository = promocodeDomainsRepository;
        tearOffMismatchedPromocodeDomainsCached = propertiesSupport.get(
                PpcPropertyNames.PROMOCODE_DOMAIN_CLIENT_CHECK_IS_ENABLED,
                Duration.ofMinutes(1L)
        );

        LocalDateTime border = LocalDateTime.of(DEFAULT_ANTI_FRAUD_START_DATE, LocalTime.MIDNIGHT);
        this.antiFraudStartDateCached = Suppliers.ofInstance(border);
        this.tearOffPromocodeEnabledCached = Suppliers.ofInstance(true);
    }

    /**
     * Получить граничную дату активации промокода, после которой должны применяться правила антифрода.
     *
     * @return граничная дата (полночь)
     */
    public LocalDateTime getAntiFraudStartDate() {
        return antiFraudStartDateCached.get();
    }

    /**
     * Оторвать промокоды.
     * В зависимости от значения {@link #tearOffPromocodeEnabledCached} - дергает балансовую ручку по отрыву,
     * или только логирует отрыв.
     *
     * @param serviceId  идентификатор сервиса
     * @param campaignId идентификатор кампании
     * @param promocodes список промокодов для отрыва
     * @param reason     причина отрыва для журналирования
     */
    public void tearOffPromocodes(int serviceId,
                                  long campaignId,
                                  List<PromocodeInfo> promocodes,
                                  TearOffReason reason) {
        boolean realTearOff = tearOffPromocodeEnabledCached.get();
        boolean mailAlreadySent = false;

        // среди промокодов могут быть дубли (одинаковые id, разные счета)
        // https://st.yandex-team.ru/BALANCE-28045#1527079276000
        Set<Long> processedPromocodes = new HashSet<>();

        for (PromocodeInfo promocode : promocodes) {
            if (processedPromocodes.contains(promocode.getId())) {
                logger.debug("Skip promocode {} - already processed", promocode);
                continue;
            }

            TearOffPromocodeRequest request = new TearOffPromocodeRequest()
                    .withOperatorUid(SYSTEM_OPERATOR_UID)
                    .withServiceId(serviceId)
                    .withServiceOrderId(campaignId)
                    .withPromocodeId(promocode.getId());

            if (realTearOff) {
                logger.info("Going to tear off {} from order {}-{}, reason: {}",
                        promocode, serviceId, campaignId, reason);
                List<ProcessedInvoiceItem> result = balanceClient.tearOffPromocode(request);
                logger.info("Processed promocode_id: {}", promocode.getId());
                logger.info("Affected invoices: {}", result);

                if (!mailAlreadySent) {
                    mailSenderService.sendPromocodesTearOffMail(campaignId);
                    mailAlreadySent = true;
                }
            } else {
                logger.info(SKIP_TEAR_OFF_MESSAGE, promocode, serviceId, campaignId, reason);
            }

            processedPromocodes.add(promocode.getId());
        }
    }

    /**
     * Можно ли зачислять промокоды с несовпавшим доменом или клиентом или нужно позвать tearOffPromocodes
     * Временный метод, вход как у tearOffPromocodes за вычетом причины отрыва
     *
     * @param serviceId  идентификатор сервиса
     * @param campaignId идентификатор кампании
     * @param promocodes коллекция промокодов для отрыва
     * @return правда, если нужно отрывать. Неправда, если можно зачислять.
     */
    public boolean shouldTearOffMismatched(int serviceId,
                                           long campaignId,
                                           Collection<PromocodeInfo> promocodes) {
        boolean shouldTearOff = tearOffMismatchedPromocodeDomainsCached.getOrDefault(false);
        if (shouldTearOff) {
            return true;
        }
        for (PromocodeInfo promocode : promocodes) {
            logger.info(SKIP_TEAR_OFF_MESSAGE, promocode, serviceId, campaignId, TearOffReason.MISMATCH);
        }
        return false;
    }

    /**
     * Определить уникальный используемый домен.
     * Кампании, в которых ищется домен, определяются с помощью {@link #getAffectedCampaignIds(int, long)}.
     * У доменов отбрасывается незначащий www и они приводятся в юникодную форму.
     * Домен должен быть второго уровня или непосредственно под публичным доменом второго уровня или хостингом
     *
     * @param campaignId id кампании для определения
     * @return уникальный домен или {@code null} в случае, если домен не уникален или не используется ни одного
     */
    @Nullable
    public String determineRestrictedDomain(long campaignId) {
        String domain = getUniqueStrippedDomain(getCandidateDomains(campaignId));
        if (isRestrictedDomainAcceptable(domain)) {
            return domain;
        }
        return null;
    }

    Set<String> getCandidateDomains(long campaignId) {
        /*
         * Допустимое количество кандидатов на уникальный домен.
         * Четыре - так как могут быть варианты с www. и без (и еще пара в punycode),
         * для которых потребуется дополнительная проверка.
         * Если из базы выберется больше - то домены точно не уникальны
         */
        int domainCandidatesLimit = 4;

        int shard = shardHelper.getShardByCampaignId(campaignId);
        List<Long> affectedCampaignIds = getAffectedCampaignIds(shard, campaignId);

        Set<String> domainCandidates = bannerDomainRepository
                .getUniqueBannersDomainsByCampaignIds(shard, affectedCampaignIds, domainCandidatesLimit + 1);
        logger.info("Determining restricted domain for campaign {} (affected campaigns are: {}), candidates: {}",
                campaignId, affectedCampaignIds, domainCandidates);

        if (domainCandidates.size() > domainCandidatesLimit) {
            return Collections.emptySet();
        }
        return domainCandidates;
    }

    @Nullable
    String getUniqueStrippedDomain(Set<String> domainCandidates) {
        if (domainCandidates.isEmpty()) {
            return null;
        }

        Set<String> strippedDomains = getStrippedDomains(domainCandidates);

        if (strippedDomains.size() != 1) {
            return null;
        }

        return Iterables.getOnlyElement(strippedDomains);
    }

    Set<String> getStrippedDomains(Collection<String> domains) {
        return domains.stream()
                .map(IDN::toUnicode)
                .map(hostingsHandler::stripWww)
                .collect(Collectors.toSet());
    }

    /**
     * Проверяем, были ли показы у домена и его главного зеркала (все в обоих кодировках, ASCII и unicode),
     * и пишем предупреждение в лог
     *
     * @param domain домен для проверки
     * @return есть ли показы
     */
    public boolean domainHasStat(String domain) {
        return anyDomainHasStat(singleton(domain));
    }

    boolean anyDomainHasStat(Collection<String> domains) {
        Set<String> toCheck = StreamEx.of(domains)
                .map(String::toLowerCase)
                .flatMap(domain -> StreamEx.of(domain, IDN.toASCII(domain), IDN.toUnicode(domain)))
                .toSet();
        toCheck.addAll(domainRepository.getMainMirrors(toCheck).values());

        logger.debug("Domains to check: {})", toCheck);
        return !domainRepository.getDomainsWithStat(toCheck).isEmpty();
    }

    boolean isRestrictedDomainAcceptable(@Nullable String domain) {
        if (domain == null) {
            return false;
        }
        boolean isSecondLevelDomain = getDomainLevel(hostingsHandler.stripWww(domain)) == 2;
        return hostingsHandler.isFirstSubdomainOfPublicDomainOrHosting(domain) || isSecondLevelDomain;
    }

    /**
     * Является ли кампания относительно новым кошельком
     *
     * @param campaign — кампания, на которую хотят записать промокод
     * @return является или не является
     */
    public boolean isAcceptableWallet(Campaign campaign) {
        return campaign.getType().equals(CampaignType.WALLET)
                && campaign.getCreateTime().isAfter(LocalDateTime.now().minusDays(1));
    }

    /**
     * Получить список кампаний, по которым будет вычислен уникальный домен.
     * В зависимости от типа кампании:<ul>
     * <li>подходящего типа, из {@link CampaignTypeKinds#ANTIFRAUD_PROMOCODES} - сама кампания</li>
     * <li>общий счет - кампании подходящего типа, подключенные к этому общему счета</li>
     * <li>пустой список в остальных случаях</li>
     * </ul>
     *
     * @param shard      шард
     * @param campaignId идентификатор кампании
     * @return список идентификаторов кампаний
     */
    List<Long> getAffectedCampaignIds(int shard, long campaignId) {
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(campaignId));
        CampaignType campaignType = campaignRepository
                .getCampaignsTypeMap(shard, clientId, Collections.singletonList(campaignId))
                .get(campaignId);

        if (campaignType == CampaignType.WALLET) {
            List<Long> campaignIdsUnderWallet = campaignRepository.getCampaignIdsUnderWallet(shard, campaignId);
            Map<Long, CampaignType> campaignsTypeMap = campaignRepository.getCampaignsTypeMap(shard, clientId,
                    campaignIdsUnderWallet, CampaignTypeKinds.ANTIFRAUD_PROMOCODES);
            return EntryStream.of(campaignsTypeMap).keys().toList();
        } else if (CampaignTypeKinds.ANTIFRAUD_PROMOCODES.contains(campaignType)) {
            return Collections.singletonList(campaignId);
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * Сравнить данные о промокодах с тем, что мы себе записали по данным Комдепа.
     * Сценарий использования предполагает, что все промокоды относятся к одной кампании.
     *
     * @param promocodes — промокоды, по которым ищем записи, можно как вводит его клиент
     * @param clientId   — ClientId для сравнения
     * @param domain     — домен для сравнения, если null, то при сравнении будет считаться несовпавшим
     * @return словарь из промокодов как на входе в результаты сравнения в формате enum PromocodeDomainsCheckResult
     */
    public Map<String, PromocodeDomainsCheckResult> checkPromocodeDomains(Collection<String> promocodes,
                                                                          ClientId clientId,
                                                                          @Nullable String domain) {
        var promocodeDomains = getNormalizedPromocodeDomains(promocodes);
        return checkPromocodeDomains(promocodes, clientId, domain, promocodeDomains);
    }

    /**
     * Сравнить данные о промокодах с тем, что мы себе записали по данным Комдепа.
     * Сценарий использования предполагает, что все промокоды относятся к одной кампании.
     *
     * @param promocodes       — промокоды, по которым ищем записи, можно как вводит его клиент
     * @param clientId         — ClientId для сравнения
     * @param domain           — домен для сравнения, если null, то при сравнении будет считаться несовпавшим
     * @param promocodeDomains — словарь из нормализованных промокодов в соответствующие PromocodeClientDomain
     * @return словарь из промокодов как на входе в результаты сравнения в формате enum PromocodeDomainsCheckResult
     */
    public Map<String, PromocodeDomainsCheckResult> checkPromocodeDomains(Collection<String> promocodes,
                                                                  ClientId clientId,
                                                                  @Nullable String domain,
                                                                  Map<String, PromocodeClientDomain> promocodeDomains) {
        Map<String, PromocodeDomainsCheckResult> result = new HashMap<>();
        String normalizedDomain = (domain == null) ? "no single domain" : normalizeDomain(domain);
        for (String promocode : promocodes) {
            PromocodeClientDomain promocodeClientDomain = promocodeDomains.get(normalizePromocode(promocode));
            if (promocodeClientDomain == null) {
                logger.info("Domain not found for promocode {}", promocode);
                result.put(promocode, PromocodeDomainsCheckResult.NOT_FOUND);
                continue;
            }

            if (domain == null || !normalizedDomain.equals(promocodeClientDomain.getDomain())) {
                logger.info("Wrong domain for promocode {}: expected {}, got {}",
                        promocode, promocodeClientDomain.getDomain(), normalizedDomain);
                result.put(promocode, PromocodeDomainsCheckResult.DOMAIN_MISMATCH);
            } else if (!clientId.equals(promocodeClientDomain.getClientId())) {
                logger.info("Wrong clientId for promocode {}: expected {}, got {}",
                        promocode, promocodeClientDomain.getClientId(), clientId);
                result.put(promocode, PromocodeDomainsCheckResult.CLIENT_MISMATCH);
            } else {
                logger.info("Domain and client match successfully for promocode {}", promocode);
                result.put(promocode, PromocodeDomainsCheckResult.OK);
            }
        }
        return result;
    }

    /**
     * Получить данные о промокодах из базы по их нормализованному представлению.
     * Результат пригоден для передачи в checkPromocodeDomains
     *
     * @param promocodes — промокоды, по которым ищем записи, можно как вводит его клиент
     * @return словарь из найденных нормализованных промокодов в запись со словарём и клиентом, указанных у промокода
     */
    public Map<String, PromocodeClientDomain> getNormalizedPromocodeDomains(Collection<String> promocodes) {
        return promocodeDomainsRepository.getPromocodeDomains(
                promocodes.stream().map(PromocodesAntiFraudService::normalizePromocode).collect(Collectors.toList())
        );
    }

    /**
     * Сохранит связь промокода с клиентом и доменом в базу
     *
     * @param promocode — текст промокода
     * @param clientId  — клиент, которому выдан промокод
     * @param domain    — домен, на который выдан промокод
     */
    public void addPromocodeDomain(String promocode, ClientId clientId, String domain) {
        String code = normalizePromocode(promocode);
        if (!promocodeDomainsRepository.getPromocodeDomains(List.of(code)).isEmpty()) {
            logger.warn("Promocode data for the code {} will be replaced", code);
        }
        promocodeDomainsRepository.addPromocodeDomain(new PromocodeClientDomain()
                .withPromocode(code)
                .withDomain(normalizeDomain(domain))
                .withClientId(clientId)
        );
    }

    /**
     * Получить нормализованное представление промокода
     *
     * @param promocode — текст промокода
     * @return нормализованный промокод
     */
    public static String normalizePromocode(String promocode) {
        return promocode.strip().replace("-", "").toUpperCase();
    }

    private String normalizeDomain(String domain) {
        return IDN.toUnicode(hostingsHandler.stripWww(domain));
    }
}
