package ru.yandex.direct.core.entity.promocodes.service;

import java.util.EnumMap;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderTemplateParams;

/**
 * Сервис отправки писем об аннулировании промокодов
 *
 * @see <a href="https://st.yandex-team.ru/DIRECT-86336">DIRECT-86336: добавить отправку письма при отрыве промокода</a>
 * Шаблоны в тестинге
 * https://test.sender.yandex-team.ru/direct/campaign/11133/overview
 * https://test.sender.yandex-team.ru/direct/campaign/11134/overview
 * Шаблоны в проде:
 * https://sender.yandex-team.ru/direct/campaign/38446/overview
 * https://sender.yandex-team.ru/direct/campaign/239559/overview
 */
@Service
@ParametersAreNonnullByDefault
public class PromocodesTearOffMailSenderService {
    static final String NAME = "name";
    static final String CLIENT_ID = "ClientID";

    private final RbacService rbacService;
    private final UserService userService;
    private final YandexSenderClient senderClient;
    private final ShardHelper shardHelper;
    private final EnumMap<Language, String> language2slug;

    @Autowired
    public PromocodesTearOffMailSenderService(RbacService rbacService, UserService userService,
                                              DirectConfig config, YandexSenderClient senderClient, ShardHelper shardHelper) {
        this.rbacService = rbacService;
        this.userService = userService;
        this.senderClient = senderClient;
        this.shardHelper = shardHelper;
        this.language2slug = new EnumMap<>(Language.class);

        DirectConfig promocodeTearOffMailSlugs = config.getBranch("promocodes_antifraud");
        for (Language lang : Language.values()) {
            String langString = lang.getLangString();
            String slug = promocodeTearOffMailSlugs.getString(langString);
            this.language2slug.put(lang, slug);
        }
    }

    /**
     * Отослать главному представителю клиента письмо об аннулировании промокода.
     *
     * @param campaignId идентификатор кампании клиента
     */
    void sendPromocodesTearOffMail(long campaignId) {
        User chiefUser = getChiefByCampaignId(campaignId);
        YandexSenderTemplateParams params = prepareParamsForEmail(chiefUser);
        senderClient.sendTemplate(params);
    }

    /**
     * Подготовка параметров для отправки письма об отрыве промокода
     *
     * @param chiefUser главный представитель клиента
     * @return объектр, содержащий необходимые данные для отправки транзакционного письма
     */
    YandexSenderTemplateParams prepareParamsForEmail(User chiefUser) {
        String email = chiefUser.getEmail();
        Language lang = chiefUser.getLang();
        String campaignSlug = language2slug.get(lang);
        String name = chiefUser.getFio();
        String clientId = chiefUser.getClientId().toString();

        return new YandexSenderTemplateParams.Builder()
                .withToEmail(email)
                .withCampaignSlug(campaignSlug)
                .withAsync(Boolean.TRUE)
                .withArgs(ImmutableMap.of(NAME, name, CLIENT_ID, clientId))
                .build();
    }

    /**
     * Получить главного представителя клиента, которому принадлежит кампания с данным id
     *
     * @param campaignId идентификатор кампании клиента
     * @return главный представитель клиента
     */
    User getChiefByCampaignId(long campaignId) {
        ClientId clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(campaignId));
        long uid = rbacService.getChiefByClientId(clientId);
        return userService.getUser(uid);
    }
}
