package ru.yandex.direct.core.entity.promoextension

import com.google.common.base.Preconditions.checkArgument
import org.jooq.Condition
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.promoextension.model.PromoExtensionUnit
import ru.yandex.direct.core.entity.promoextension.model.PromoExtension
import ru.yandex.direct.dbschema.ppc.tables.CampaignPromoactions.CAMPAIGN_PROMOACTIONS
import ru.yandex.direct.dbschema.ppc.tables.Promoactions.PROMOACTIONS
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.kotlinwrite.JooqKotlinWriterBuilder
import ru.yandex.direct.jooqmapper.kotlinwrite.KotlinWriterBuilders
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.UpdateHelper
import ru.yandex.direct.model.KtModelChanges

@Repository
class PromoExtensionRepository(
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
) {
    private val promoExtensionWriter = JooqKotlinWriterBuilder.builder<PromoExtension>()
        .writeField(PROMOACTIONS.ID, KotlinWriterBuilders.fromOneKProperty(PromoExtension::promoExtensionId))
        .writeField(PROMOACTIONS.CLIENT_ID, KotlinWriterBuilders.fromOneKProperty(PromoExtension::clientId) { it.asLong() })
        .writeField(PROMOACTIONS.TYPE, KotlinWriterBuilders.fromOneKProperty(PromoExtension::type))
        .writeField(PROMOACTIONS.AMOUNT, KotlinWriterBuilders.fromOneKProperty(PromoExtension::amount))
        .writeField(PROMOACTIONS.UNIT, KotlinWriterBuilders.fromOneKProperty(PromoExtension::unit) { it?.dbValue })
        .writeField(PROMOACTIONS.PREFIX, KotlinWriterBuilders.fromOneKProperty(PromoExtension::prefix))
        .writeField(PROMOACTIONS.HREF, KotlinWriterBuilders.fromOneKProperty(PromoExtension::href))
        .writeField(PROMOACTIONS.DESCRIPTION, KotlinWriterBuilders.fromOneKProperty(PromoExtension::description))
        .writeField(PROMOACTIONS.START_DATE, KotlinWriterBuilders.fromOneKProperty(PromoExtension::startDate))
        .writeField(PROMOACTIONS.FINISH_DATE, KotlinWriterBuilders.fromOneKProperty(PromoExtension::finishDate))
        .writeField(PROMOACTIONS.STATUS_MODERATE, KotlinWriterBuilders.fromOneKProperty(PromoExtension::statusModerate))
        .build()

    private fun getByCondition(shard: Int, condition: Condition): List<PromoExtension> {
        return dslContextProvider.ppc(shard)
            .select(
                PROMOACTIONS.ID, PROMOACTIONS.CLIENT_ID, PROMOACTIONS.TYPE,
                PROMOACTIONS.AMOUNT, PROMOACTIONS.UNIT, PROMOACTIONS.PREFIX,
                PROMOACTIONS.HREF, PROMOACTIONS.DESCRIPTION,
                PROMOACTIONS.START_DATE, PROMOACTIONS.FINISH_DATE, PROMOACTIONS.STATUS_MODERATE,
            )
            .from(PROMOACTIONS)
            .where(condition)
            .fetch()
            .map {
                PromoExtension(
                    promoExtensionId = it.get(PROMOACTIONS.ID),
                    clientId = ClientId.fromLong(it.get(PROMOACTIONS.CLIENT_ID)),
                    type = it.get(PROMOACTIONS.TYPE),
                    amount = it.get(PROMOACTIONS.AMOUNT),
                    unit = it.get(PROMOACTIONS.UNIT)?.let { dbValue -> PromoExtensionUnit.fromDbValue(dbValue) },
                    prefix = it.get(PROMOACTIONS.PREFIX),
                    href = it.get(PROMOACTIONS.HREF),
                    description = it.get(PROMOACTIONS.DESCRIPTION),
                    startDate = it.get(PROMOACTIONS.START_DATE),
                    finishDate = it.get(PROMOACTIONS.FINISH_DATE),
                    statusModerate = it.get(PROMOACTIONS.STATUS_MODERATE),
                )
            }
    }

    fun getByIds(shard: Int, ids: Collection<Long>): List<PromoExtension> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return getByCondition(shard, PROMOACTIONS.ID.`in`(ids))
    }

    fun getClientPromoExtensionsByIds(shard: Int, clientId: ClientId, ids: Collection<Long>): List<PromoExtension> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return getByCondition(shard, PROMOACTIONS.ID.`in`(ids).and(PROMOACTIONS.CLIENT_ID.eq(clientId.asLong())))
    }

    fun getAllClientPromoExtensions(clientId: ClientId): List<PromoExtension> {
        return getByCondition(shardHelper.getShardByClientId(clientId), PROMOACTIONS.CLIENT_ID.eq(clientId.asLong()))
    }

    fun getAllClientPromoExtensionIds(shard: Int, clientId: ClientId): Set<Long> {
        return dslContextProvider.ppc(shard)
            .select(PROMOACTIONS.ID)
            .from(PROMOACTIONS)
            .where(PROMOACTIONS.CLIENT_ID.`in`(clientId.asLong()))
            .fetchSet(PROMOACTIONS.ID)
    }

    /**
     * @throws IllegalArgumentException если есть промоакция с ненулевым id
     */
    fun add(shard: Int, clientId: ClientId, promoExtensions: List<PromoExtension>) {
        checkArgument(promoExtensions.all { it.promoExtensionId == null }, "have nonnull promoextension id in add")
        shardHelper
            .generatePromoExtensionIds(clientId.asLong(), promoExtensions.size)
            .zip(promoExtensions)
            .forEach { (id, action) -> action.promoExtensionId = id }
        InsertHelper(dslContextProvider.ppc(shard), PROMOACTIONS)
            .addAll(promoExtensionWriter, promoExtensions)
            .execute()
    }

    fun update(shard: Int, promoExtensions: List<KtModelChanges<Long, PromoExtension>>) {
        UpdateHelper(dslContextProvider.ppc(shard), PROMOACTIONS.ID)
            .processUpdateAllKt(promoExtensionWriter, promoExtensions)
            .execute()
    }

    fun delete(shard: Int, ids: Set<Long>) {
        if (ids.isNotEmpty()) {
            dslContextProvider.ppc(shard)
                .delete(PROMOACTIONS)
                .where(PROMOACTIONS.ID.`in`(ids))
                .execute()
        }
    }

    fun getPromoExtensionIdsByCids(shard: Int, cids: Set<Long>): Map<Long, Long> {
        if (cids.isEmpty()) {
            return mapOf()
        }
        return dslContextProvider.ppc(shard)
            .select(
                CAMPAIGN_PROMOACTIONS.CID, CAMPAIGN_PROMOACTIONS.PROMOACTION_ID,
            )
            .from(CAMPAIGN_PROMOACTIONS)
            .where(CAMPAIGN_PROMOACTIONS.CID.`in`(cids))
            .fetchMap(CAMPAIGN_PROMOACTIONS.CID, CAMPAIGN_PROMOACTIONS.PROMOACTION_ID)
    }

    fun getPromoExtensionsByCids(shard: Int, cids: Set<Long>): Map<Long, PromoExtension> {
        if (cids.isEmpty()) {
            return emptyMap()
        }
        val promoIdByCid = getPromoExtensionIdsByCids(shard, cids)
        val promoIdToPromoExtension = getByIds(shard, promoIdByCid.values).associateBy { it.id }
        return promoIdByCid
            .filterValues(promoIdToPromoExtension::containsKey)
            .mapValues { promoIdToPromoExtension[it.value]!! }
    }

    fun getCidsByPromoExtensionIds(shard: Int, promoIds: Collection<Long>): Map<Long, List<Long>> {
        if (promoIds.isEmpty()) {
            return emptyMap()
        }
        return dslContextProvider.ppc(shard)
            .select(
                CAMPAIGN_PROMOACTIONS.PROMOACTION_ID, CAMPAIGN_PROMOACTIONS.CID,
            )
            .from(CAMPAIGN_PROMOACTIONS)
            .where(CAMPAIGN_PROMOACTIONS.PROMOACTION_ID.`in`(promoIds))
            .fetchGroups(CAMPAIGN_PROMOACTIONS.PROMOACTION_ID, CAMPAIGN_PROMOACTIONS.CID)
    }

    fun getCidsByPromoExtensionIds(promoIds: Set<Long>): Map<Long, List<Long>> {
        if (promoIds.isEmpty()) {
            return emptyMap()
        }
        return shardHelper.groupByShard(promoIds, ShardKey.PROMOACTION_ID)
            .stream()
            .mapKeyValue(this::getCidsByPromoExtensionIds)
            .flatMapToEntry { it }
            .toMap()
    }
}
