package ru.yandex.direct.core.entity.promoextension

import com.google.common.base.Preconditions.checkArgument
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.EntityService
import ru.yandex.direct.core.copyentity.RelationshipService
import ru.yandex.direct.core.entity.campaign.model.CampaignLinkedToPromoExtension
import ru.yandex.direct.core.entity.promoextension.PromoExtensionValidationConstants.MAX_PROMO_EXTENSION_COUNT_FOR_CLIENT
import ru.yandex.direct.core.entity.promoextension.model.PromoExtension
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate.Ready
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.ResultConverters
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.result.ValidationResult.ValidationResultTransformer
import ru.yandex.direct.validation.util.validateList
import java.util.stream.Collectors
import kotlin.reflect.KParameter
import kotlin.reflect.full.instanceParameter
import kotlin.reflect.full.memberFunctions
import kotlin.reflect.full.memberProperties

@Service
class PromoExtensionService(
    private val shardHelper: ShardHelper,
    private val promoExtensionRepository: PromoExtensionRepository,
) : EntityService<PromoExtension, Long> {
    private val moderationSensitiveProperies = setOf(
        PromoExtension::description, PromoExtension::amount, PromoExtension::href,
        PromoExtension::prefix, PromoExtension::type, PromoExtension::unit
    )
    private val logger = LoggerFactory.getLogger(PromoExtensionService::class.java)

    fun add(
        clientId: ClientId,
        promoExtensions: List<PromoExtension>,
        applicability: Applicability = Applicability.PARTIAL
    ): MassResult<Long?> {
        val massResultInternal = addReturnResultWithPromos(clientId, promoExtensions, applicability)
        return ResultConverters.massResultValueConverter(
            { obj: PromoExtension? -> obj?.promoExtensionId }
        ).convert(massResultInternal)
    }

    fun addReturnResultWithPromos(
        clientId: ClientId,
        promoExtensions: List<PromoExtension>,
        applicability: Applicability = Applicability.PARTIAL
    ): MassResult<PromoExtension?> {
        val shard = shardHelper.getShardByClientId(clientId)
        val validator = PromoExtensionValidator()
        val currentClientExtensionCount = promoExtensionRepository.getAllClientPromoExtensionIds(shard, clientId).size
        var validationResult = validateList(promoExtensions) {
            check(
                Constraint.fromPredicate(
                    { list -> list.size + currentClientExtensionCount <= MAX_PROMO_EXTENSION_COUNT_FOR_CLIENT },
                    maxExtensionsForClientId(MAX_PROMO_EXTENSION_COUNT_FOR_CLIENT)
                )
            )
        }
        if (validationResult.hasAnyErrors()) {
            logValidationResultErrors(validationResult)
            return MassResult.brokenMassAction(promoExtensions, validationResult)
        }
        validationResult = validateList(validationResult) {
            checkEachBy(validator)
        }
        if (validationResult.hasAnyErrors()) {
            logValidationResultErrors(validationResult)
            if (Applicability.isFull(applicability)) {
                return MassResult.brokenMassAction(promoExtensions, validationResult)
            }
        }
        val validItems = ValidationResult.getValidItemsWithIndex(validationResult)
        if (validItems.isEmpty()) {
            return MassResult.brokenMassAction(promoExtensions, validationResult)
        }
        validItems.values.forEach { it.statusModerate = Ready }
        val newToOldIndex: Map<Int, Int> = validItems.entries.mapIndexed { index, mutableEntry ->
            index to mutableEntry.key
        }.toMap()
        val itemsToSaveInDb = (0..(validItems.size - 1)).map { validItems[newToOldIndex[it]]!! }
        promoExtensionRepository.add(shard, clientId, itemsToSaveInDb)
        return MassResult.successfulMassAction(promoExtensions, validationResult)
    }

    fun update(
        clientId: ClientId,
        promoExtensionChanges: List<KtModelChanges<Long, PromoExtension>>,
        applicability: Applicability = Applicability.PARTIAL
    ): MassResult<Long?> {
        val massResultInternal = updateReturnResultWithPromos(clientId, promoExtensionChanges, applicability)
        return ResultConverters.massResultValueConverter(
            { obj: PromoExtension? -> obj?.promoExtensionId }
        ).convert(massResultInternal)
    }

    fun updateReturnResultWithPromos(
        clientId: ClientId,
        promoExtensionChanges: List<KtModelChanges<Long, PromoExtension>>,
        applicability: Applicability = Applicability.PARTIAL
    ): MassResult<PromoExtension?> {
        checkArgument(
            promoExtensionChanges.none { it.isPresent(PromoExtension::statusModerate) },
            "internal status moderate field provided"
        )
        val shard = shardHelper.getShardByClientId(clientId)
        val existingPromoExtensions = promoExtensionRepository.getClientPromoExtensionsByIds(
            shard, clientId, promoExtensionChanges.map { it.id }
        ).associateBy { it.promoExtensionId }

        val newPromoExtensionValues =
            promoExtensionChanges.map { applyChangeToPromoExtension(it, existingPromoExtensions[it.id]) }
        val validator = PromoExtensionValidator()
        val transformer = ValidationResultTransformerLocal()
        val validationResult = validateList(newPromoExtensionValues) {
            checkEachBy { extension ->
                if (extension == null) {
                    ValidationResult.failed(extension, CommonDefects.objectNotFound())
                } else {
                    validator.apply(extension).transformUnchecked(transformer)
                }
            }
        }
        if (validationResult.hasAnyErrors()) {
            logValidationResultErrors(validationResult)
            if (Applicability.isFull(applicability)) {
                return MassResult.brokenMassAction(newPromoExtensionValues, validationResult)
            }
        }
        val validItems = ValidationResult.getValidItemsWithIndex(validationResult)
        if (validItems.isEmpty()) {
            return MassResult.brokenMassAction(
                promoExtensionChanges.map { existingPromoExtensions[it.id] },
                validationResult
            )
        }
        val newToOldIndex: Map<Int, Int> = validItems.entries.mapIndexed { index, mutableEntry ->
            index to mutableEntry.key
        }.toMap()
        val itemsToSaveInDb = (0..(validItems.size - 1))
            .map {
                finalizeChange(validItems[newToOldIndex[it]!!]!!, promoExtensionChanges[newToOldIndex[it]!!])
                promoExtensionChanges[newToOldIndex[it]!!]
            }
        promoExtensionRepository.update(shard, itemsToSaveInDb)
        return MassResult.successfulMassAction(newPromoExtensionValues, validationResult)
    }

    fun delete(
        clientId: ClientId,
        promoExtensionIds: List<Long>,
        applicability: Applicability = Applicability.PARTIAL
    ): MassResult<Long?> {
        if (promoExtensionIds.isEmpty()) {
            return MassResult.emptyMassAction()
        }
        val shard = shardHelper.getShardByClientId(clientId)
        val validator =
            PromoExtensionDeleteValidator(
                promoExtensionRepository.getCidsByPromoExtensionIds(shard, promoExtensionIds),
                promoExtensionRepository.getClientPromoExtensionsByIds(shard, clientId, promoExtensionIds)
                    .map { promoExtension -> promoExtension.id!! }.toSet()
            )
        val validationResult = validateList(promoExtensionIds) {
            checkEachBy(validator)
        }

        if (validationResult.hasAnyErrors()) {
            logValidationResultErrors(validationResult)
            if (Applicability.isFull(applicability)) {
                return MassResult.brokenMassAction(promoExtensionIds, validationResult)
            }
        }
        val validItems = ValidationResult.getValidItems(validationResult)
        if (validItems.isEmpty()) {
            return MassResult.brokenMassAction(promoExtensionIds, validationResult)
        }
        promoExtensionRepository.delete(shard, validItems.toSet())

        return MassResult.successfulMassAction(promoExtensionIds, validationResult)
    }

    private fun applyChangeToPromoExtension(
        change: KtModelChanges<Long, PromoExtension>,
        promoExtension: PromoExtension?,
    ): PromoExtension? {
        if (promoExtension == null) {
            return null
        }

        val modelChangesAsMap = PromoExtension::class.memberProperties
            .filter { change.isPresent(it) }
            .associate { it.name to change.getVal(it) }
        val copyFunction = PromoExtension::class.memberFunctions.first { it.name == "copy" }
        val parameters = mutableMapOf<KParameter, Any?>()
        parameters.put(copyFunction.instanceParameter!!, promoExtension)
        copyFunction.parameters.forEach {
            if (modelChangesAsMap.containsKey(it.name)) {
                parameters.put(
                    it,
                    modelChangesAsMap.get(it.name)
                )
            }
        }
        val result = copyFunction.callBy(parameters) as PromoExtension

        if (shouldSendToModeration(promoExtension, result)) {
            result.statusModerate = Ready
        }
        return result
    }

    private fun shouldSendToModeration(before: PromoExtension, after: PromoExtension): Boolean {
        return moderationSensitiveProperies.any { it.get(before) != it.get(after) }
    }

    private fun finalizeChange(
        promoExtension: PromoExtension,
        change: KtModelChanges<Long, PromoExtension>,
    ) {
        if (promoExtension.statusModerate == Ready) {
            change.process(PromoExtension::statusModerate, Ready)
        }
    }

    private fun logValidationResultErrors(vr: ValidationResult<*, Defect<*>>?) {
        if (vr != null && vr.hasAnyErrors()) {
            val vrErrors = vr.flattenErrors().stream()
                .map { it.toString() }
                .collect(Collectors.joining("; "))
            logger.error("errors in validation: {}", vrErrors)
        }
    }

    private inner class ValidationResultTransformerLocal : ValidationResultTransformer<Defect<*>>

    override fun get(clientId: ClientId, operatorUid: Long, ids: Collection<Long>): List<PromoExtension> {
        return promoExtensionRepository.getClientPromoExtensionsByIds(
            shardHelper.getShardByClientId(clientId),
            clientId,
            ids
        )
    }

    override fun add(
        clientId: ClientId,
        operatorUid: Long,
        entities: List<PromoExtension>,
        applicability: Applicability
    ): MassResult<Long?> {
        return add(clientId, entities, applicability)
    }

    override fun copy(
        copyContainer: CopyOperationContainer,
        entities: List<PromoExtension>,
        applicability: Applicability
    ): MassResult<Long?> {
        return add(copyContainer.clientIdTo, copyContainer.operatorUid, entities, applicability)
    }
}

@Service
class CampaignLinkedToPromoExtensionService(
    private val shardHelper: ShardHelper,
    private val promoExtensionRepository: PromoExtensionRepository
) : RelationshipService<CampaignLinkedToPromoExtension, Long, Long> {
    override fun getChildEntityIdsByParentIds(
        clientId: ClientId,
        operatorId: Long,
        promoExtensionIds: Set<Long>
    ): Set<Long> {
        return promoExtensionRepository.getCidsByPromoExtensionIds(
            shardHelper.getShardByClientId(clientId),
            promoExtensionIds
        )
            .values.flatten().toSet()
    }
}
