package ru.yandex.direct.core.entity.promoextension

import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator
import java.time.LocalDate
import ru.yandex.direct.core.entity.banner.type.href.BannerHrefValidator
import ru.yandex.direct.core.entity.promoextension.PromoExtensionNumericDefectIds.MAX_COUNT_OF_PROMO_EXTENSIONS_CREATED
import ru.yandex.direct.core.entity.promoextension.PromoExtensionValidationConstants.MAX_COMPOUND_DESCRIPTION_WITHOUT_DATE_LENGTH
import ru.yandex.direct.core.entity.promoextension.PromoExtensionValidationConstants.MIN_DESCRIPTION_LENGTH
import ru.yandex.direct.core.entity.promoextension.PromoExtensionValidationConstants.TYPES_WITH_AMOUNT_PREFIX_UNIT
import ru.yandex.direct.core.entity.promoextension.model.PromoExtension
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.cashback
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.discount
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType.profit
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.isNull
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.constraint.DateConstraints.endDateIsNotBeforeThan
import ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan
import ru.yandex.direct.validation.constraint.NumberConstraints.inRange
import ru.yandex.direct.validation.constraint.StringConstraints.minStringLength
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.params.NumberDefectParams
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.DefectId
import ru.yandex.direct.validation.result.Path
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

internal class PromoExtensionValidator(
    private val now: LocalDate = LocalDate.now(),
) : Validator<PromoExtension, Defect<*>> {
    private val compoundDescriptionValidator = PromoExtensionCompoundDescriptionValidator()

    override fun apply(promoExtension: PromoExtension): ValidationResult<PromoExtension, Defect<*>> {
        return validateObject(promoExtension) {
            property(PromoExtension::description) {
                check(minStringLength(MIN_DESCRIPTION_LENGTH))
                checkBy {
                    val vr = compoundDescriptionValidator.apply(promoExtension.compoundDescriptionWithoutDate)
                    vr.transformUnchecked(CompoundDescriptionValidationResultTransformer(it))
                }
            }
            property(PromoExtension::amount) {
                check(isNull(), When.isFalse(promoExtension.type in TYPES_WITH_AMOUNT_PREFIX_UNIT))
                if (promoExtension.unit != null) {//делаем через if, чтобы не словить NPE при передаче границ в inRange
                    check(inRange(promoExtension.unit!!.minAmountValue, promoExtension.unit!!.maxAmountValue))
                }
            }
            property(PromoExtension::prefix) {
                check(isNull(), When.isFalse(promoExtension.type in TYPES_WITH_AMOUNT_PREFIX_UNIT))
            }
            property(PromoExtension::unit) {
                check(isNull(), When.isFalse(promoExtension.type in TYPES_WITH_AMOUNT_PREFIX_UNIT))
                check(isNull(), When.isTrue(promoExtension.amount == null))
            }
            property(PromoExtension::href) {
                checkBy(BannerHrefValidator(), When.notNull())
            }
            property(PromoExtension::finishDate) {
                check(
                    endDateIsNotBeforeThan(promoExtension.startDate),
                    When.notNullAnd(When.isTrue(promoExtension.startDate != null))
                )
                check(isNotBeforeThan(now), When.notNull())
            }
        }
    }
}

internal class PromoExtensionDeleteValidator(
    private val promoExtensionIdToCidMap: Map<Long, List<Long>>,
    private val clientPromoExtensionIds: Set<Long>
) : Validator<Long, Defect<*>> {
    // Long? - потому что
    // иначе передается примитив long в валидатор, а в валидаторе создается новое значение.
    // И при мерже возникает ошибка
    // здесь: ru.yandex.direct.validation.builder.ListValidationBuilder:internalCheckEachBy:416
    override fun apply(promoExtensionId: Long?): ValidationResult<Long, Defect<*>> {
        return validateObject(promoExtensionId!!) {
            check(validId())
            check(
                Constraint.fromPredicate(
                    { id -> clientPromoExtensionIds.contains(id) },
                    CommonDefects.objectNotFound()
                ), When.isValid()
            )
            check(
                Constraint.fromPredicate(
                    { id -> !promoExtensionIdToCidMap.containsKey(id) },
                    CommonDefects.unableToDelete()
                ), When.isValid()
            )
        }
    }
}

internal class PromoExtensionCompoundDescriptionValidator : Validator<String, Defect<*>> {
    private val textValidator =
        BannerTextValidator.builder(MAX_COMPOUND_DESCRIPTION_WITHOUT_DATE_LENGTH, Int.MAX_VALUE)
            .withMaxNumberOfNarrowCharacters(Int.MAX_VALUE)
            .withTemplateDisallowed()
            .build()

    override fun apply(compoundDescription: String?): ValidationResult<String, Defect<*>> {
        return textValidator.apply(compoundDescription?.replace("\u00a0", ""))
    }
}

internal class CompoundDescriptionValidationResultTransformer(
    private val newValue: String
) : ValidationResult.ValidationResultTransformer<Defect<*>> {
    override fun <OV : Any?> transformValue(path: Path, oldValue: OV?): Any? =
        if (path.isEmpty) newValue else oldValue
}

internal fun maxExtensionsForClientId(maxObjects: Int): Defect<NumberDefectParams> {
    return Defect(
        MAX_COUNT_OF_PROMO_EXTENSIONS_CREATED,
        NumberDefectParams().withMax(maxObjects)
    )
}

enum class PromoExtensionNumericDefectIds : DefectId<NumberDefectParams> {
    MAX_COUNT_OF_PROMO_EXTENSIONS_CREATED
}

object PromoExtensionValidationConstants {
    internal val TYPES_WITH_AMOUNT_PREFIX_UNIT = setOf(discount, cashback, profit)
    const val MIN_DESCRIPTION_LENGTH = 4
    const val MAX_COMPOUND_DESCRIPTION_WITHOUT_DATE_LENGTH = 45
    const val MAX_PROMO_EXTENSION_COUNT_FOR_CLIENT = 1000
}
