package ru.yandex.direct.core.entity.recommendation.repository;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Operator;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationOnlineInfo;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.recommendation.repository.RecommendationStatusRepository.NUMBER_OF_ROWS;
import static ru.yandex.direct.dbschema.ppc.tables.RecommendationsOnline.RECOMMENDATIONS_ONLINE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Добавление, изменение, удаление, получение статусов рекомендаций (таблица RECOMMENDATIONS_ONLINE)
 */
@Repository
@ParametersAreNonnullByDefault
public class RecommendationOnlineRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<RecommendationOnlineInfo> mapper;

    @Autowired
    public RecommendationOnlineRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    /**
     * Получает рекомендации со статусом
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации в качестве ключей
     * @return {@link Collection} список рекомендаций со статусом
     */
    @Nonnull
    public Collection<RecommendationOnlineInfo> get(int shard, Collection<RecommendationOnlineInfo> recommendations) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(RECOMMENDATIONS_ONLINE)
                .where(createMassCondition(recommendations))
                .fetch()
                .map(mapper::fromDb);
    }

    /**
     * Добавляет рекомендации в таблицу
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации
     * @return {@code int} количество добавленных рекомендаций
     */
    public int add(int shard, Collection<RecommendationOnlineInfo> recommendations) {
        return new InsertHelper<>(dslContextProvider.ppc(shard), RECOMMENDATIONS_ONLINE)
                .addAll(mapper, recommendations)
                .executeIfRecordsAdded();
    }

    /**
     * Обновляет данные рекомендации
     *
     * @param shard          номер шарда
     * @param recommendation рекомендация
     * @return {@code int} количество измененных рекомендаций (0 или 1)
     */
    public int update(int shard, RecommendationOnlineInfo recommendation) {
        return update(dslContextProvider.ppc(shard), recommendation);
    }

    /**
     * Обновляет данные рекомендации
     *
     * @param dslContext     контекст транзакции
     * @param recommendation рекомендация
     * @return {@code int} количество измененных рекомендаций (0 или 1)
     */
    public int update(DSLContext dslContext, RecommendationOnlineInfo recommendation) {
        return dslContext.update(RECOMMENDATIONS_ONLINE)
                .set(RECOMMENDATIONS_ONLINE.DATA, recommendation.getJsonData())
                .where(createMassCondition(singleton(recommendation)))
                .execute();
    }

    /**
     * Удаляет записи
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации в качестве ключей
     * @return {@code int} количество удаленных рекомендаций
     */
    public int delete(int shard, Collection<? extends RecommendationKey> recommendations) {
        if (recommendations.isEmpty()) {
            return 0;
        }

        return dslContextProvider.ppc(shard).delete(RECOMMENDATIONS_ONLINE)
                .where(createMassCondition(recommendations))
                .execute();
    }

    /**
     * Удаляет старые записи пачками по 10к
     * Выполняется через запрос строкой, так как у jooq-a нет возможности лимитировать количество удаляемых строк
     *
     * @param shard     номер шарда
     * @param timestamp время, раньше которого нужно удалять рекоммендации
     * @return {@code int} количество удаленных рекомендаций
     */
    public int deleteOlderThan(int shard, long timestamp) {
        int rows;
        int deletedRows = 0;
        do {
            rows = dslContextProvider.ppc(shard)
                    .execute(RecommendationStatusRepository.SQL_DELETE_QUERY, DSL.field(RECOMMENDATIONS_ONLINE.getName()),
                            DSL.field(RECOMMENDATIONS_ONLINE.TIMESTAMP.getName()),
                            timestamp, NUMBER_OF_ROWS);
            Preconditions.checkState(rows >= 0); // некоторые виды запросов могут вернуть -1, проверяем, что это не так
            deletedRows = deletedRows + rows;
        } while (rows != 0);
        return deletedRows;
    }

    private Condition createMassCondition(Collection<? extends RecommendationKey> recommendations) {
        return DSL.condition(Operator.OR,
                recommendations.stream().map(
                        e -> RECOMMENDATIONS_ONLINE.CLIENT_ID.eq(e.getClientId())
                                .and(RECOMMENDATIONS_ONLINE.TYPE.eq(e.getType()))
                                .and(RECOMMENDATIONS_ONLINE.CID.eq(e.getCampaignId()))
                                .and(RECOMMENDATIONS_ONLINE.PID.eq(e.getAdGroupId()))
                                .and(RECOMMENDATIONS_ONLINE.BID.eq(e.getBannerId()))
                                .and(RECOMMENDATIONS_ONLINE.USER_KEY_1.eq(e.getUserKey1()))
                                .and(RECOMMENDATIONS_ONLINE.USER_KEY_2.eq(e.getUserKey2()))
                                .and(RECOMMENDATIONS_ONLINE.USER_KEY_3.eq(e.getUserKey3()))
                                .and(RECOMMENDATIONS_ONLINE.TIMESTAMP.eq(e.getTimestamp()))
                ).collect(toList()));
    }

    public static JooqMapperWithSupplier<RecommendationOnlineInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(RecommendationOnlineInfo::new)
                .map(property(RecommendationOnlineInfo.CLIENT_ID, RECOMMENDATIONS_ONLINE.CLIENT_ID))
                .map(property(RecommendationOnlineInfo.TYPE, RECOMMENDATIONS_ONLINE.TYPE))
                .map(property(RecommendationOnlineInfo.CAMPAIGN_ID, RECOMMENDATIONS_ONLINE.CID))
                .map(property(RecommendationOnlineInfo.AD_GROUP_ID, RECOMMENDATIONS_ONLINE.PID))
                .map(property(RecommendationOnlineInfo.BANNER_ID, RECOMMENDATIONS_ONLINE.BID))
                .map(property(RecommendationOnlineInfo.USER_KEY1, RECOMMENDATIONS_ONLINE.USER_KEY_1))
                .map(property(RecommendationOnlineInfo.USER_KEY2, RECOMMENDATIONS_ONLINE.USER_KEY_2))
                .map(property(RecommendationOnlineInfo.USER_KEY3, RECOMMENDATIONS_ONLINE.USER_KEY_3))
                .map(property(RecommendationOnlineInfo.TIMESTAMP, RECOMMENDATIONS_ONLINE.TIMESTAMP))
                .map(property(RecommendationOnlineInfo.JSON_DATA, RECOMMENDATIONS_ONLINE.DATA))
                .build();
    }
}
