package ru.yandex.direct.core.entity.recommendation.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationQueueInfo;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.tables.RecommendationsQueue.RECOMMENDATIONS_QUEUE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Добавление, изменение, удаление, получение рекомендаций (таблица RECOMMENDATIONS_QUEUE)
 */
@Repository
@ParametersAreNonnullByDefault
public class RecommendationQueueRepository {
    private static final int RECOMMENDATIONS_FOR_PROCESS_LIMIT = 10;
    private final DslContextProvider dslContextProvider;
    public final JooqMapperWithSupplier<RecommendationQueueInfo> mapper;

    @Autowired
    public RecommendationQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    /**
     * Добавляет рекомендацию
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации
     * @return {@code int} количество добавленных рекомендаций
     */
    public int add(int shard, Collection<RecommendationQueueInfo> recommendations) {
        return new InsertHelper<>(dslContextProvider.ppc(shard), RECOMMENDATIONS_QUEUE)
                .addAll(mapper, recommendations)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    /**
     * Получить порцию рекомендаций для обработки
     * В первую очередь выбираются уже захваченные, но по какой-то причине не обработанные
     *
     * @param shard номер шарда
     * @param parId номер воркера
     * @return {@link Collection} список рекомендаций
     */
    public List<RecommendationQueueInfo> getPortion(int shard, int parId) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(RECOMMENDATIONS_QUEUE)
                .where(RECOMMENDATIONS_QUEUE.PAR_ID.isNull().or(RECOMMENDATIONS_QUEUE.PAR_ID.eq((long) parId)))
                .orderBy(RECOMMENDATIONS_QUEUE.PAR_ID.asc().nullsLast(), RECOMMENDATIONS_QUEUE.SEQ_TIME)
                .limit(RECOMMENDATIONS_FOR_PROCESS_LIMIT)
                .fetch()
                .map(mapper::fromDb);
    }

    /**
     * Получить захваченные рекомендаци для обработки
     *
     * @param shard номер шарда
     * @param parId номер воркера
     * @return {@link Collection} список рекомендаций
     */
    @Nonnull
    public Collection<RecommendationQueueInfo> getLocked(int shard, int parId) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(RECOMMENDATIONS_QUEUE)
                .where(RECOMMENDATIONS_QUEUE.PAR_ID.eq((long) parId))
                .fetch()
                .map(mapper::fromDb);
    }

    /**
     * Получить все рекомендации (для тестирования)
     *
     * @param shard номер шарда
     * @param ids   идентификаторы рекомендаций
     * @return {@link Collection} список рекомендаций
     */
    @Nonnull
    public Collection<RecommendationQueueInfo> get(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(RECOMMENDATIONS_QUEUE)
                .where(RECOMMENDATIONS_QUEUE.ID.in(ids))
                .fetch()
                .map(mapper::fromDb);
    }

    /**
     * Захватывает записи для изменения путем установки поля par_id в номер воркера
     *
     * @param shard номер шарда
     * @param parId номер воркера
     * @param ids   идентификаторы рекомендаций
     * @return {@code int} количество измененных рекомендаций
     */
    public int lock(int shard, int parId, Collection<Long> ids) {
        return dslContextProvider.ppc(shard).update(RECOMMENDATIONS_QUEUE)
                .set(RECOMMENDATIONS_QUEUE.PAR_ID, (long) parId)
                .where(RECOMMENDATIONS_QUEUE.ID.in(ids))
                .and(RECOMMENDATIONS_QUEUE.PAR_ID.isNull())
                .execute();
    }

    /**
     * Удаляет записи
     *
     * @param shard номер шарда
     * @param ids   идентификаторы рекомендаций
     * @return {@code int} количество удаленных рекомендаций
     */
    public int delete(int shard, Collection<Long> ids) {
        return delete(dslContextProvider.ppc(shard), ids);
    }

    /**
     * Удаляет записи
     *
     * @param dslContext контекст транзакции
     * @param ids        идентификаторы рекомендаций
     * @return {@code int} количество удаленных рекомендаций
     */
    public int delete(DSLContext dslContext, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return 0;
        }

        return dslContext.delete(RECOMMENDATIONS_QUEUE)
                .where(RECOMMENDATIONS_QUEUE.ID.in(ids))
                .execute();
    }

    private JooqMapperWithSupplier<RecommendationQueueInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(RecommendationQueueInfo::new)
                .map(property(RecommendationQueueInfo.ID, RECOMMENDATIONS_QUEUE.ID))
                .map(property(RecommendationQueueInfo.CLIENT_ID, RECOMMENDATIONS_QUEUE.CLIENT_ID))
                .map(property(RecommendationQueueInfo.TYPE, RECOMMENDATIONS_QUEUE.TYPE))
                .map(property(RecommendationQueueInfo.CAMPAIGN_ID, RECOMMENDATIONS_QUEUE.CID))
                .map(property(RecommendationQueueInfo.BANNER_ID, RECOMMENDATIONS_QUEUE.BID))
                .map(property(RecommendationQueueInfo.AD_GROUP_ID, RECOMMENDATIONS_QUEUE.PID))
                .map(property(RecommendationQueueInfo.USER_KEY1, RECOMMENDATIONS_QUEUE.USER_KEY_1))
                .map(property(RecommendationQueueInfo.USER_KEY2, RECOMMENDATIONS_QUEUE.USER_KEY_2))
                .map(property(RecommendationQueueInfo.USER_KEY3, RECOMMENDATIONS_QUEUE.USER_KEY_3))
                .map(property(RecommendationQueueInfo.TIMESTAMP, RECOMMENDATIONS_QUEUE.TIMESTAMP))
                .map(property(RecommendationQueueInfo.SEC_TIME, RECOMMENDATIONS_QUEUE.SEQ_TIME))
                .map(property(RecommendationQueueInfo.UID, RECOMMENDATIONS_QUEUE.UID))
                .map(property(RecommendationQueueInfo.QUEUE_TIME, RECOMMENDATIONS_QUEUE.QUEUE_TIME))
                .map(integerProperty(RecommendationQueueInfo.PAR_ID, RECOMMENDATIONS_QUEUE.PAR_ID))
                .map(property(RecommendationQueueInfo.JSON_DATA, RECOMMENDATIONS_QUEUE.DATA))
                .build();
    }
}
