package ru.yandex.direct.core.entity.recommendation.repository;

import java.util.Collection;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Operator;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationStatusInfo;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.recommendation.model.RecommendationStatus.toSource;
import static ru.yandex.direct.dbschema.ppc.tables.RecommendationsStatus.RECOMMENDATIONS_STATUS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Добавление, изменение, удаление, получение статусов рекомендаций (таблица RECOMMENDATIONS_STATUS)
 */

@Repository
@ParametersAreNonnullByDefault
public class RecommendationStatusRepository {
    public static final int ACTUALITY_DAYS_LIMIT = 4;
    public static final int NUMBER_OF_ROWS = 10000;
    public static final String SQL_DELETE_QUERY = "DELETE FROM {0} WHERE {1} <= {2} LIMIT {3}";

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<RecommendationStatusInfo> mapper;

    @Autowired
    public RecommendationStatusRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    /**
     * Получает рекомендации со статусом
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации в качестве ключей
     * @return {@link Set} список рекомендаций со статусом
     */
    @Nonnull
    public Set<RecommendationStatusInfo> get(int shard, Collection<RecommendationStatusInfo> recommendations) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(RECOMMENDATIONS_STATUS)
                .where(createMassCondition(recommendations))
                .fetchSet(mapper::fromDb);
    }

    /**
     * Добавляет статусы в таблицу
     *
     * @param shard    номер шарда
     * @param statuses рекомендации
     * @return {@code int} количество добавленных статусов
     */
    public int add(int shard, Collection<RecommendationStatusInfo> statuses) {
        return new InsertHelper<>(dslContextProvider.ppc(shard), RECOMMENDATIONS_STATUS)
                .addAll(mapper, statuses)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    /**
     * Обновляет статус записи
     *
     * @param shard          номер шарда
     * @param recommendation рекомендация
     * @return {@code int} количество измененных статусов (0 или 1)
     */
    public int update(int shard, RecommendationStatusInfo recommendation) {
        return update(dslContextProvider.ppc(shard), recommendation);
    }

    /**
     * Обновляет статус записи
     *
     * @param dslContext     контекст транзакции
     * @param recommendation рекомендация
     * @return {@code int} количество измененных статусов (0 или 1)
     */
    public int update(DSLContext dslContext, RecommendationStatusInfo recommendation) {
        return dslContext.update(RECOMMENDATIONS_STATUS)
                .set(RECOMMENDATIONS_STATUS.STATUS, toSource(recommendation.getStatus()))
                .where(createMassCondition(singleton(recommendation)))
                .execute();
    }

    /**
     * Удаляет записи
     *
     * @param shard           номер шарда
     * @param recommendations рекомендации в качестве ключей
     * @return {@code int} количество удаленных статусов
     */
    public int delete(int shard, Collection<? extends RecommendationKey> recommendations) {
        if (recommendations.isEmpty()) {
            return 0;
        }

        return dslContextProvider.ppc(shard).delete(RECOMMENDATIONS_STATUS)
                .where(createMassCondition(recommendations))
                .execute();
    }

    /**
     * Удаляет старые записи пачками по 10к
     * Выполняется через запрос строкой, так как у jooq-a нет возможности лимитировать количество удаляемых строк
     *
     * @param shard     номер шарда
     * @param timestamp время, раньше которого нужно удалять рекоммендации
     * @return {@code int} количество удаленных статусов
     */
    public int deleteOlderThan(int shard, long timestamp) {
        int rows;
        int deletedRows = 0;
        do {
            rows = dslContextProvider.ppc(shard)
                    .execute(SQL_DELETE_QUERY, DSL.field(RECOMMENDATIONS_STATUS.getName()),
                            DSL.field(RECOMMENDATIONS_STATUS.TIMESTAMP.getName()),
                            timestamp, NUMBER_OF_ROWS);
            Preconditions.checkState(rows >= 0); // некоторые виды запросов могут вернуть -1, проверяем, что это не так
            deletedRows = deletedRows + rows;
        } while (rows != 0);
        return deletedRows;
    }

    private Condition createMassCondition(Collection<? extends RecommendationKey> recommendations) {
        return DSL.condition(Operator.OR,
                recommendations.stream().map(
                        e -> RECOMMENDATIONS_STATUS.CLIENT_ID.eq(e.getClientId())
                                .and(RECOMMENDATIONS_STATUS.TYPE.eq(e.getType()))
                                .and(RECOMMENDATIONS_STATUS.CID.eq(e.getCampaignId()))
                                .and(RECOMMENDATIONS_STATUS.PID.eq(e.getAdGroupId()))
                                .and(RECOMMENDATIONS_STATUS.BID.eq(e.getBannerId()))
                                .and(RECOMMENDATIONS_STATUS.USER_KEY_1.eq(e.getUserKey1()))
                                .and(RECOMMENDATIONS_STATUS.USER_KEY_2.eq(e.getUserKey2()))
                                .and(RECOMMENDATIONS_STATUS.USER_KEY_3.eq(e.getUserKey3()))
                                .and(RECOMMENDATIONS_STATUS.TIMESTAMP.eq(e.getTimestamp()))
                ).collect(toList()));
    }

    public static JooqMapperWithSupplier<RecommendationStatusInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(RecommendationStatusInfo::new)
                .map(property(RecommendationStatusInfo.CLIENT_ID, RECOMMENDATIONS_STATUS.CLIENT_ID))
                .map(property(RecommendationStatusInfo.TYPE, RECOMMENDATIONS_STATUS.TYPE))
                .map(property(RecommendationStatusInfo.CAMPAIGN_ID, RECOMMENDATIONS_STATUS.CID))
                .map(property(RecommendationStatusInfo.AD_GROUP_ID, RECOMMENDATIONS_STATUS.PID))
                .map(property(RecommendationStatusInfo.BANNER_ID, RECOMMENDATIONS_STATUS.BID))
                .map(property(RecommendationStatusInfo.USER_KEY1, RECOMMENDATIONS_STATUS.USER_KEY_1))
                .map(property(RecommendationStatusInfo.USER_KEY2, RECOMMENDATIONS_STATUS.USER_KEY_2))
                .map(property(RecommendationStatusInfo.USER_KEY3, RECOMMENDATIONS_STATUS.USER_KEY_3))
                .map(property(RecommendationStatusInfo.TIMESTAMP, RECOMMENDATIONS_STATUS.TIMESTAMP))
                .map(convertibleProperty(RecommendationStatusInfo.STATUS, RECOMMENDATIONS_STATUS.STATUS,
                        RecommendationStatus::fromSource, RecommendationStatus::toSource))
                .build();
    }
}
