package ru.yandex.direct.core.entity.recomtracer.repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.recomtracer.container.BannersLoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.CampaignsLoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.LoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.PhrasesLoadedObject;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.when;

@Repository
@ParametersAreNonnullByDefault
public class RecomTracerRepository {

    private final DslContextProvider dslContextProvider;
    private final CampaignRepository campaignRepository;

    // Максимальное кол-во айдишников, отправляемых в 1 запросе к mysql
    // (по идее там конечно довольно большое ограничение в 16Мб на запрос, но лучше не слать огромные пачки айдишников)
    private static final int CHUNK_SIZE = 1000;

    @Autowired
    public RecomTracerRepository(DslContextProvider dslContextProvider, CampaignRepository campaignRepository) {
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
    }

    public Map<Long, LoadedObject> loadBannerEntities(int shard, Set<Long> primaryKeys) {
        return loadEntitiesChunked(shard, primaryKeys, this::loadBannerEntities);
    }

    private Map<Long, LoadedObject> loadBannerEntities(int shard, List<Long> primaryKeys) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(Banners.BANNERS.BID, Phrases.PHRASES.PID, Campaigns.CAMPAIGNS.CID,
                        Campaigns.CAMPAIGNS.CLIENT_ID)
                .from(Banners.BANNERS)
                .join(Phrases.PHRASES)
                .on(Banners.BANNERS.PID.eq(Phrases.PHRASES.PID))
                .join(Campaigns.CAMPAIGNS)
                .on(Phrases.PHRASES.CID.eq(Campaigns.CAMPAIGNS.CID))
                .where(Banners.BANNERS.BID.in(primaryKeys))
                .fetchMap(
                        r -> r.getValue(Banners.BANNERS.BID),
                        r -> new BannersLoadedObject(
                                r.getValue(Banners.BANNERS.BID),
                                r.getValue(Phrases.PHRASES.PID),
                                r.getValue(Campaigns.CAMPAIGNS.CID),
                                r.getValue(Campaigns.CAMPAIGNS.CLIENT_ID)));
    }

    public Map<Long, LoadedObject> loadPhrasesEntities(int shard, Set<Long> primaryKeys) {
        return loadEntitiesChunked(shard, primaryKeys, this::loadPhrasesEntities);
    }

    private Map<Long, LoadedObject> loadPhrasesEntities(int shard, List<Long> primaryKeys) {
        return dslContextProvider.ppc(shard)
                .select(Phrases.PHRASES.PID, Campaigns.CAMPAIGNS.CID, Campaigns.CAMPAIGNS.CLIENT_ID,
                        when(count(Banners.BANNERS.BID).eq(0), true).otherwise(false).as("all_banners_are_archived"))
                .from(Phrases.PHRASES)
                .join(Campaigns.CAMPAIGNS)
                .on(Phrases.PHRASES.CID.eq(Campaigns.CAMPAIGNS.CID))
                .leftJoin(Banners.BANNERS)
                .on(Banners.BANNERS.PID.eq(Phrases.PHRASES.PID))
                .and(Banners.BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .where(Phrases.PHRASES.PID.in(primaryKeys))
                .groupBy(Phrases.PHRASES.PID, Campaigns.CAMPAIGNS.CID, Campaigns.CAMPAIGNS.CLIENT_ID)
                .fetchMap(
                        r -> r.getValue(Phrases.PHRASES.PID),
                        record -> new PhrasesLoadedObject(
                                record.getValue(Phrases.PHRASES.PID),
                                record.getValue(Campaigns.CAMPAIGNS.CID),
                                record.getValue(Campaigns.CAMPAIGNS.CLIENT_ID),
                                record.getValue("all_banners_are_archived", Boolean.class)));
    }

    public Map<Long, LoadedObject> loadCampaignsEntities(int shard, Set<Long> primaryKeys) {
        return loadEntitiesChunked(shard, primaryKeys, this::loadCampaignsEntities);
    }

    private Map<Long, LoadedObject> loadCampaignsEntities(int shard, List<Long> primaryKeys) {
        Map<Long, Long> cidToClientIdMap = campaignRepository.getClientIdsForCids(shard, primaryKeys);
        return cidToClientIdMap.entrySet().stream()
                .collect(toMap(
                        Map.Entry::getKey,
                        cidToClientId -> new CampaignsLoadedObject(cidToClientId.getKey(), cidToClientId.getValue()))
                );
    }

    private Map<Long, LoadedObject> loadEntitiesChunked(int shard, Set<Long> primaryKeys, BiFunction<Integer,
            List<Long>,
            Map<Long, LoadedObject>> loadEntitiesFunction) {
        if (primaryKeys.isEmpty()) {
            return emptyMap();
        }
        List<List<Long>> chunks = Lists.partition(new ArrayList<>(primaryKeys), CHUNK_SIZE);
        Map<Long, LoadedObject> result = new HashMap<>();
        for (List<Long> chunk : chunks) {
            result.putAll(loadEntitiesFunction.apply(shard, chunk));
        }
        return result;
    }
}
