package ru.yandex.direct.core.entity.redirectcheckqueue.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.monitor.tables.RedirectCheckDict.REDIRECT_CHECK_DICT;

/**
 * Репозиторий для работы с кешем очереди проверки на редиректы
 */
@Repository
@ParametersAreNonnullByDefault
public class RedirectCheckDictionaryRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public RedirectCheckDictionaryRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void doInTransaction(TransactionalRunnable transactional) {
        dslContextProvider.monitorTransaction(transactional);
    }

    /**
     * Получить из базы идентификаторы записей с данными доменами
     */
    public List<Long> getItemIdsWithHttpLinks(Collection<String> httpLinks) {
        return getItemIdsWithHttpLinks(dslContextProvider.monitor(), httpLinks);
    }

    /**
     * Получить из базы идентификаторы записей с данными доменами
     */
    public List<Long> getItemIdsWithHttpLinks(DSLContext dslContext, Collection<String> httpLinks) {
        if (httpLinks.isEmpty()) {
            return Collections.emptyList();
        }

        return dslContext
                .select(REDIRECT_CHECK_DICT.ID)
                .from(REDIRECT_CHECK_DICT)
                .where(REDIRECT_CHECK_DICT.HREF.in(httpLinks))
                .forUpdate()
                .fetch(REDIRECT_CHECK_DICT.ID);
    }

    /**
     * Получить из базы идентификаторы записей старше заданной даты
     */
    public List<Long> getItemIdsOlderThanDate(LocalDateTime dateTime) {
        return getItemIdsOlderThanDate(dslContextProvider.monitor(), dateTime);
    }

    /**
     * Получить из базы идентификаторы записей старше заданной даты
     */
    public List<Long> getItemIdsOlderThanDate(DSLContext dslContext, LocalDateTime dateTime) {
        return dslContext
                .select(REDIRECT_CHECK_DICT.ID)
                .from(REDIRECT_CHECK_DICT)
                .where(REDIRECT_CHECK_DICT.LOGDATE.lessThan(dateTime))
                .forUpdate()
                .fetch(REDIRECT_CHECK_DICT.ID);
    }

    /**
     * Удалить из кеша проверки ссылок все записи с переданными идентификаторами
     */
    public int removeFromDictionary(Collection<Long> itemIds) {
        return removeFromDictionary(dslContextProvider.monitor(), itemIds);
    }

    public int removeFromDictionary(DSLContext dslContext, Collection<Long> itemIds) {
        if (itemIds.isEmpty()) {
            return 0;
        }

        return dslContext
                .deleteFrom(REDIRECT_CHECK_DICT)
                .where(REDIRECT_CHECK_DICT.ID.in(itemIds))
                .execute();
    }
}
