package ru.yandex.direct.core.entity.redirectcheckqueue.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStep3;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.redirectcheckqueue.model.CheckRedirectTask;
import ru.yandex.direct.core.entity.redirectcheckqueue.model.RedirectCheckQueueDomainStat;
import ru.yandex.direct.dbschema.ppc.enums.RedirectCheckQueueObjectType;
import ru.yandex.direct.dbschema.ppc.tables.records.RedirectCheckQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.RedirectCheckQueue.REDIRECT_CHECK_QUEUE;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;

/**
 * Репозиторий для работы с очередью проверки на редиректы
 */
@Repository
@ParametersAreNonnullByDefault
public class RedirectCheckQueueRepository {
    private static final long DEFAULT_TRIES_NUM = 0L;

    // через сколько секунд простукивать снова после каждой неудачной попытки
    private static final Map<Long, Duration> TRIES_INTERVAL = Map.of(
            1L, Duration.ofHours(2L),
            2L, Duration.ofHours(24L)
    );
    // максимальное число повторных простукиваний
    private static final long MAX_TRIES = TRIES_INTERVAL.size();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public RedirectCheckQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить всю статистику очереди на проверку на редиректы в разрезе по доменам в данном шарде
     */
    public List<RedirectCheckQueueDomainStat> getDomainCheckStat(int shard) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.DOMAIN, DSL.count(), DSL.countDistinct(BANNERS.CID),
                        DSL.min(REDIRECT_CHECK_QUEUE.LOGTIME))
                .from(REDIRECT_CHECK_QUEUE
                        .join(BANNERS).on(
                                REDIRECT_CHECK_QUEUE.OBJECT_ID.eq(BANNERS.BID)
                                        .and(REDIRECT_CHECK_QUEUE.OBJECT_TYPE.eq(RedirectCheckQueueObjectType.banner))))
                .groupBy(BANNERS.DOMAIN).fetch().map(r -> new RedirectCheckQueueDomainStat()
                        .withDomain(r.value1())
                        .withBannersNum(r.value2())
                        .withCampaignsNum(r.value3())
                        .withOldestEntryAge(r.value4())
                );
    }

    /**
     * Добавить в очередь проверки баннеры из переданного списка в данном шарде
     */
    public int pushBannersIntoQueue(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return 0;
        }
        InsertValuesStep3<RedirectCheckQueueRecord, Long, RedirectCheckQueueObjectType, LocalDateTime> insertStep =
                dslContextProvider.ppc(shard)
                        .insertInto(REDIRECT_CHECK_QUEUE)
                        .columns(REDIRECT_CHECK_QUEUE.OBJECT_ID, REDIRECT_CHECK_QUEUE.OBJECT_TYPE,
                                REDIRECT_CHECK_QUEUE.LOGTIME);
        LocalDateTime logTime = LocalDateTime.now().truncatedTo(ChronoUnit.SECONDS);
        for (Long bannerId : bannerIds) {
            insertStep = insertStep.values(bannerId, RedirectCheckQueueObjectType.banner, logTime);
        }
        return insertStep.onDuplicateKeyUpdate()
                .set(REDIRECT_CHECK_QUEUE.LOGTIME, logTime)
                .set(REDIRECT_CHECK_QUEUE.TRIES, DEFAULT_TRIES_NUM)
                .execute();
    }

    /**
     * Удалить баннеры из очереди.
     */
    public int deleteBannersFromQueue(DSLContext context, Collection<Long> bannerIds) {
        return context.deleteFrom(REDIRECT_CHECK_QUEUE)
                .where(REDIRECT_CHECK_QUEUE.OBJECT_ID.in(bannerIds))
                .and(REDIRECT_CHECK_QUEUE.OBJECT_TYPE.eq(RedirectCheckQueueObjectType.banner))
                .execute();
    }

    public List<CheckRedirectTask> getTasksOlderThan(int shard, LocalDateTime borderDateTime, int limit) {
        return dslContextProvider.ppc(shard)
                .select(REDIRECT_CHECK_QUEUE.ID, BANNERS.BID, BANNERS.HREF, CAMPAIGNS.UID)
                .from(REDIRECT_CHECK_QUEUE)
                .leftJoin(BANNERS).on(BANNERS.BID.eq(REDIRECT_CHECK_QUEUE.OBJECT_ID))
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(REDIRECT_CHECK_QUEUE.LOGTIME.lessOrEqual(borderDateTime))
                .and(REDIRECT_CHECK_QUEUE.OBJECT_TYPE.eq(RedirectCheckQueueObjectType.banner))
                .orderBy(REDIRECT_CHECK_QUEUE.LOGTIME, REDIRECT_CHECK_QUEUE.ID)
                .limit(limit)
                .fetch(r -> new CheckRedirectTask()
                        .withTaskId(r.get(REDIRECT_CHECK_QUEUE.ID))
                        .withBannerId(r.get(BANNERS.BID))
                        .withHref(r.get(BANNERS.HREF))
                        .withUserId(r.get(CAMPAIGNS.UID)));
    }

    public void markTasksFailed(int shard, Collection<Long> taskIds) {
        if (taskIds.isEmpty()) {
            return;
        }
        // ставим logtime в будущее, имея в виду "проверять только после указанной даты"
        LocalDateTime now = LocalDateTime.now();
        Map<Long, LocalDateTime> logtimeByTries = EntryStream.of(TRIES_INTERVAL)
                .mapValues(now::plus)
                .toMap();

        dslContextProvider.ppc(shard)
                .update(REDIRECT_CHECK_QUEUE)
                .set(REDIRECT_CHECK_QUEUE.TRIES, REDIRECT_CHECK_QUEUE.TRIES.plus(1L))
                .set(REDIRECT_CHECK_QUEUE.LOGTIME, makeCaseStatement(REDIRECT_CHECK_QUEUE.TRIES,
                        REDIRECT_CHECK_QUEUE.LOGTIME, logtimeByTries))
                .where(REDIRECT_CHECK_QUEUE.ID.in(taskIds))
                .execute();

        dslContextProvider.ppc(shard)
                .deleteFrom(REDIRECT_CHECK_QUEUE)
                .where(REDIRECT_CHECK_QUEUE.ID.in(taskIds))
                .and(REDIRECT_CHECK_QUEUE.TRIES.greaterThan(MAX_TRIES))
                .execute();
    }

    public void deleteTasksByIds(int shard, Collection<Long> taskIds) {
        if (taskIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .deleteFrom(REDIRECT_CHECK_QUEUE)
                .where(REDIRECT_CHECK_QUEUE.ID.in(taskIds))
                .execute();
    }
}
