package ru.yandex.direct.core.entity.redirectcheckqueue.service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.redirectcheckqueue.model.RedirectCheckQueueDomainStat;
import ru.yandex.direct.core.entity.redirectcheckqueue.repository.RedirectCheckDictionaryRepository;
import ru.yandex.direct.core.entity.redirectcheckqueue.repository.RedirectCheckQueueRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

/**
 * Сервис для работы с очередью проверки на редиректы, а так же с ее кешем
 */
@Service
@ParametersAreNonnullByDefault
public class RedirectCheckQueueService {
    private static final int DELETE_CHUNK_SIZE = 1000;
    private static final int INSERT_CHUNK_SIZE = 1000;
    private static final Duration DEFAULT_DICTIONARY_CLEAR_DURATION = Duration.ofDays(2);

    private final RedirectCheckQueueRepository redirectCheckQueueRepository;
    private final RedirectCheckDictionaryRepository redirectCheckDictionaryRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public RedirectCheckQueueService(RedirectCheckQueueRepository redirectCheckQueueRepository,
                                     RedirectCheckDictionaryRepository redirectCheckDictionaryRepository,
                                     ShardHelper shardHelper) {
        this.redirectCheckQueueRepository = redirectCheckQueueRepository;
        this.redirectCheckDictionaryRepository = redirectCheckDictionaryRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Получить всю статистику очереди на проверку на редиректы в разрезе по доменам
     */
    public List<RedirectCheckQueueDomainStat> getDomainCheckStat() {
        Map<String, RedirectCheckQueueDomainStat> collect = shardHelper.dbShards().stream()
                .map(redirectCheckQueueRepository::getDomainCheckStat)
                .flatMap(List::stream)
                .collect(Collectors.groupingBy(
                        RedirectCheckQueueDomainStat::getDomain,
                        Collector.of(
                                RedirectCheckQueueDomainStat::new,
                                this::mergeStat,
                                this::mergeStat
                        )
                ));
        return new ArrayList<>(collect.values());
    }

    private RedirectCheckQueueDomainStat mergeStat(RedirectCheckQueueDomainStat aggregator,
                                                   RedirectCheckQueueDomainStat item) {
        if (aggregator.getDomain() == null) {
            aggregator
                    .withDomain(item.getDomain())
                    .withCampaignsNum(0)
                    .withBannersNum(0)
                    .withOldestEntryAge(LocalDateTime.MAX);
            if (aggregator.getDomain() == null) {
                aggregator.setDomain("");
            }
        }
        return aggregator.withBannersNum(aggregator.getBannersNum() + item.getBannersNum())
                .withCampaignsNum(aggregator.getCampaignsNum() + item.getCampaignsNum())
                .withOldestEntryAge(aggregator.getOldestEntryAge().isBefore(item.getOldestEntryAge()) ? aggregator
                        .getOldestEntryAge() : item.getOldestEntryAge());
    }

    /**
     * Удалить из кеша проверки ссылок все записи, относящиеся к переданным HTTP-ссылкам
     */
    public void removeLinksFromCache(Collection<String> links) {
        redirectCheckDictionaryRepository.doInTransaction(configuration -> {
            DSLContext txContext = DSL.using(configuration);
            List<Long> ids = redirectCheckDictionaryRepository.getItemIdsWithHttpLinks(txContext, links);
            Lists.partition(ids, DELETE_CHUNK_SIZE)
                    .forEach(i -> redirectCheckDictionaryRepository.removeFromDictionary(txContext, i));
        });
    }

    /**
     * Удалить из кеша проверки ссылок все записи, которые мы считаем старыми (сейчас это записи старше двух дней)
     */
    public void removeOldEnriesFromCache() {
        redirectCheckDictionaryRepository.doInTransaction(configuration -> {
            DSLContext txContext = DSL.using(configuration);
            List<Long> ids = redirectCheckDictionaryRepository
                    .getItemIdsOlderThanDate(txContext, LocalDateTime.now().minus(DEFAULT_DICTIONARY_CLEAR_DURATION));
            Lists.partition(ids, DELETE_CHUNK_SIZE)
                    .forEach(i -> redirectCheckDictionaryRepository.removeFromDictionary(txContext, i));
        });
    }

    /**
     * Добавить в очередь проверки баннеры из переданного списка
     *
     * @return количество добавленных объектов
     */
    public int pushBannersIntoQueue(Collection<Long> bannerIds) {
        return shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .chunkedBy(INSERT_CHUNK_SIZE)
                .stream()
                .mapToInt(e -> redirectCheckQueueRepository.pushBannersIntoQueue(e.getKey(), e.getValue()))
                .sum();
    }
}
