package ru.yandex.direct.core.entity.region;

import java.util.Set;
import java.util.function.Function;

import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.types.Identity;

/**
 * Класс для хранения допустимых регионов показа для языка и дополнительной информации.
 */
public class AllowedRegionsForLanguageContainer {

    public static final AllowedRegionsForLanguageContainer EMPTY =
            new AllowedRegionsForLanguageContainer(null, null, bannerId -> null);

    /**
     * Допустимые регионы показа для языка.
     */
    private final Set<Long> allowedRegionIds;

    /**
     * Сообщение об ошибке несоответствия языка регионам на уровне баннеров.
     */
    private final Translatable bannerDefectMessage;

    /**
     * Функция для получения сообщения об ошибке несоответствия языка регионам на уровне группы.
     */
    private final Function<Identity, Translatable> adGroupDefectMessageGetter;

    private AllowedRegionsForLanguageContainer(Set<Long> allowedRegionIds, Translatable bannerDefectMessage,
                                               Function<Identity, Translatable> adGroupDefectMessageGetter) {
        this.allowedRegionIds = allowedRegionIds;
        this.bannerDefectMessage = bannerDefectMessage;
        this.adGroupDefectMessageGetter = adGroupDefectMessageGetter;
    }

    public Set<Long> getAllowedRegionIds() {
        return allowedRegionIds;
    }

    public Translatable getBannerDefectMessage() {
        return bannerDefectMessage;
    }

    public Translatable getAdGroupDefectMessage(Identity bannerId) {
        return adGroupDefectMessageGetter.apply(bannerId);
    }

    public static Builder builder() {
        return new Builder();
    }

    static class Builder {

        private Set<Long> allowedRegionIds;
        private Translatable bannerDefectMessage;
        private Function<Identity, Translatable> adGroupDefectMessageGetter;

        Builder withAllowedRegionIds(Set<Long> allowedRegionIds) {
            this.allowedRegionIds = allowedRegionIds;
            return this;
        }

        Builder withBannerDefectMessage(Translatable bannerDefectMessage) {
            this.bannerDefectMessage = bannerDefectMessage;
            return this;
        }

        Builder withAdGroupDefectMessageGetter(Function<Identity, Translatable> adGroupDefectMessageGetter) {
            this.adGroupDefectMessageGetter = adGroupDefectMessageGetter;
            return this;
        }

        public AllowedRegionsForLanguageContainer build() {
            return new AllowedRegionsForLanguageContainer(allowedRegionIds, bannerDefectMessage,
                    adGroupDefectMessageGetter);
        }
    }
}
