package ru.yandex.direct.core.entity.region.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.lang.Math.abs;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoEmptyRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoIncorrectRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoMinusRegionMatchesPlusRegion;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoMinusRegionsWithoutPlusRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoNoPlusRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoNonUniqueRegions;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.StringUtils.joinLongsToString;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;

/**
 * Проводит основные валидации списка идентификаторов регионов
 */
public class RegionIdsValidator {

    /**
     * Создает экземпляр валидатора списка регионов
     */
    public RegionIdsValidator() {
    }

    /**
     * Применение валидации с учётом переданного геодерева (которое может быть транслокальным).
     */
    public ValidationResult<List<Long>, Defect> apply(List<Long> regionIds, GeoTree geoTree) {
        return ItemValidationBuilder.<List<Long>, Defect>of(regionIds)
                .check(eachNotNull(), geoEmptyRegions())
                .check(minListSize(1), geoEmptyRegions())
                .check(allRegionsExist(geoTree), When.isValid())
                .check(noMinusRegionEqualToPlusRegion(), When.isValid())
                .check(regionsAreUnique(geoTree), When.isValid())
                .check(hasPlusRegions(), When.isValid())
                .check(minusRegionsAreContainedInPlusRegions(geoTree), When.isValid())
                .getResult();
    }

    public static Constraint<List<Long>, Defect> allRegionsExist(GeoTree geoTree) {
        return regionIds -> {
            List<Long> incorrectRegions = filterList(regionIds, rid -> !geoTree.hasRegion(abs(rid)));
            return incorrectRegions.isEmpty() ? null : geoIncorrectRegions(joinLongsToString(incorrectRegions));
        };
    }

    private static Constraint<List<Long>, Defect> noMinusRegionEqualToPlusRegion() {
        return regionIds -> {
            Set<Long> positive = listToSet(positive(regionIds), id -> id);
            Set<Long> negative = listToSet(negative(regionIds), id -> -id);
            negative.retainAll(positive);
            return negative.isEmpty()
                    ? null
                    : geoMinusRegionMatchesPlusRegion(joinLongsToString(mapList(negative, id -> -id)),
                    joinLongsToString(negative));
        };
    }

    private static Constraint<List<Long>, Defect> regionsAreUnique(GeoTree geoTree) {
        return regionIds -> {
            Set<Long> unique = new HashSet<>();
            Set<Long> nonUnique = new HashSet<>();
            for (Long regionId : regionIds) {
                if (!unique.contains(regionId)) {
                    unique.add(regionId);
                } else {
                    nonUnique.add(regionId);
                }
            }

            List<Region> nonUniqueRegions = nonUnique.stream()
                    .map(Math::abs)
                    .map(geoTree::getRegion)
                    .collect(Collectors.toList());

            return nonUnique.isEmpty() ? null : geoNonUniqueRegions(nonUniqueRegions);
        };
    }

    private static Constraint<List<Long>, Defect> hasPlusRegions() {
        return Constraint.fromPredicate(
                regions -> regions.stream().anyMatch(region -> region >= 0),
                geoNoPlusRegions());
    }

    private static Constraint<List<Long>, Defect> minusRegionsAreContainedInPlusRegions(GeoTree geoTree) {
        return regionIds -> {

            // если все предыдущие проверки прошли, и при этом в списке регионов есть родительский регион,
            // то значит он там один и любые минус-регионы точно включены в него и проверку можно пропустить.
            if (regionIds.contains(GLOBAL_REGION_ID)) {
                return null;
            }

            Set<Long> plusRegions = regionIds.stream().filter(id -> id > 0).collect(toSet());
            List<Region> minusRegionsNotInPlusRegions = regionIds.stream()
                    .filter(id -> id < 0)
                    .filter(negativeId -> geoTree.upRegionTo(-negativeId, plusRegions) == GLOBAL_REGION_ID)
                    .map(id -> geoTree.getRegion(-id))
                    .collect(toList());

            return minusRegionsNotInPlusRegions.isEmpty()
                    ? null
                    : geoMinusRegionsWithoutPlusRegions(minusRegionsNotInPlusRegions);
        };
    }

    private static List<Long> positive(List<Long> regionIds) {
        return filterList(regionIds, rid -> rid >= 0);
    }

    private static List<Long> negative(List<Long> regionIds) {
        return filterList(regionIds, rid -> rid < 0);
    }
}
