package ru.yandex.direct.core.entity.relevancematch.container;

import java.util.Map;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;

public class RelevanceMatchUpdateContainer {
    // идентификатор пользователя, являющего инициатором операции
    private Long operatorUid;
    // идентификатор клиента, для которого выполняется операция
    private ClientId clientId;
    // кампании клиента по id
    private Map<Long, Campaign> campaignsByIds;
    // id кампаний по id групп клиента, в которых обновляется бесфразный таргетинг
    private Map<Long, Long> campaignIdsByAdGroupIds;
    // id групп по id бесфразного таргетинга клиента
    private Map<Long, Long> adGroupIdsByRelevanceMatchIds;
    // бесфразный таргетинги клиента по id
    private Map<Long, RelevanceMatch> relevanceMatchesByIds;

    private RelevanceMatchUpdateContainer() {
    }

    /**
     * Так как для операции обновления клиент не выставляет id кампании и id группы, его выставляем мы.
     * Делаем проверку, что данные консистентны и кампания находится.
     */
    public static RelevanceMatchUpdateContainer createRelevanceMatchUpdateOperationContainer(
            Long operatorUid, ClientId clientId, Map<Long, Campaign> campaignsByIds,
            Map<Long, Long> campaignIdsByAdGroupIds,
            Map<Long, Long> adGroupIdsByRelevanceMatchIds,
            Map<Long, RelevanceMatch> relevanceMatchesByIds) {
        boolean allRelevanceMatchesHasAdGroupIds =
                StreamEx.of(relevanceMatchesByIds.keySet())
                        .allMatch(adGroupIdsByRelevanceMatchIds::containsKey);
        checkState(allRelevanceMatchesHasAdGroupIds);

        boolean allRelevanceMatchesHasCampaignIds = StreamEx.of(relevanceMatchesByIds.keySet())
                .map(adGroupIdsByRelevanceMatchIds::get)
                .allMatch(campaignIdsByAdGroupIds::containsKey);
        checkState(allRelevanceMatchesHasCampaignIds);

        boolean allRelevanceMatchesHasCampaign = StreamEx.of(relevanceMatchesByIds.keySet())
                .map(adGroupIdsByRelevanceMatchIds::get)
                .map(campaignIdsByAdGroupIds::get)
                .allMatch(campaignsByIds::containsKey);
        checkState(allRelevanceMatchesHasCampaign);

        return new RelevanceMatchUpdateContainer()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withCampaignsByIds(campaignsByIds)
                .withCampaignIdsByAdGroupIds(campaignIdsByAdGroupIds)
                .withAdGroupIdsByRelevanceMatchIds(adGroupIdsByRelevanceMatchIds)
                .withRelevanceMatchesByIds(relevanceMatchesByIds);
    }

    public Long getOperatorUid() {
        return operatorUid;
    }

    public RelevanceMatchUpdateContainer withOperatorUid(final Long operatorUid) {
        this.operatorUid = operatorUid;
        return this;
    }

    public ClientId getClientId() {
        return clientId;
    }

    public RelevanceMatchUpdateContainer withClientId(final ClientId clientId) {
        this.clientId = clientId;
        return this;
    }

    public Campaign getCampaignByRelevanceMatchId(Long relevanceMatchId) {
        Long adGroupId = adGroupIdsByRelevanceMatchIds.get(relevanceMatchId);
        Long campaignId = campaignIdsByAdGroupIds.get(adGroupId);
        return campaignsByIds.get(campaignId);
    }

    public Campaign getCampaignById(Long campaignId) {
        return campaignsByIds.get(campaignId);
    }

    public Map<Long, Campaign> getCampaignsByIds() {
        return campaignsByIds;
    }

    private RelevanceMatchUpdateContainer withCampaignsByIds(
            Map<Long, Campaign> campaignsByIds) {
        this.campaignsByIds = campaignsByIds;
        return this;
    }

    public Map<Long, Long> getCampaignIdsByAdGroupIds() {
        return campaignIdsByAdGroupIds;
    }

    private RelevanceMatchUpdateContainer withCampaignIdsByAdGroupIds(
            Map<Long, Long> campaignIdsByAdGroupIds) {
        this.campaignIdsByAdGroupIds = campaignIdsByAdGroupIds;
        return this;
    }

    public Map<Long, Long> getAdGroupIdsByRelevanceMatchIds() {
        return adGroupIdsByRelevanceMatchIds;
    }

    private RelevanceMatchUpdateContainer withAdGroupIdsByRelevanceMatchIds(
            Map<Long, Long> adGroupIdsByRelevanceMatchIds) {
        this.adGroupIdsByRelevanceMatchIds = adGroupIdsByRelevanceMatchIds;
        return this;
    }

    public Map<Long, RelevanceMatch> getRelevanceMatchesByIds() {
        return relevanceMatchesByIds;
    }

    private RelevanceMatchUpdateContainer withRelevanceMatchesByIds(
            Map<Long, RelevanceMatch> relevanceMatchesByIds) {
        this.relevanceMatchesByIds = relevanceMatchesByIds;
        return this;
    }
}
