package ru.yandex.direct.core.entity.relevancematch.repository;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.bids.service.BidBaseOpt;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.jooqmapper.write.PropertyValues;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;

import static java.lang.Boolean.TRUE;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RelevanceMatchMapping {
    public static final ImmutableMap<String, ModelProperty<? super RelevanceMatch, ?>> BID_OPTIONS_FLAGS_ENTRIES =
            ImmutableMap.<String, ModelProperty<? super RelevanceMatch, ?>>builder()
                    .put(RelevanceMatch.IS_DELETED.name(), RelevanceMatch.IS_DELETED)
                    .put(RelevanceMatch.IS_SUSPENDED.name(), RelevanceMatch.IS_SUSPENDED)
                    .build();

    public static final ImmutableSet<ModelProperty<? super RelevanceMatch, ?>> BID_OPTION_FLAGS =
            ImmutableSet.copyOf(BID_OPTIONS_FLAGS_ENTRIES.values());

    private RelevanceMatchMapping() {
    }

    static String relevanceMatchPropsToDbOpts(PropertyValues<RelevanceMatch> propertyValues) {
        return StreamEx.of(
                TRUE.equals(propertyValues.get(RelevanceMatch.IS_SUSPENDED))
                        ? BidBaseOpt.SUSPENDED.getTypedValue() : null,
                TRUE.equals(propertyValues.get(RelevanceMatch.IS_DELETED)) ? BidBaseOpt.DELETED.getTypedValue() : null
        )
                .nonNull()
                .joining(",");
    }

    public static BigDecimal priceToDbFormat(BigDecimal price) {
        return nvl(price, BigDecimal.ZERO);
    }

    public static BigDecimal priceFromDbFormat(BigDecimal price) {
        return price == null || price.compareTo(BigDecimal.ZERO) == 0 ? null : price;
    }

    public static String relevanceMatchCategoriesToDbFormat(@Nullable Set<RelevanceMatchCategory> relevanceMatchCategories) {
        return nvl(relevanceMatchCategories, EnumSet.noneOf(RelevanceMatchCategory.class))
                .stream()
                .map(Enum::toString)
                .collect(joining(","));
    }

    public static Set<RelevanceMatchCategory> relevanceMatchCategoriesFromDbFormat(String s) {
        return StringUtils.isBlank(s) ? Collections.emptySet()
                : Arrays.stream(s.split("\\s*,\\s*"))
                .map(RelevanceMatchCategory::valueOf)
                .collect(Collectors.toSet());
    }

    public static List<ModelChanges<RelevanceMatch>> relevanceMatchesToCoreModelChanges(
            List<RelevanceMatch> relevanceMatches) {
        return mapList(relevanceMatches, RelevanceMatchMapping::relevanceMatchToCoreModelChanges);
    }

    public static ModelChanges<RelevanceMatch> relevanceMatchToCoreModelChanges(RelevanceMatch relevanceMatch) {
        ModelChanges<RelevanceMatch> modelChanges = new ModelChanges<>(relevanceMatch.getId(), RelevanceMatch.class);
        modelChanges.process(relevanceMatch.getAutobudgetPriority(), RelevanceMatch.AUTOBUDGET_PRIORITY);
        modelChanges.process(relevanceMatch.getPrice(), RelevanceMatch.PRICE);
        modelChanges.process(relevanceMatch.getPriceContext(), RelevanceMatch.PRICE_CONTEXT);
        modelChanges.process(relevanceMatch.getIsSuspended(), RelevanceMatch.IS_SUSPENDED);
        modelChanges.process(nvl(relevanceMatch.getIsDeleted(), false), RelevanceMatch.IS_DELETED);
        modelChanges.process(relevanceMatch.getHrefParam1(), RelevanceMatch.HREF_PARAM1);
        modelChanges.process(relevanceMatch.getHrefParam2(), RelevanceMatch.HREF_PARAM2);
        modelChanges.process(relevanceMatch.getRelevanceMatchCategories(), RelevanceMatch.RELEVANCE_MATCH_CATEGORIES);
        return modelChanges;
    }
}
