package ru.yandex.direct.core.entity.relevancematch.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectForUpdateStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.repository.BidMappings;
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt;
import ru.yandex.direct.core.entity.keyword.model.ServingStatus;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsBaseRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsHrefParamsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapper.write.PropertyValues;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.AddedModelId;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.bids.container.ShowConditionType.toBidsBaseBidType;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchValidationService.MAX_RELEVANCE_MATCHES_IN_GROUP;
import static ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.tables.BidsHrefParams.BIDS_HREF_PARAMS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperties;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromSupplier;

@Repository
public class RelevanceMatchRepository {
    private static final int UPDATE_CHUNK_SIZE = 500;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<RelevanceMatch> jooqMapper;
    private final JooqWriter<RelevanceMatch> hrefParamsJooqWriter;

    @Autowired
    public RelevanceMatchRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.jooqMapper = buildBidJooqMapper();
        this.hrefParamsJooqWriter = buildHrefParamsJooqWriter();
    }

    @Nonnull
    private JooqMapperWithSupplier<RelevanceMatch> buildBidJooqMapper() {
        return JooqMapperWithSupplierBuilder.builder(RelevanceMatch::new)
                .map(property(RelevanceMatch.ID, BIDS_BASE.BID_ID))
                .map(property(RelevanceMatch.CAMPAIGN_ID, BIDS_BASE.CID))
                .map(property(RelevanceMatch.AD_GROUP_ID, BIDS_BASE.PID))
                .writeField(BIDS_BASE.BID_TYPE, fromSupplier(() -> BidsBaseBidType.relevance_match))
                .map(integerProperty(RelevanceMatch.AUTOBUDGET_PRIORITY, BIDS_BASE.AUTOBUDGET_PRIORITY))
                .map(convertibleProperty(RelevanceMatch.PRICE, BIDS_BASE.PRICE,
                        RelevanceMatchMapping::priceFromDbFormat,
                        RelevanceMatchMapping::priceToDbFormat))
                .map(convertibleProperty(RelevanceMatch.PRICE_CONTEXT, BIDS_BASE.PRICE_CONTEXT,
                        RelevanceMatchMapping::priceFromDbFormat,
                        RelevanceMatchMapping::priceToDbFormat))
                .map(property(RelevanceMatch.LAST_CHANGE_TIME, BIDS_BASE.LAST_CHANGE))
                .map(convertibleProperty(RelevanceMatch.STATUS_BS_SYNCED, BIDS_BASE.STATUS_BS_SYNCED,
                        BidMappings::statusBsSyncedFromDbFormat,
                        BidMappings::statusBsSyncedToDbFormat))
                .readProperty(RelevanceMatch.IS_SUSPENDED,
                        fromField(BIDS_BASE.OPTS).by(BidMappings::isSuspendedFromDbOpts))
                .readProperty(RelevanceMatch.IS_DELETED,
                        fromField(BIDS_BASE.OPTS).by(BidMappings::isDeletedFromDbOpts))
                .writeField(BIDS_BASE.OPTS,
                        fromProperties(RelevanceMatch.IS_SUSPENDED, RelevanceMatch.IS_DELETED)
                                .by(BidMappings::bidPropsToDbOpts))
                .readProperty(RelevanceMatch.HREF_PARAM1, fromField(BIDS_HREF_PARAMS.PARAM1))
                .readProperty(RelevanceMatch.HREF_PARAM2, fromField(BIDS_HREF_PARAMS.PARAM2))
                .map(convertibleProperty(RelevanceMatch.RELEVANCE_MATCH_CATEGORIES,
                        BIDS_BASE.RELEVANCE_MATCH_CATEGORIES,
                        RelevanceMatchMapping::relevanceMatchCategoriesFromDbFormat,
                        RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat))
                .build();
    }

    private JooqWriter<RelevanceMatch> buildHrefParamsJooqWriter() {
        return JooqWriterBuilder.<RelevanceMatch>builder()
                .writeField(BIDS_HREF_PARAMS.ID, fromProperty(RelevanceMatch.ID))
                .writeField(BIDS_HREF_PARAMS.CID, fromProperty(RelevanceMatch.CAMPAIGN_ID))
                .writeField(BIDS_HREF_PARAMS.PARAM1, fromProperty(RelevanceMatch.HREF_PARAM1))
                .writeField(BIDS_HREF_PARAMS.PARAM2, fromProperty(RelevanceMatch.HREF_PARAM2))
                .build();
    }

    /**
     * При добавлении нового бт проверяем:
     * если в группе есть бт и он удален, восстанавливаем его, с новыми
     * настройками и старым id,
     * если в группе есть бт и он не удален, ничего не меняем, прокидываем старый id,
     * иначе добавляем новый бт.
     * <p>
     * Важно, чтобы порядок возвращаемых бт соответствовал порядку запрашиваемых relevanceMatches - DIRECT-122121
     */
    public List<AddedModelId> addRelevanceMatches(Configuration config, ClientId clientId,
                                                  List<RelevanceMatch> relevanceMatches, Set<Long> affectedAdGroupIds) {
        Map<Long, RelevanceMatch> relevanceMatchesWithDeletedByAdGroupIds =
                getRelevanceMatchesWithDeletedByAdGroupIds(config, clientId, affectedAdGroupIds);

        List<RelevanceMatch> toAdd = new ArrayList<>();
        List<RelevanceMatch> toUpdate = new ArrayList<>();
        List<AddedModelId> result = new ArrayList<>();

        for (RelevanceMatch relevanceMatch : relevanceMatches) {
            RelevanceMatch existingRelevanceMatch =
                    relevanceMatchesWithDeletedByAdGroupIds.get(relevanceMatch.getAdGroupId());

            boolean relevanceMatchExists = existingRelevanceMatch != null;
            boolean isDeleted = relevanceMatchExists && existingRelevanceMatch.getIsDeleted();
            boolean isNotDeleted = relevanceMatchExists && !existingRelevanceMatch.getIsDeleted();

            if (isDeleted) {
                relevanceMatch.setId(existingRelevanceMatch.getId());
                toUpdate.add(relevanceMatch);
                result.add(AddedModelId.ofNew(existingRelevanceMatch.getId()));
            } else if (isNotDeleted) {
                relevanceMatch.setId(existingRelevanceMatch.getId());
                result.add(AddedModelId.ofExisting(existingRelevanceMatch.getId()));
            } else {
                var newRelevanceMatchId = shardHelper.generatePhraseIds(1).get(0);
                relevanceMatch.setId(newRelevanceMatchId);
                toAdd.add(relevanceMatch);
                result.add(AddedModelId.ofNew(newRelevanceMatchId));
            }
        }

        addNewRelevanceMatches(config, toAdd);
        updateDeletedRelevanceMatch(config, toUpdate);

        return result;

    }

    /**
     * Добавление бесфразного таргетинга в группу, в которой нет бесфразного таргетинга.
     */
    private void addNewRelevanceMatches(Configuration config, List<RelevanceMatch> relevanceMatches) {
        new InsertHelper<>(DSL.using(config), BIDS_BASE)
                .addAll(jooqMapper, relevanceMatches)
                .executeIfRecordsAdded();

        addOrUpdateHrefParams(config, relevanceMatches);
    }

    /**
     * восстанавливаем удаленный бесфразный таргетинг, с новыми настройками
     */
    private void updateDeletedRelevanceMatch(Configuration config, List<RelevanceMatch> relevanceMatches) {
        List<AppliedChanges<RelevanceMatch>> appliedChanges = StreamEx.of(relevanceMatches)
                .mapToEntry(RelevanceMatchMapping::relevanceMatchToCoreModelChanges)
                .mapKeyValue((rm, changes) -> changes.applyTo(
                        // изменения на таргетинг со старыми идентфикаторами
                        new RelevanceMatch()
                                .withId(rm.getId())
                                .withAdGroupId(rm.getAdGroupId())
                                .withCampaignId(rm.getCampaignId())
                ))
                .toList();
        update(config, appliedChanges);
    }

    /**
     * метод используется при добавлении нового бесфразного таргетинга.
     * на текущий момент, у группы может быть только один бесфразный таргетинг. логики для добавления нескольких бт нет.
     *
     * @return возвращает бесфразный таргетинг клиента, принадлежащие переданным группам,
     * включая бесфразный таргетинг с флагом isDeleted = true
     */
    private Map<Long, RelevanceMatch> getRelevanceMatchesWithDeletedByAdGroupIds(Configuration config,
                                                                                 ClientId clientId,
                                                                                 Collection<Long> adGroupIds) {
        List<RelevanceMatch> relevanceMatches = config.dsl()
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_BASE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
                .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.CID.eq(BIDS_BASE.CID)
                        .and(BIDS_HREF_PARAMS.ID.eq(BIDS_BASE.BID_ID)))
                .where(BIDS_BASE.PID.in(adGroupIds))
                .and(BIDS_BASE.BID_TYPE.in(
                        toBidsBaseBidType(ShowConditionType.RELEVANCE_MATCH),
                        BidsBaseBidType.relevance_match_search)
                )
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(jooqMapper::fromDb);

        return StreamEx.of(relevanceMatches)
                .mapToEntry(RelevanceMatch::getAdGroupId)
                .invert()
                .toMap();
    }

    /**
     * @return возвращает бесфразный таргетинг клиента по передеанным id,
     * игнорируя бесфразный таргетинг с флагом isDeleted = true
     */
    public Map<Long, RelevanceMatch> getRelevanceMatchesByIds(int shard, ClientId clientId,
                                                              Collection<Long> relevanceMatchIds) {
        ShowConditionSelectionCriteria showConditionSelectionCriteria =
                new ShowConditionSelectionCriteria().withShowConditionIds(relevanceMatchIds);
        List<RelevanceMatch> relevanceMatches =
                getRelevanceMatches(shard, clientId, showConditionSelectionCriteria, LimitOffset.maxLimited(), false);
        return StreamEx.of(relevanceMatches)
                .mapToEntry(RelevanceMatch::getId)
                .invert()
                .toMap();
    }

    /**
     * @return возвращает бесфразные таргетинги клиента, принадлежащие переданным группам,
     * игнорируя бесфразные таргетинги с флагом isDeleted = true
     */
    public Map<Long, RelevanceMatch> getRelevanceMatchesByAdGroupIds(int shard, ClientId clientId,
                                                                     Collection<Long> adGroupIds) {
        return getRelevanceMatchesByAdGroupIds(shard, clientId, adGroupIds, false);
    }

    /**
     * @return возвращает бесфразные таргетинги клиента, принадлежащие переданным группам
     */
    public Map<Long, RelevanceMatch> getRelevanceMatchesByAdGroupIds(int shard, ClientId clientId,
                                                                     Collection<Long> adGroupIds, boolean withDeleted) {
        ShowConditionSelectionCriteria showConditionSelectionCriteria =
                new ShowConditionSelectionCriteria().withAdGroupIds(adGroupIds);
        List<RelevanceMatch> relevanceMatches =
                getRelevanceMatches(shard, clientId, showConditionSelectionCriteria, LimitOffset.maxLimited(),
                        withDeleted);
        return StreamEx.of(relevanceMatches)
                .mapToEntry(RelevanceMatch::getAdGroupId)
                .invert()
                .toMap();
    }

    /**
     * @return возвращает ids бесфразных таргетингов клиента, принадлежащие переданным группам,
     * игнорируя бесфразный таргетинг с флагом isDeleted = true
     */
    public Multimap<Long, Long> getRelevanceMatchIdsByAdGroupIds(int shard, ClientId clientId,
                                                                 Collection<Long> adGroupIds) {
        ShowConditionSelectionCriteria showConditionSelectionCriteria =
                new ShowConditionSelectionCriteria().withAdGroupIds(adGroupIds);

        List<RelevanceMatch> relevanceMatches = getRelevanceMatches(shard, clientId, showConditionSelectionCriteria,
                LimitOffset.maxLimited(), asList(BIDS_BASE.BID_ID, BIDS_BASE.PID), false);

        Multimap<Long, Long> relevanceIdsByAdGroupIds =
                MultimapBuilder.hashKeys().hashSetValues(MAX_RELEVANCE_MATCHES_IN_GROUP).build();
        StreamEx.of(relevanceMatches)
                .mapToEntry(RelevanceMatch::getAdGroupId, RelevanceMatch::getId)
                .forKeyValue(relevanceIdsByAdGroupIds::put);
        return relevanceIdsByAdGroupIds;
    }

    public List<RelevanceMatch> getRelevanceMatches(int shard, @Nullable ClientId clientId,
                                                    ShowConditionSelectionCriteria selection,
                                                    LimitOffset limitOffset,
                                                    boolean withDeleted) {
        return getRelevanceMatches(shard, clientId, selection, limitOffset, jooqMapper.getFieldsToRead(), withDeleted);
    }

    /**
     * Возвращает relevance_match по перенному {@code selection}
     */
    private List<RelevanceMatch> getRelevanceMatches(int shard, @Nullable ClientId clientId,
                                                     ShowConditionSelectionCriteria selection,
                                                     LimitOffset limitOffset,
                                                     Collection<Field<?>> fields,
                                                     boolean withDeleted) {
        if (selection.getShowConditionIds().isEmpty() && selection.getAdGroupIds().isEmpty() && selection
                .getCampaignIds().isEmpty()) {
            return Collections.emptyList();
        }

        Set<ServingStatus> servingStatuses = selection.getServingStatuses();

        var condition = (selection.getCampaignIds().isEmpty() ? DSL.trueCondition()
                : BIDS_BASE.CID.in(selection.getCampaignIds()))
                .and(selection.getShowConditionIds().isEmpty() ? DSL.trueCondition()
                        : BIDS_BASE.BID_ID.in(selection.getShowConditionIds()))
                .and(selection.getAdGroupIds().isEmpty() ? DSL.trueCondition()
                        : BIDS_BASE.PID.in(selection.getAdGroupIds()))
                .and(BIDS_BASE.BID_TYPE
                        .in(BidsBaseBidType.relevance_match, BidsBaseBidType.relevance_match_search))
                .and(servingStatuses.isEmpty() ? DSL.trueCondition()
                        : PHRASES.IS_BS_RARELY_LOADED.in(servingStatuses))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(clientId == null ? DSL.trueCondition() : CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
        if (!withDeleted) {
            condition =
                    condition.andNot(setField(BIDS_BASE.OPTS).contains(BidBaseOpt.DELETED.getTypedValue()));
        }

        SelectForUpdateStep<Record> select = dslContextProvider.ppc(shard)
                .select(fields)
                .from(BIDS_BASE)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_BASE.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
                .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.CID.eq(BIDS_BASE.CID)
                        .and(BIDS_HREF_PARAMS.ID.eq(BIDS_BASE.BID_ID)))
                .where(condition)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());

        return select
                .fetch().stream().map(jooqMapper::fromDb).collect(toList());
    }

    /**
     * обновляет бесфразный таргетинг
     */
    public void update(Configuration config, List<AppliedChanges<RelevanceMatch>> appliedChanges) {
        updateBidsBase(config, appliedChanges);
        List<RelevanceMatch> relevanceMatchToAddOrUpdateHrefParams = StreamEx.of(appliedChanges)
                .filter(change -> change.changed(RelevanceMatch.HREF_PARAM1) || change
                        .changed(RelevanceMatch.HREF_PARAM2))
                .map(AppliedChanges::getModel)
                .collect(toList());
        addOrUpdateHrefParams(config, relevanceMatchToAddOrUpdateHrefParams);
    }

    private void updateBidsBase(Configuration config, List<AppliedChanges<RelevanceMatch>> appliedChanges) {
        for (List<AppliedChanges<RelevanceMatch>> chunk : StreamEx.ofSubLists(appliedChanges, UPDATE_CHUNK_SIZE)) {
            JooqUpdateBuilder<BidsBaseRecord, RelevanceMatch> ub =
                    new JooqUpdateBuilder<>(BIDS_BASE.BID_ID, chunk);
            ub.processProperty(RelevanceMatch.PRICE, BIDS_BASE.PRICE, RelevanceMatchMapping::priceToDbFormat);
            ub.processProperty(RelevanceMatch.PRICE_CONTEXT, BIDS_BASE.PRICE_CONTEXT,
                    RelevanceMatchMapping::priceToDbFormat);
            ub.processProperty(RelevanceMatch.AUTOBUDGET_PRIORITY, BIDS_BASE.AUTOBUDGET_PRIORITY,
                    RepositoryUtils::intToLong);
            ub.processProperty(RelevanceMatch.LAST_CHANGE_TIME, BIDS_BASE.LAST_CHANGE);
            ub.processProperty(RelevanceMatch.STATUS_BS_SYNCED, BIDS_BASE.STATUS_BS_SYNCED,
                    status -> BidsBaseStatusbssynced.valueOf(status.toDbFormat()));
            ub.processProperties(RelevanceMatchMapping.BID_OPTION_FLAGS, BIDS_BASE.OPTS,
                    relevanceMatch -> RelevanceMatchMapping.relevanceMatchPropsToDbOpts(
                            new PropertyValues<>(ImmutableSet.of(
                                    RelevanceMatch.IS_SUSPENDED, RelevanceMatch.IS_DELETED),
                                    relevanceMatch)));
            ub.processProperty(RelevanceMatch.RELEVANCE_MATCH_CATEGORIES, BIDS_BASE.RELEVANCE_MATCH_CATEGORIES,
                    RelevanceMatchMapping::relevanceMatchCategoriesToDbFormat);

            DSL.using(config).update(BIDS_BASE)
                    .set(ub.getValues())
                    .where(BIDS_BASE.BID_ID.in(ub.getChangedIds()))
                    .execute();
        }
    }

    private void addOrUpdateHrefParams(Configuration config, Collection<RelevanceMatch> relevanceMatches) {
        InsertHelper<BidsHrefParamsRecord> insertHelper =
                new InsertHelper<>(DSL.using(config), BIDS_HREF_PARAMS)
                        .addAll(hrefParamsJooqWriter, relevanceMatches);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BIDS_HREF_PARAMS.PARAM1, MySQLDSL.values(BIDS_HREF_PARAMS.PARAM1))
                    .set(BIDS_HREF_PARAMS.PARAM2, MySQLDSL.values(BIDS_HREF_PARAMS.PARAM2));
        }
        insertHelper.executeIfRecordsAdded();
    }

    public Set<Long> getRelevanceMatchIds(int shard, ClientId clientId, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_BASE.BID_ID)
                .from(BIDS_BASE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(BIDS_BASE.BID_TYPE
                        .in(BidsBaseBidType.relevance_match, BidsBaseBidType.relevance_match_search))
                .and(BIDS_BASE.BID_ID.in(ids))
                .fetchSet(BIDS_BASE.BID_ID);
    }
}
