package ru.yandex.direct.core.entity.relevancematch.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.keyword.service.KeywordRecentStatisticsProvider;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.relevancematch.container.AdGroupInfoForRelevanceMatchAdd;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddOperationParams;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects;
import ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchValidationService;
import ru.yandex.direct.core.entity.showcondition.Constants;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils.isExtendedRelevanceMatchAllowedForCampaign;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchPredicates.RM_MISSING_AUTOBUDGET_PRIORITY;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchPredicates.RM_MISSING_CONTEXT_PRICE;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchPredicates.RM_MISSING_SEARCH_PRICE;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Операция добавления Автотаргетинга.
 * <p>
 * <p>
 * Параметр {@code autoPrices} включает режим автоматического выставления
 * ставок, если их не было в запросе, но они нужны в текущей стратегии.
 * При этом контейнер {@code autoPriceParams} в конструкторе должен быть не
 * {@code null}.
 * <p>
 * В режиме {@code autoPrices} сначала ищем ставки в контейнере
 * {@code autoPriceParams.getFixedAutoPrices()}, но если там не нашли, то
 * вычисляем ставки хитрым образом на основе ставок ключевых фраз в группе,
 * см. {@link RelevanceMatchAutoPricesCalculator}.
 * Также автотаргетингам выставляется приоритет автобюджета
 * по умолчанию, если его нет, но он нужен в текущей стратегии.
 * <p>
 * Если режим {@code autoPrices} выключен, наличие нужных ставок и приоритетов
 * автобюджета проверяется валидацией.
 */
public class RelevanceMatchAddOperation extends SimpleAbstractAddOperation<RelevanceMatch, AddedModelId> {
    private final LogPriceService logPriceService;
    private final RelevanceMatchValidationService relevanceMatchValidationService;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final AdGroupRepository adGroupRepository;
    private final DslContextProvider dslContextProvider;

    private final RelevanceMatchAddContainer relevanceMatchAddOperationContainer;
    private final RelevanceMatchAddOperationParams operationParams;


    /**
     * Контейнер с фиксированными автоматическими ставками.
     * Должен быть не {@code null}, если включен режим {@code autoPrices}
     */
    @Nullable
    private final ShowConditionFixedAutoPrices fixedAdGroupAutoPrices;
    private final int shard;
    private final ClientId clientId;
    private final Long operatorUid;
    private final Currency clientCurrency;

    /**
     * Калькулятор автоматических недостающих ставок.
     * Должен быть не {@code null}, если включен режим {@code autoPrices}
     */
    @Nullable
    private RelevanceMatchAutoPricesCalculator autoPricesCalculator;

    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;

    private Map<Integer, AdGroupInfoForRelevanceMatchAdd> adGroupsInfo;

    public RelevanceMatchAddOperation(Applicability applicability, RelevanceMatchAddOperationParams operationParams,
                                      List<RelevanceMatch> relevanceMatches,
                                      LogPriceService logPriceService, RelevanceMatchValidationService relevanceMatchValidationService,
                                      RelevanceMatchRepository relevanceMatchRepository,
                                      AdGroupRepository adGroupRepository,
                                      DslContextProvider dslContextProvider,
                                      KeywordService keywordService,
                                      RelevanceMatchAddContainer relevanceMatchAddOperationContainer,
                                      @Nullable ShowConditionAutoPriceParams autoPriceParams,
                                      int shard, ClientId clientId, Currency clientCurrency, Long operatorUid) {
        super(applicability, relevanceMatches);
        this.operationParams = operationParams;
        this.clientCurrency = clientCurrency;
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;

        this.logPriceService = logPriceService;
        this.adGroupRepository = adGroupRepository;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.relevanceMatchValidationService = relevanceMatchValidationService;
        this.dslContextProvider = dslContextProvider;

        this.relevanceMatchAddOperationContainer = relevanceMatchAddOperationContainer;

        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            checkArgument(relevanceMatchAddOperationContainer != null
                            && relevanceMatchAddOperationContainer.getCampaignIdsByAdGroupIds() == null
                            && relevanceMatchAddOperationContainer.getCampaignsByIds() == null,
                    "relevanceMatchAddOperationContainer should be not null but empty when ad groups nonexistent on prepare");
        }

        ShowConditionFixedAutoPrices showConditionFixedAutoPrices = null;
        if (operationParams.isAutoPrices()) {
            checkArgument(autoPriceParams != null,
                    "autoPriceParams must be specified in autoPrices mode");
            KeywordRecentStatisticsProvider keywordRecentStatisticsProvider =
                    autoPriceParams.getKeywordRecentStatisticsProvider();
            showConditionFixedAutoPrices = autoPriceParams.getFixedAutoPrices();

            autoPricesCalculator = new RelevanceMatchAutoPricesCalculator(
                    keywordRecentStatisticsProvider,
                    keywordService,
                    clientCurrency,
                    clientId
            );
        }
        this.fixedAdGroupAutoPrices = showConditionFixedAutoPrices;
    }

    public void setAdGroupInfoForRelevanceMatchAdd(
            Map<Integer, AdGroupInfoForRelevanceMatchAdd> adGroupInfoForRelevanceMatchAdd) {
        checkState(operationParams.isAdGroupsNonexistentOnPrepare(), "ad group info can't be set if groups exist");
        checkState(!isPrepared(), "operation is already prepared");
        this.adGroupsInfo = adGroupInfoForRelevanceMatchAdd;
    }

    public void setAdGroupsIds(Map<Integer, Long> adGroupIds) {
        checkState(operationParams.isAdGroupsNonexistentOnPrepare(),
                "ids can be set only when ad groups non existent on prepare");
        checkState(isPrepared(), "operation is not prepared yet");
        checkState(!isExecuted(), "operation is already executed");
        List<RelevanceMatch> relevanceMatches = getModels();
        for (int i = 0; i < relevanceMatches.size(); ++i) {
            relevanceMatches.get(i).setAdGroupId(adGroupIds.get(i));
        }
    }

    @Override
    protected void validate(ValidationResult<List<RelevanceMatch>, Defect> preValidationResult) {
        boolean mustHavePrices = !operationParams.isAutoPrices();
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            relevanceMatchValidationService.validateAddRelevanceMatchesWithNonexistentAdGroups(preValidationResult,
                    adGroupsInfo, mustHavePrices);
        } else {
            relevanceMatchValidationService.validateAddRelevanceMatches(
                    preValidationResult, relevanceMatchAddOperationContainer, mustHavePrices);
        }
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<RelevanceMatch> modelsValidatedStep) {
        Map<Integer, RelevanceMatch> relevanceMatches = modelsValidatedStep.getValidModelsMap();
        prepareSystemFields(relevanceMatches);
    }

    private void prepareSystemFields(Map<Integer, RelevanceMatch> relevanceMatches) {
        LocalDateTime now = LocalDateTime.now();
        relevanceMatches.forEach((index, relevanceMatch) -> {
            relevanceMatch
                    .withLastChangeTime(now)
                    .withStatusBsSynced(StatusBsSynced.NO);
            if (operationParams.isAdGroupsNonexistentOnPrepare()) {
                relevanceMatch.withCampaignId(adGroupsInfo.get(index).getCampaign().getId());
            } else {
                relevanceMatch.withCampaignId(relevanceMatchAddOperationContainer.getCampaignIdsByAdGroupIds()
                        .get(relevanceMatch.getAdGroupId()));
            }
        });
    }

    @Override
    protected void beforeExecution(Map<Integer, RelevanceMatch> validModelsMapToApply) {
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            checkAllAdGroupIdsAreSet();
        }

        Collection<RelevanceMatch> relevanceMatches = validModelsMapToApply.values();
        for (RelevanceMatch relevanceMatch : relevanceMatches) {
            Campaign campaign = checkNotNull(getCampaignsById().get(relevanceMatch.getCampaignId()));
            if (!isExtendedRelevanceMatchAllowedForCampaign(campaign)) {
                RelevanceMatchUtils.resetExtendedAttributes(relevanceMatch);
            }
        }
        // Автоматические ставки нужно обязательно считать на этапе apply(), т.к. они рассчитываются на основе ставок
        // ключевых фраз из той же группы, которых на этапе prepare() может еще не быть. Например, в комплексной
        // операции создания/обновления группы
        if (operationParams.isAutoPrices()) {
            checkNotNull(autoPricesCalculator);
            Map<Long, Campaign> campaignsById = getCampaignsById();
            setFixedAutoPrices(relevanceMatches, campaignsById);
            autoPricesCalculator.calcAutoPricesInAdd(relevanceMatches, campaignsById);
            setMissingAutobudgetPriorities(relevanceMatches, campaignsById);
        }

        setMissingPrices(relevanceMatches);

        // дополнительные действия нельзя подготавливать до того, как будут
        // посчитаны ставки
        computeAdditionalTask(relevanceMatches);
        computeTransactionalTask(relevanceMatches);
    }

    /**
     * Проверяет, что у всех бесфразных таргетингов были выставлены id групп, если их не было в {@link #prepare()}
     */
    private void checkAllAdGroupIdsAreSet() {
        List<RelevanceMatch> models = getModels();
        Set<Integer> relevanceMatchWithoutAdGroupId = intRange(0, models.size())
                .stream()
                .filter(index -> models.get(index).getAdGroupId() == null)
                .collect(toSet());
        checkState(relevanceMatchWithoutAdGroupId.isEmpty(),
                "no ad group id for relevance matches with indexes: " + relevanceMatchWithoutAdGroupId);
    }

    /**
     * @return мапа с кампаниями, в которые добавляются автотаргетинги
     */
    private Map<Long, Campaign> getCampaignsById() {
        if (operationParams.isAdGroupsNonexistentOnPrepare()) {
            return StreamEx.of(adGroupsInfo.values()).map(AdGroupInfoForRelevanceMatchAdd::getCampaign)
                    .distinct(Campaign::getId)
                    .toMap(Campaign::getId, Function.identity());
        } else {
            return relevanceMatchAddOperationContainer.getCampaignsByIds();
        }
    }

    /**
     * Попытка выставить фиксированные ставки автотаргетингам, если они не
     * указаны явно, но нужны в текущей стратегии
     *
     * @param relevanceMatches автотаргетинги, которым возможно нужно выставить ставку
     * @param campaignsById    кампании, в которые входят автотаргетинги
     */
    private void setFixedAutoPrices(Collection<RelevanceMatch> relevanceMatches, Map<Long, Campaign> campaignsById) {
        checkNotNull(fixedAdGroupAutoPrices);
        for (RelevanceMatch rm : relevanceMatches) {
            BigDecimal fixedPrice = null;
            if (fixedAdGroupAutoPrices.hasGlobalFixedPrice()) {
                fixedPrice = fixedAdGroupAutoPrices.getGlobalFixedPrice();
            } else if (!operationParams.isAdGroupsNonexistentOnPrepare()
                    && fixedAdGroupAutoPrices.hasAdGroupFixedPrice(rm.getAdGroupId())) {
                fixedPrice = fixedAdGroupAutoPrices.getAdGroupFixedPrice(rm.getAdGroupId());
            }
            if (fixedPrice == null) {
                continue;
            }

            Campaign campaign = campaignsById.get(rm.getCampaignId());
            if (RM_MISSING_SEARCH_PRICE.test(rm, campaign.getStrategy())) {
                rm.setPrice(fixedPrice);
            }
            if (isExtendedRelevanceMatchAllowedForCampaign(campaign)
                    && RM_MISSING_CONTEXT_PRICE.test(rm, campaign.getStrategy())) {
                rm.setPriceContext(fixedPrice);
            }
        }
    }

    /**
     * Выставление автотаргетингам приоритета автобюджета по умолчанию, если он
     * явно не указан, но нужен в текущей стратегии
     *
     * @param relevanceMatches автотаргетинги, которым возможно нужно выставить приоритет автобюджета
     * @param campaignsById    кампании, в которые входят автотаргетинги
     */
    private void setMissingAutobudgetPriorities(Collection<RelevanceMatch> relevanceMatches,
                                                Map<Long, Campaign> campaignsById) {
        for (RelevanceMatch rm : relevanceMatches) {
            Campaign campaign = campaignsById.get(rm.getCampaignId());
            if (RM_MISSING_AUTOBUDGET_PRIORITY.test(rm, campaign.getStrategy())) {
                rm.setAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY);
            }
        }
    }

    /**
     * Выставление цен по умолчанию, если они явно не указаны
     *
     * @param relevanceMatches автотаргетинги, которым возможно нужно выставить цены
     */
    private void setMissingPrices(Collection<RelevanceMatch> relevanceMatches) {
        relevanceMatches.forEach(relevanceMatch -> {
            relevanceMatch.setPrice(nvl(relevanceMatch.getPrice(), BigDecimal.ZERO));
            relevanceMatch.setPriceContext(nvl(relevanceMatch.getPriceContext(), BigDecimal.ZERO));
        });
    }

    private void computeAdditionalTask(Collection<RelevanceMatch> relevanceMatches) {
        additionalTask = () -> {
            List<LogPriceData> priceDataList = computeLogPriceDataList(relevanceMatches);
            logPriceService.logPrice(priceDataList, operatorUid);
        };
    }

    private List<LogPriceData> computeLogPriceDataList(Collection<RelevanceMatch> newRelevanceMatch) {
        Function<RelevanceMatch, LogPriceData> relevanceMatchToLogFn = relevanceMatch -> {
            checkState(relevanceMatch.getId() != null,
                    "attempt to log relevanceMatch without id (may be repository does not set id to added relevanceMatch "
                            + "or computing log records is called before relevanceMatch is added.");
            Double priceSearch = relevanceMatch.getPrice() != null ? relevanceMatch.getPrice().doubleValue() : 0;
            Double priceContext =
                    relevanceMatch.getPriceContext() != null ? relevanceMatch.getPriceContext().doubleValue() : 0;
            return new LogPriceData(
                    relevanceMatch.getCampaignId(),
                    relevanceMatch.getAdGroupId(),
                    relevanceMatch.getId(),
                    priceContext,
                    priceSearch,
                    clientCurrency.getCode(),
                    LogPriceData.OperationType.INSERT_1);
        };
        return mapList(newRelevanceMatch, relevanceMatchToLogFn);
    }

    private void computeTransactionalTask(Collection<RelevanceMatch> relevanceMatches) {
        Set<Long> affectedAdGroupIds = listToSet(relevanceMatches, RelevanceMatch::getAdGroupId);
        transactionalAdditionalTask = conf -> {
            if (!affectedAdGroupIds.isEmpty()) {
                adGroupRepository.dropStatusModerateExceptDraftsAndModerated(conf, affectedAdGroupIds);
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, affectedAdGroupIds, StatusBsSynced.NO);
                adGroupRepository.updateLastChange(conf, affectedAdGroupIds);
            }
        };
    }

    @Override
    protected List<AddedModelId> execute(List<RelevanceMatch> validModelsToApply) {
        List<AddedModelId> relevanceMatchIds = new ArrayList<>();
        TransactionalRunnable saveFn = conf -> {
            Set<Long> affectedAdGroupIds = listToSet(validModelsToApply, RelevanceMatch::getAdGroupId);
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

            relevanceMatchIds.addAll(
                    relevanceMatchRepository
                            .addRelevanceMatches(conf, clientId, validModelsToApply, affectedAdGroupIds));
            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current().profile("relevanceMatchAdd:write", "", validModelsToApply.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }

        return relevanceMatchIds;
    }

    @Override
    protected MassResult<AddedModelId> createMassResult(Map<Integer, AddedModelId> resultMap,
                                                        ValidationResult<List<RelevanceMatch>, Defect> validationResult, Set<Integer> canceledElementIndexes) {
        for (int i = 0; i < getModels().size(); i++) {
            AddedModelId modelId = resultMap.get(i);
            if (modelId != null) {
                ValidationResult<RelevanceMatch, Defect> subValidationResult =
                        validationResult.getOrCreateSubValidationResult(index(i), getModels().get(i));
                if (!modelId.isAdded()) {
                    subValidationResult.addWarning(RelevanceMatchDefects.maxRelevanceMatchesInAdGroup());
                }
            }
        }
        return super.createMassResult(resultMap, validationResult, canceledElementIndexes);
    }
}
