package ru.yandex.direct.core.entity.relevancematch.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.KeywordRecentStatistics;
import ru.yandex.direct.core.entity.keyword.service.KeywordRecentStatisticsProvider;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils.calculatePrice;
import static ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils.isExtendedRelevanceMatchAllowedForCampaign;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchPredicates.RM_MISSING_CONTEXT_PRICE;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchPredicates.RM_MISSING_SEARCH_PRICE;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Калькулятор для расчета автоматических ставок для автотаргетингов, если
 * они не указаны явно, но нужны в текущей стратегии.
 * <p>
 * Ставки расчитываются на основе ставок в ключевых фразах, входящих в ту же
 * группу. Для поиска и сети расчеты производятся сильно очень по-разному, см.
 * {@link RelevanceMatchUtils#calculatePrice(List, Currency)} и
 * {@link #calcContextAutoPrices(Collection, Map)}
 */
@ParametersAreNonnullByDefault
class RelevanceMatchAutoPricesCalculator {

    private final KeywordRecentStatisticsProvider keywordRecentStatisticsProvider;
    private final KeywordService keywordService;
    private final Currency clientCurrency;
    private final ClientId clientId;

    IdentityHashMap<RelevanceMatch, BigDecimal> autoSearchPrices;
    IdentityHashMap<RelevanceMatch, BigDecimal> autoContextPrices;

    /**
     * @param keywordRecentStatisticsProvider объект для получения недавней статистики по показам
     *                                        ключевых фраз
     */
    public RelevanceMatchAutoPricesCalculator(
            KeywordRecentStatisticsProvider keywordRecentStatisticsProvider,
            KeywordService keywordService,
            Currency clientCurrency,
            ClientId clientId) {
        this.keywordRecentStatisticsProvider = keywordRecentStatisticsProvider;
        this.keywordService = keywordService;
        this.clientCurrency = clientCurrency;
        this.clientId = clientId;
    }

    /**
     * Выставляет автоматические недостающие ставки в коллекции AppliedChanges.
     *
     * @param appliedChanges коллекция AppliedChanges автотаргетингов, у которых нужно проверить ставки.
     * @param campaignsById  мапа с кампаниями, в которые входят автотаргетинги
     */
    void calcAutoPricesInUpdate(Collection<AppliedChanges<RelevanceMatch>> appliedChanges,
                                Map<Long, Campaign> campaignsById) {
        initializeAutoPricesMaps(mapList(appliedChanges, AppliedChanges::getModel), campaignsById);

        if (notNeedToCalcAutoPrices()) {
            return;
        }

        appliedChanges.forEach(ac -> {
            RelevanceMatch model = ac.getModel();
            if (autoSearchPrices.containsKey(model)) {
                ac.modify(RelevanceMatch.PRICE, autoSearchPrices.get(model));
            }
            if (autoContextPrices.containsKey(model)) {
                ac.modify(RelevanceMatch.PRICE_CONTEXT, autoContextPrices.get(model));
            }
        });
    }

    /**
     * Выставяет автоматические недостающие ставки в коллекции автотаргетингов.
     *
     * @param relevanceMatches автотаргетинги, у которых нужно проверить и выставить ставку
     * @param campaignsById    мапа с кампаниями, в которые входят автотаргетинги
     */
    void calcAutoPricesInAdd(Collection<RelevanceMatch> relevanceMatches, Map<Long, Campaign> campaignsById) {
        initializeAutoPricesMaps(relevanceMatches, campaignsById);

        if (notNeedToCalcAutoPrices()) {
            return;
        }

        relevanceMatches.forEach(rm -> {
            if (autoSearchPrices.containsKey(rm)) {
                rm.setPrice(autoSearchPrices.get(rm));
            }
            if (autoContextPrices.containsKey(rm)) {
                rm.setPriceContext(autoContextPrices.get(rm));
            }
        });
    }

    /**
     * Проверяет, есть ли у автотаргетингов необходимые ставки и если нет,
     * составляет мапы из автотаргетингов, которым нужно выставить автоставку,
     * и их ставок.
     *
     * @param relevanceMatches автотаргетинги, у которых возможно нужно выставить ставки
     * @param campaignsById    кампании, в которые входят автотаргетинги
     */
    private void initializeAutoPricesMaps(Collection<RelevanceMatch> relevanceMatches,
                                          Map<Long, Campaign> campaignsById) {
        Predicate<RelevanceMatch> needsSearchAutoPrice = rm ->
                RM_MISSING_SEARCH_PRICE.test(rm, campaignsById.get(rm.getCampaignId()).getStrategy());
        Predicate<RelevanceMatch> needsContextAutoPrice = rm ->
                isExtendedRelevanceMatchAllowedForCampaign(campaignsById.get(rm.getCampaignId()))
                        && RM_MISSING_CONTEXT_PRICE.test(rm, campaignsById.get(rm.getCampaignId()).getStrategy());

        Collection<RelevanceMatch> rmNeedSearchPrice = filterList(relevanceMatches, needsSearchAutoPrice);
        Collection<RelevanceMatch> rmNeedContextPrice = filterList(relevanceMatches, needsContextAutoPrice);

        if (rmNeedSearchPrice.isEmpty() && rmNeedContextPrice.isEmpty()) {
            autoSearchPrices = new IdentityHashMap<>();
            autoContextPrices = new IdentityHashMap<>();
            return;
        }

        try (TraceProfile profile = Trace.current()
                .profile("relevanceMatchAutoPriceCalculator:initializeAutoPricesMaps")) {
            Set<Long> adGroupIdsForAutoPrices =
                    Stream.concat(rmNeedSearchPrice.stream(), rmNeedContextPrice.stream())
                            .map(RelevanceMatch::getAdGroupId)
                            .collect(toSet());
            Map<Long, List<Keyword>> keywordsByAdGroupIds =
                    keywordService.getKeywordsByAdGroupIds(clientId, adGroupIdsForAutoPrices);
            autoSearchPrices = calcSearchAutoPrices(rmNeedSearchPrice, keywordsByAdGroupIds);
            autoContextPrices = calcContextAutoPrices(rmNeedContextPrice, keywordsByAdGroupIds);
        }
    }

    private boolean notNeedToCalcAutoPrices() {
        return (autoSearchPrices == null || autoSearchPrices.isEmpty()) &&
                (autoContextPrices == null || autoContextPrices.isEmpty());
    }

    private IdentityHashMap<RelevanceMatch, BigDecimal> calcSearchAutoPrices(
            Collection<RelevanceMatch> relevanceMatches,
            Map<Long, List<Keyword>> keywordsByAdGroupIds) {
        return StreamEx.of(relevanceMatches)
                .mapToEntry(rm -> calcSearchAutoPrice(keywordsByAdGroupIds.get(rm.getAdGroupId())))
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Расчитывает поисковую ставку для автотаргетинга на основе поисковых
     * ставок ключевых фраз.
     * Если ключевых фраз нет, возвращается ставка по умолчанию в валюте клиента.
     * Если у всех ключевых фраз нет поисковой ставки, возвращается минимальная.
     *
     * @param adGroupKeywords ключевые фразы, из которых нужно вывести поисковую ставку для автотаргетинга
     */
    private BigDecimal calcSearchAutoPrice(List<Keyword> adGroupKeywords) {
        List<Keyword> keywordsWithSearchPrices = filterList(adGroupKeywords, kw -> kw.getPrice() != null);
        if (!isEmpty(keywordsWithSearchPrices)) {
            List<BigDecimal> kwSearchPrices = mapList(keywordsWithSearchPrices, Keyword::getPrice);
            return calculatePrice(kwSearchPrices, clientCurrency);
        } else {
            if (isEmpty(adGroupKeywords)) {
                return clientCurrency.getDefaultPrice();
            } else {
                return clientCurrency.getMinPrice();
            }
        }
    }

    /**
     * Расчет недостающих ставок в сети для автотаргетингов.
     * <p>
     * Расчет производится на основе ставок (в сети, и если их нет - поисковых)
     * ключевых фраз, входящих в ту же группу, что и автотаргетинг.
     * От ставок ключевых фраз берется средневзвешенное значение, где весами
     * являются количества показов КФ в сети/на поиске за последнее время.
     * <p>
     * Если в группе нет ключевых фраз, выставляется ставка по умолчанию
     * в валюте клиента.
     *
     * @param rmNeedContextPrice   автотаргетинги, которым возможно нужно выставить ставку в сети
     * @param keywordsByAdGroupIds мапа со списками ключевых фраз по Id групп, в которые входят автотаргетинги
     */
    private IdentityHashMap<RelevanceMatch, BigDecimal> calcContextAutoPrices(
            Collection<RelevanceMatch> rmNeedContextPrice,
            Map<Long, List<Keyword>> keywordsByAdGroupIds) {
        IdentityHashMap<RelevanceMatch, BigDecimal> result = new IdentityHashMap<>();
        if (rmNeedContextPrice.isEmpty()) {
            return result;
        }

        List<Keyword> requests = keywordsByAdGroupIds.values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        Map<Long, KeywordRecentStatistics> keywordRecentStats =
                keywordRecentStatisticsProvider.getKeywordRecentStatistics(requests);


        for (RelevanceMatch rm : rmNeedContextPrice) {
            List<Keyword> keywords = keywordsByAdGroupIds.get(rm.getAdGroupId());
            if (isEmpty(keywords)) {
                result.put(rm, clientCurrency.getDefaultPrice());
                continue;
            }

            BigDecimal priceContext = calcKeywordsWeighedAverageContextishPrice(keywords, keywordRecentStats);
            if (priceContext != null) {
                result.put(rm, Money.valueOf(priceContext, clientCurrency.getCode())
                        .adjustToCurrencyRange()
                        // округление вниз до цента для того, чтобы ставка 2.4000000000000004 не округлялась до 2.5
                        .roundToCentDown()
                        .roundToAuctionStepUp()
                        .bigDecimalValue());
            } else {
                result.put(rm, clientCurrency.getMinPrice());
            }
        }
        return result;
    }

    /**
     * Расчет ставки в сети для автотаргетинга в рамках одной группы на основе
     * ставок ключевых фраз.
     * От ставок ключевых фраз берется средневзвешенное значение, где весами
     * являются количества кликов в сети/на поиске за последнее время.
     * <p>
     * Если у фразы есть ставка в сети, учитывается эта ставка, и кол-во кликов
     * в сети.
     * Если ставки в сети нет, учитывается ставка на поиске, и кол-во кликов на
     * поиске.
     * Если ставок вообще нет, то странно, и возвращается {@code null}
     *
     * @param keywords           список ключевых фраз в группе, в которую входит автотаргетинг
     * @param keywordRecentStats мапа с недавней статистикой по показам ключевых фраз. Ключ - директовский Id фразы
     * @return ставка в сети для автотаргетинга, или {@code null}, если список КФ пустой, или у них нет ставок
     */
    @Nullable
    private BigDecimal calcKeywordsWeighedAverageContextishPrice(List<Keyword> keywords,
                                                                 Map<Long, KeywordRecentStatistics> keywordRecentStats) {
        if (isEmpty(keywords)) {
            return null;
        }

        double priceSum = 0;
        double clicksSum = 0;

        for (Keyword kw : keywords) {
            KeywordRecentStatistics stats = keywordRecentStats.getOrDefault(kw.getId(), null);
            double price;
            long clicks;
            if (kw.getPriceContext() != null) {
                price = kw.getPriceContext().doubleValue();
                clicks = (stats != null ? stats.getNetworkClicks() : 0) + 1;
            } else if (kw.getPrice() != null) {
                price = kw.getPrice().doubleValue();
                clicks = (stats != null ? stats.getSearchClicks() : 0) + 1;
            } else {
                continue;
            }

            priceSum += price * clicks;
            clicksSum += clicks;
        }

        if (clicksSum > 0) {
            return BigDecimal.valueOf(priceSum / clicksSum);
        } else {
            return null;
        }
    }
}
