package ru.yandex.direct.core.entity.relevancematch.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RelevanceMatchDeleteOperation extends AbstractOperationWithId {
    private final RelevanceMatchValidationService relevanceMatchValidationService;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final AdGroupRepository adGroupRepository;
    private final LogPriceService logPriceService;
    private final DslContextProvider dslContextProvider;
    //бесфразные таргетинги по id, которые собираемся удалить
    private final Map<Long, RelevanceMatch> relevanceMatchesByIds;

    private final int shard;
    private final Long operatorUid;
    private final ClientId clientId;

    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;

    private List<AppliedChanges<RelevanceMatch>> appliedChanges;

    public RelevanceMatchDeleteOperation(Applicability applicability, List<Long> modelIds,
                                         RelevanceMatchRepository relevanceMatchRepository,
                                         RelevanceMatchValidationService relevanceMatchValidationService, LogPriceService logPriceService,
                                         AdGroupRepository adGroupRepository, DslContextProvider dslContextProvider,
                                         Map<Long, RelevanceMatch> relevanceMatchesByIds,
                                         int shard,
                                         Long operatorUid,
                                         ClientId clientId) {
        super(applicability, modelIds);
        this.shard = shard;
        this.operatorUid = operatorUid;
        this.clientId = clientId;

        this.relevanceMatchRepository = relevanceMatchRepository;
        this.relevanceMatchValidationService = relevanceMatchValidationService;
        this.logPriceService = logPriceService;
        this.adGroupRepository = adGroupRepository;
        this.dslContextProvider = dslContextProvider;

        this.relevanceMatchesByIds = relevanceMatchesByIds;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        return relevanceMatchValidationService.validateDeleteRelevanceMatches(ids, relevanceMatchesByIds,
                operatorUid, clientId);
    }


    @Override
    protected void beforeExecution(List<Long> ids) {
        computeAdditionalTask(ids);
        computeTransactionalTask(ids);
        //удаление бесфрзаного таргетинга заключается в обнулении полей
        appliedChanges = StreamEx.of(ids)
                .map(this::prepareToDelete)
                .map(relevanceMatchModelChanges -> relevanceMatchModelChanges
                        .applyTo(relevanceMatchesByIds.get(relevanceMatchModelChanges.getId())))
                .toList();
    }

    private void computeAdditionalTask(List<Long> ids) {
        List<RelevanceMatch> relevanceMatches = StreamEx.of(ids)
                .map(relevanceMatchesByIds::get)
                .toList();

        List<LogPriceData> priceDataList = computeLogPriceDataList(relevanceMatches);
        additionalTask = () -> logPriceService.logPrice(priceDataList, operatorUid);
    }

    private List<LogPriceData> computeLogPriceDataList(Collection<RelevanceMatch> newRelevanceMatch) {
        return mapList(newRelevanceMatch, this::relevanceMatchToLog);
    }

    private LogPriceData relevanceMatchToLog(RelevanceMatch relevanceMatch) {
        return new LogPriceData(
                relevanceMatch.getCampaignId(),
                relevanceMatch.getAdGroupId(),
                relevanceMatch.getId(),
                null,
                null,
                null,
                LogPriceData.OperationType.DELETE_1);
    }

    private void computeTransactionalTask(List<Long> ids) {
        Set<Long> affectedAdGroupIds = StreamEx.of(ids)
                .map(relevanceMatchesByIds::get)
                .map(RelevanceMatch::getAdGroupId)
                .toSet();

        transactionalAdditionalTask = conf -> {
            if (!affectedAdGroupIds.isEmpty()) {
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, affectedAdGroupIds, StatusBsSynced.NO);
                adGroupRepository.updateLastChange(conf, affectedAdGroupIds);
            }
        };
    }

    private ModelChanges<RelevanceMatch> prepareToDelete(Long id) {
        ModelChanges<RelevanceMatch> changes = new ModelChanges<>(id, RelevanceMatch.class);
        changes.process(null, RelevanceMatch.PRICE);
        changes.process(null, RelevanceMatch.PRICE_CONTEXT);
        changes.process(null, RelevanceMatch.AUTOBUDGET_PRIORITY);
        changes.process(true, RelevanceMatch.IS_DELETED);
        changes.process(null, RelevanceMatch.HREF_PARAM1);
        changes.process(null, RelevanceMatch.HREF_PARAM2);
        return changes;
    }

    @Override
    protected void execute(List<Long> ids) {
        TransactionalRunnable saveFn = conf -> {
            Set<Long> affectedAdGroupIds = StreamEx.of(relevanceMatchesByIds.values())
                    .filter(relevanceMatch -> ids.contains(relevanceMatch.getId()))
                    .map(RelevanceMatch::getAdGroupId)
                    .toSet();

            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);
            relevanceMatchRepository.update(conf, appliedChanges);
            transactionalAdditionalTask.run(conf);
        };

        dslContextProvider.ppcTransaction(shard, saveFn);
    }

    @Override
    protected void afterExecution(List<Long> ids) {
        additionalTask.run();
    }
}
