package ru.yandex.direct.core.entity.relevancematch.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModificationResult;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchUpdateContainer;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

/**
 * Комплексная операция обновления/добавления/удаления бесфразных таргетингов
 */
public class RelevanceMatchModifyOperation {
    private final RelevanceMatchDeleteOperation relevanceMatchDeleteOperation;
    private final RelevanceMatchAddOperation relevanceMatchAddOperation;
    private final RelevanceMatchUpdateOperation relevanceMatchUpdateOperation;
    private final RelevanceMatchModification relevanceMatchModification;
    private ValidationResult<RelevanceMatchModification, Defect> validationResult;
    private Result<RelevanceMatchModificationResult> result;

    private boolean prepared;
    private boolean executed;

    /**
     * @param autoPrices      нужно ли расчитывать автоматические ставки при добавлении/обновлении автотаргетингов,
     *                        если они явно не указаны, но нужны в текущей стратегии.
     *                        См. {@link RelevanceMatchAddOperation}, {@link RelevanceMatchUpdateOperation}
     * @param autoPriceParams параметры для расчета автоматических ставок. Должен быть не {@code null}, если включен
     *                        режим {@code autoPrices}
     */
    public RelevanceMatchModifyOperation(
            RelevanceMatchService relevanceMatchService,
            ClientService clientService,
            ClientId clientId, long clientUid, long operatorUid, int shard,
            RelevanceMatchModification relevanceMatchModification,
            CampaignRepository campaignRepository, AdGroupRepository adGroupRepository,
            RelevanceMatchRepository relevanceMatchRepository,
            boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams) {
        this.relevanceMatchModification = relevanceMatchModification;

        List<ModelChanges<RelevanceMatch>> relevanceMatchesForUpdate =
                relevanceMatchModification.getRelevanceMatchUpdate();

        List<RelevanceMatch> relevanceMatchesForAdd = relevanceMatchModification.getRelevanceMatchAdd();

        Set<Long> updatingRelevanceMatchIds = StreamEx.of(relevanceMatchesForUpdate)
                .map(ModelChanges::getId)
                .toSet();

        List<Long> relevanceMatchIds = StreamEx.of(updatingRelevanceMatchIds)
                .append(relevanceMatchModification.getRelevanceMatchIdsDelete())
                .toList();

        Map<Long, RelevanceMatch> relevanceMatchesByIds =
                relevanceMatchRepository.getRelevanceMatchesByIds(shard, clientId, relevanceMatchIds);

        Map<Long, RelevanceMatch> deletingRelevanceMatchesByIds = EntryStream.of(relevanceMatchesByIds)
                .filterKeys(relevanceMatchModification.getRelevanceMatchIdsDelete()::contains)
                .toMap();

        Map<Long, RelevanceMatch> updatingRelevanceMatchesByIds = EntryStream.of(relevanceMatchesByIds)
                .filterKeys(updatingRelevanceMatchIds::contains)
                .toMap();

        Set<Long> adGroupIds = StreamEx.of(updatingRelevanceMatchesByIds.values())
                .append(relevanceMatchesForAdd)
                .map(RelevanceMatch::getAdGroupId)
                .toSet();

        Set<Long> clientAdGroupIds = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIds);

        Currency currency = clientService.getWorkCurrency(clientId);

        Map<Long, Long> clientCampaignIdsByAdGroupIds =
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, clientAdGroupIds);

        List<Long> clientCampaignIds = new ArrayList<>(clientCampaignIdsByAdGroupIds.values());
        Map<Long, Campaign> clientCampaignsByIds =
                listToMap(campaignRepository.getCampaigns(shard, clientCampaignIds), Campaign::getId);

        relevanceMatchDeleteOperation = relevanceMatchService.createFullDeleteOperation(clientId, operatorUid,
                relevanceMatchModification.getRelevanceMatchIdsDelete(), deletingRelevanceMatchesByIds);

        RelevanceMatchAddContainer relevanceMatchAddOperationContainer =
                RelevanceMatchAddContainer.createRelevanceMatchAddOperationContainer(
                        operatorUid, clientId, clientCampaignsByIds,
                        clientCampaignIdsByAdGroupIds);

        relevanceMatchAddOperation = relevanceMatchService.createFullAddOperation(currency, clientId, operatorUid,
                relevanceMatchModification.getRelevanceMatchAdd(), relevanceMatchAddOperationContainer,
                autoPrices, autoPriceParams);

        Map<Long, Long> adGroupIdsByRelevanceMatchIds = EntryStream.of(updatingRelevanceMatchesByIds)
                .mapValues(RelevanceMatch::getAdGroupId)
                .toMap();

        RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer =
                RelevanceMatchUpdateContainer.createRelevanceMatchUpdateOperationContainer(
                        operatorUid, clientId, clientCampaignsByIds,
                        clientCampaignIdsByAdGroupIds,
                        adGroupIdsByRelevanceMatchIds,
                        updatingRelevanceMatchesByIds
                );

        relevanceMatchUpdateOperation = relevanceMatchService.createFullUpdateOperation(currency, clientId, clientUid,
                operatorUid, relevanceMatchModification.getRelevanceMatchUpdate(),
                relevanceMatchUpdateOperationContainer, autoPrices, autoPriceParams);
    }

    public Optional<Result<RelevanceMatchModificationResult>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;
        if (!relevanceMatchModification.getRelevanceMatchIdsDelete().isEmpty()) {
            relevanceMatchDeleteOperation.prepare();
        }
        if (!relevanceMatchModification.getRelevanceMatchAdd().isEmpty()) {
            relevanceMatchAddOperation.prepare();
        }
        if (!relevanceMatchModification.getRelevanceMatchUpdate().isEmpty()) {
            relevanceMatchUpdateOperation.prepare();
        }

        boolean failed = relevanceMatchDeleteOperation.getResult().isPresent() ||
                relevanceMatchAddOperation.getResult().isPresent() ||
                relevanceMatchUpdateOperation.getResult().isPresent();


        if (failed) {
            validationResult = buildValidationResult();
            result = Result.broken(validationResult);
        }
        return Optional.ofNullable(result);
    }


    public Result<RelevanceMatchModificationResult> prepareAndApply() {
        return prepare().orElseGet(this::apply);
    }

    private ValidationResult<RelevanceMatchModification, Defect> buildValidationResult() {
        ValidationResult<RelevanceMatchModification, Defect> validationResult =
                new ValidationResult<>(relevanceMatchModification);
        if (relevanceMatchDeleteOperation.getResult().isPresent()) {
            ValidationResult<List<Long>, Defect> deleteRelevanceMatchResult = validationResult
                    .getOrCreateSubValidationResult(field(RelevanceMatchModification.RELEVANCE_MATCH_IDS_DELETE.name()),
                            relevanceMatchModification.getRelevanceMatchIdsDelete());
            ValidationResult<?, Defect> vr =
                    relevanceMatchDeleteOperation.getResult().get().getValidationResult();
            if (vr != null) {
                transferSubNodesWithIssues(vr, deleteRelevanceMatchResult);
            }
        }

        if (relevanceMatchAddOperation.getResult().isPresent()) {
            ValidationResult<List<RelevanceMatch>, Defect> addRelevanceMatchResult = validationResult
                    .getOrCreateSubValidationResult(field(RelevanceMatchModification.RELEVANCE_MATCH_ADD.name()),
                            relevanceMatchModification.getRelevanceMatchAdd());
            ValidationResult<?, Defect> vr =
                    relevanceMatchAddOperation.getResult().get().getValidationResult();

            if (vr != null) {
                transferSubNodesWithIssues(vr, addRelevanceMatchResult);
            }
        }

        if (relevanceMatchUpdateOperation.getResult().isPresent()) {
            ValidationResult<List<ModelChanges<RelevanceMatch>>, Defect> updateRelevanceMatchResult =
                    validationResult.getOrCreateSubValidationResult(
                            field(RelevanceMatchModification.RELEVANCE_MATCH_UPDATE.name()),
                            relevanceMatchModification.getRelevanceMatchUpdate());
            ValidationResult<?, Defect> vr =
                    relevanceMatchUpdateOperation.getResult().get().getValidationResult();
            if (vr != null) {
                transferSubNodesWithIssues(vr, updateRelevanceMatchResult);
            }
        }
        return validationResult;
    }

    public Result<RelevanceMatchModificationResult> apply() {
        checkState(prepared, "prepare() must be called before apply()");
        checkState(!executed, "apply() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
        executed = true;

        List<Long> deletedRelevanceMatchIds = null;
        List<Long> addedRelevanceMatchIds = null;
        List<Long> updatedRelevanceMatchIds = null;

        if (!relevanceMatchModification.getRelevanceMatchIdsDelete().isEmpty()) {
            relevanceMatchDeleteOperation.apply();
            deletedRelevanceMatchIds =
                    mapList(relevanceMatchDeleteOperation.getResult().get().getResult(), Result::getResult);
        }
        if (!relevanceMatchModification.getRelevanceMatchAdd().isEmpty()) {
            relevanceMatchAddOperation.apply();
            addedRelevanceMatchIds =
                    mapList(relevanceMatchAddOperation.getResult().get().getResult(),
                            result -> result.getResult().getId());
        }
        if (!relevanceMatchModification.getRelevanceMatchUpdate().isEmpty()) {
            relevanceMatchUpdateOperation.apply();
            updatedRelevanceMatchIds =
                    mapList(relevanceMatchUpdateOperation.getResult().get().getResult(), Result::getResult);
        }
        RelevanceMatchModificationResult relevanceMatchModificationResult =
                new RelevanceMatchModificationResult().withRelevanceMatchDeleteResult(deletedRelevanceMatchIds)
                        .withRelevanceMatchAddResult(addedRelevanceMatchIds)
                        .withRelevanceMatchUpdateResult(updatedRelevanceMatchIds);

        validationResult = buildValidationResult();
        result =
                Result.successful(relevanceMatchModificationResult, validationResult);
        return result;
    }
}
