package ru.yandex.direct.core.entity.relevancematch.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.service.KeywordRecentStatisticsProvider;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddOperationParams;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchUpdateContainer;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchValidationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer.createRelevanceMatchAddOperationContainer;
import static ru.yandex.direct.result.ResultConverters.massResultValueConverter;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;

@ParametersAreNonnullByDefault
@Service
public class RelevanceMatchService implements EntityService<RelevanceMatch, Long> {
    private final ShardHelper shardHelper;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final RelevanceMatchValidationService relevanceMatchValidationService;
    private final ClientService clientService;
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;
    private final DslContextProvider dslContextProvider;
    private final KeywordService keywordService;
    private final KeywordRecentStatisticsProvider keywordStatisticsProvider;

    @Autowired
    public RelevanceMatchService(ShardHelper shardHelper,
                                 RelevanceMatchValidationService relevanceMatchValidationService,
                                 RelevanceMatchRepository relevanceMatchRepository,
                                 CampaignRepository campaignRepository,
                                 AdGroupRepository adGroupRepository,
                                 AggregatedStatusesRepository aggregatedStatusesRepository,
                                 ClientService clientService, LogPriceService logPriceService,
                                 MailNotificationEventService mailNotificationEventService,
                                 DslContextProvider dslContextProvider,
                                 KeywordService keywordService,
                                 @Qualifier(CoreConfiguration.DEFAULT_COPY_RECENT_STATISTICS_PROVIDER_BEAN_NAME)
                                 KeywordRecentStatisticsProvider keywordStatisticsProvider) {
        this.shardHelper = shardHelper;
        this.relevanceMatchValidationService = relevanceMatchValidationService;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.clientService = clientService;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.dslContextProvider = dslContextProvider;
        this.keywordService = keywordService;
        this.keywordStatisticsProvider = keywordStatisticsProvider;
    }

    /**
     * @param autoPrices      включает режим досчитывания недостающих ставок. См. {@link RelevanceMatchAddOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не {@code null}, если включен режим
     *                        {@code autoPrices}
     */
    public RelevanceMatchAddOperation createFullAddOperation(Currency clientCurrency, ClientId clientId,
                                                             long operatorUid,
                                                             List<RelevanceMatch> relevanceMatchList,
                                                             RelevanceMatchAddContainer relevanceMatchAddOperationContainer,
                                                             boolean autoPrices,
                                                             @Nullable ShowConditionAutoPriceParams autoPriceParams) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createAddOperation(clientCurrency, clientId, shard, operatorUid, relevanceMatchList,
                relevanceMatchAddOperationContainer, Applicability.FULL, false, autoPrices,
                autoPriceParams);
    }

    public RelevanceMatchAddOperation createFullAddOperation(Currency clientCurrency, ClientId clientId,
                                                             long operatorUid,
                                                             List<RelevanceMatch> relevanceMatchList) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        RelevanceMatchAddContainer relevanceMatchAddOperationContainer =
                createAddContainer(relevanceMatchList, operatorUid, clientId, shard);
        return createAddOperation(clientCurrency, clientId, shard, operatorUid, relevanceMatchList,
                relevanceMatchAddOperationContainer, Applicability.FULL, false, false, null);
    }

    /**
     * @param autoPrices      включает режим {@code autoPrices}, см. коммент к классу {@link RelevanceMatchAddOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не {@code null}, если включен
     *                        режим {@code autoPrices}
     */
    public RelevanceMatchAddOperation createOperationWithNonExistingAdGroups(Currency clientCurrency, ClientId clientId,
                                                                             long operatorUid,
                                                                             List<RelevanceMatch> relevanceMatchList,
                                                                             boolean autoPrices,
                                                                             @Nullable ShowConditionAutoPriceParams autoPriceParams) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return createAddOperation(clientCurrency, clientId, shard, operatorUid, relevanceMatchList,
                createRelevanceMatchAddOperationContainer(operatorUid, clientId, null, null),
                Applicability.FULL, true, autoPrices, autoPriceParams);
    }

    public RelevanceMatchAddOperation createPartialAddOperationWithAutoPrices(Currency clientCurrency,
                                                                              ClientId clientId,
                                                                              long operatorUid,
                                                                              List<RelevanceMatch> relevanceMatchList,
                                                                              ShowConditionAutoPriceParams autoPriceParams) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        RelevanceMatchAddContainer relevanceMatchAddOperationContainer =
                createAddContainer(relevanceMatchList, operatorUid, clientId, shard);

        //выключаем валидацию contextPrice для api если не включен расширенный автотаргетинг
        return createAddOperation(clientCurrency, clientId, shard, operatorUid, relevanceMatchList,
                relevanceMatchAddOperationContainer, Applicability.PARTIAL, false, true, autoPriceParams);
    }

    /**
     * @param autoPrices      включает режим {@code autoPrices}, см. коммент к классу {@link RelevanceMatchAddOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не{@code null},
     *                        если включен
     *                        режим{@code autoPrices }
     */
    private RelevanceMatchAddOperation createAddOperation(Currency clientCurrency, ClientId clientId, int shard,
                                                          long operatorUid, List<RelevanceMatch> relevanceMatchList,
                                                          RelevanceMatchAddContainer relevanceMatchAddOperationContainer,
                                                          Applicability applicability,
                                                          boolean adGroupsNonexistentOnPrepare,
                                                          boolean autoPrices,
                                                          @Nullable ShowConditionAutoPriceParams autoPriceParams
    ) {
        RelevanceMatchAddOperationParams operationParams = RelevanceMatchAddOperationParams.builder()
                .withAdGroupsNonexistentOnPrepare(adGroupsNonexistentOnPrepare)
                .withAutoPrices(autoPrices)
                .build();

        return new RelevanceMatchAddOperation(applicability, operationParams,
                relevanceMatchList,
                logPriceService, relevanceMatchValidationService, relevanceMatchRepository,
                adGroupRepository,
                dslContextProvider, keywordService,
                relevanceMatchAddOperationContainer,
                autoPriceParams, shard, clientId, clientCurrency, operatorUid);
    }

    public RelevanceMatchAddContainer createAddContainer(List<RelevanceMatch> relevanceMatchesForAdd,
                                                         long operatorUid, ClientId clientId, int shard) {
        Set<Long> adGroupIds = StreamEx.of(relevanceMatchesForAdd)
                .map(RelevanceMatch::getAdGroupId)
                .toSet();

        Set<Long> clientAdGroupIds = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIds);

        Map<Long, Long> clientCampaignIdsByAdGroupIds =
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, clientAdGroupIds);

        List<Long> clientCampaignIds = new ArrayList<>(clientCampaignIdsByAdGroupIds.values());

        Map<Long, Campaign> clientCampaignsByIds =
                listToMap(campaignRepository.getClientsCampaignsByIds(shard, clientId, clientCampaignIds),
                        Campaign::getId);

        return createRelevanceMatchAddOperationContainer(
                operatorUid, clientId, clientCampaignsByIds,
                clientCampaignIdsByAdGroupIds);
    }

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок.
     *                        См. коммент к классу {@link RelevanceMatchUpdateOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не {@code null}, если включен
     *                        режим {@code autoPrices}
     */
    RelevanceMatchUpdateOperation createFullUpdateOperation(Currency clientCurrency, ClientId clientId,
                                                            long clientUid, long operatorUid,
                                                            List<ModelChanges<RelevanceMatch>> relevanceMatchModelChangesList,
                                                            RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
                                                            boolean autoPrices,
                                                            @Nullable ShowConditionAutoPriceParams autoPriceParams) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createUpdateOperation(clientCurrency, clientId, shard, clientUid, operatorUid,
                relevanceMatchModelChangesList,
                relevanceMatchUpdateOperationContainer, Applicability.FULL, autoPrices, autoPriceParams
        );
    }

    public RelevanceMatchUpdateOperation createFullUpdateOperation(Currency clientCurrency, ClientId clientId,
                                                                   long clientUid, long operatorUid,
                                                                   List<ModelChanges<RelevanceMatch>> relevanceMatchModelChangesList) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer =
                createUpdateContainer(relevanceMatchModelChangesList, operatorUid, clientId, shard);
        return createUpdateOperation(clientCurrency, clientId, shard, clientUid, operatorUid,
                relevanceMatchModelChangesList,
                relevanceMatchUpdateOperationContainer, Applicability.FULL, false, null
        );
    }

    public RelevanceMatchUpdateOperation createPartialUpdateOperation(Currency clientCurrency, ClientId clientId,
                                                                      long clientUid, long operatorUid,
                                                                      List<ModelChanges<RelevanceMatch>> relevanceMatchModelChangesList) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer =
                createUpdateContainer(relevanceMatchModelChangesList, operatorUid, clientId, shard);

        return createUpdateOperation(clientCurrency, clientId, shard, clientUid, operatorUid,
                relevanceMatchModelChangesList,
                relevanceMatchUpdateOperationContainer, Applicability.PARTIAL, false, null
        );
    }

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок.
     *                        См. коммент к классу {@link RelevanceMatchUpdateOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не {@code null}, если включен
     *                        режим {@code autoPrices}
     */
    private RelevanceMatchUpdateOperation createUpdateOperation(Currency clientCurrency, ClientId clientId, int shard,
                                                                long clientUid, long operatorUid,
                                                                List<ModelChanges<RelevanceMatch>> relevanceMatchModelChangesList,
                                                                RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
                                                                Applicability applicability,
                                                                boolean autoPrices,
                                                                @Nullable ShowConditionAutoPriceParams autoPriceParams
    ) {
        return new RelevanceMatchUpdateOperation(applicability,
                relevanceMatchModelChangesList,
                logPriceService, mailNotificationEventService, relevanceMatchValidationService,
                relevanceMatchRepository, adGroupRepository, aggregatedStatusesRepository, dslContextProvider,
                keywordService, autoPrices, autoPriceParams,
                relevanceMatchUpdateOperationContainer,
                shard, clientId, clientUid, clientCurrency,
                operatorUid);
    }

    private RelevanceMatchUpdateContainer createUpdateContainer(
            List<ModelChanges<RelevanceMatch>> relevanceMatchesForUpdate,
            Long operatorUid, ClientId clientId, int shard) {
        Set<Long> relevanceMatchIds = StreamEx.of(relevanceMatchesForUpdate)
                .map(ModelChanges::getId)
                .toSet();

        Map<Long, RelevanceMatch> relevanceMatchesByIds =
                relevanceMatchRepository.getRelevanceMatchesByIds(shard, clientId, relevanceMatchIds);


        Set<Long> adGroupIds = StreamEx.of(relevanceMatchesByIds.values())
                .map(RelevanceMatch::getAdGroupId)
                .toSet();

        Set<Long> clientAdGroupIds = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIds);


        Map<Long, Long> clientCampaignIdsByAdGroupIds =
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, clientAdGroupIds);

        List<Long> clientCampaignIds = new ArrayList<>(clientCampaignIdsByAdGroupIds.values());
        Map<Long, Campaign> clientCampaignsByIds =
                listToMap(campaignRepository.getClientsCampaignsByIds(shard, clientId, clientCampaignIds),
                        Campaign::getId);

        Map<Long, Long> adGroupIdsByRelevanceMatchIds = EntryStream.of(relevanceMatchesByIds)
                .mapValues(RelevanceMatch::getAdGroupId)
                .toMap();

        return RelevanceMatchUpdateContainer.createRelevanceMatchUpdateOperationContainer(
                operatorUid, clientId, clientCampaignsByIds,
                clientCampaignIdsByAdGroupIds,
                adGroupIdsByRelevanceMatchIds,
                relevanceMatchesByIds);
    }

    public RelevanceMatchDeleteOperation createFullDeleteOperation(ClientId clientId, long operatorUid, List<Long> ids,
                                                                   Map<Long, RelevanceMatch> relevanceMatchesByIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createDeleteOperation(shard, operatorUid, clientId, ids, Applicability.FULL, relevanceMatchesByIds);
    }


    public RelevanceMatchDeleteOperation createDeleteOperation(ClientId clientId, long operatorUid, List<Long> ids,
                                                               Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, RelevanceMatch> relevanceMatchesByIds =
                relevanceMatchRepository.getRelevanceMatchesByIds(shard, clientId, ids);
        return createDeleteOperation(shard, operatorUid, clientId, ids, applicability, relevanceMatchesByIds);
    }

    private RelevanceMatchDeleteOperation createDeleteOperation(int shard, long operatorUid, ClientId clientId,
                                                                List<Long> ids, Applicability applicability,
                                                                Map<Long, RelevanceMatch> relevanceMatchesByIds) {
        return new RelevanceMatchDeleteOperation(applicability, ids, relevanceMatchRepository,
                relevanceMatchValidationService, logPriceService, adGroupRepository,
                dslContextProvider, relevanceMatchesByIds,
                shard, operatorUid, clientId
        );
    }

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок.
     *                        См. коммент к классам {@link RelevanceMatchAddOperation},
     *                        {@link RelevanceMatchUpdateOperation}
     * @param autoPriceParams параметры для расчета недостающих ставок. Должны быть не {@code null}, если включен
     *                        режим {@code autoPrices}
     */
    public RelevanceMatchModifyOperation createFullModifyOperation(ClientId clientId, long clientUid, long operatorUid,
                                                                   RelevanceMatchModification relevanceMatchModification,
                                                                   boolean autoPrices,
                                                                   @Nullable ShowConditionAutoPriceParams autoPriceParams) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new RelevanceMatchModifyOperation(this, clientService, clientId, clientUid, operatorUid,
                shard, relevanceMatchModification, campaignRepository,
                adGroupRepository, relevanceMatchRepository, autoPrices, autoPriceParams);
    }

    /**
     * Получить множество идентификаторов, соотвествующих RelevanceMatch клиента, по списку
     * предположительных идентификаторов
     */
    public Set<Long> getRelevanceMatchIds(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return relevanceMatchRepository.getRelevanceMatchIds(shard, clientId, ids);
    }

    public List<RelevanceMatch> getRelevanceMatchByIds(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return StreamEx.of(relevanceMatchRepository.getRelevanceMatchesByIds(shard, clientId, ids).values()).toList();
    }

    @Override
    public List<RelevanceMatch> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return getRelevanceMatchByIds(clientId, ids);
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<RelevanceMatch> entities,
                                Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Client client = checkNotNull(clientService.getClient(clientId));

        var relevanceMatchAddOperationContainer =
                createAddContainer(entities, operatorUid, clientId, shard);

        var operation = new ConvertResultOperation<>(
                createAddOperation(client.getWorkCurrency().getCurrency(),
                        clientId,
                        shard,
                        operatorUid,
                        entities,
                        relevanceMatchAddOperationContainer,
                        applicability,
                        false,
                        false, null),
                massResultValueConverter(nullSafeConverter(AddedModelId::getId)));
        return operation.prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<RelevanceMatch> entities,
                                 Applicability applicability) {
        int shard = copyContainer.getShardTo();
        Client client = copyContainer.getClientTo();

        var relevanceMatchAddOperationContainer =
                createAddContainer(entities, copyContainer.getOperatorUid(), copyContainer.getClientIdTo(), shard);

        ShowConditionAutoPriceParams autoPriceParams = new ShowConditionAutoPriceParams(
                copyContainer.getShowConditionFixedAutoPrices(), keywordStatisticsProvider);

        var operation = new ConvertResultOperation<>(
                createAddOperation(client.getWorkCurrency().getCurrency(),
                        copyContainer.getClientIdTo(),
                        shard,
                        copyContainer.getOperatorUid(),
                        entities,
                        relevanceMatchAddOperationContainer,
                        applicability,
                        false,
                        true,
                        autoPriceParams),
                massResultValueConverter(nullSafeConverter(AddedModelId::getId)));
        return operation.prepareAndApply();
    }
}
