package ru.yandex.direct.core.entity.relevancematch.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchUpdateContainer;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchValidationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent.changedSearchPriceEvent;
import static ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils.isExtendedRelevanceMatchAllowedForCampaign;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Параметр {@code autoPrices} включает режим выставления ставок, которые
 * явно не указаны, но нужны в текущей стратегии.
 * Это может быть нужно при обновлении автотаргетингов во время переходного
 * периода, когда клиентам уже дали возможность пользоваться автотаргетингом
 * в сети, и валидация уже проверяет наличие ставки в сети, но самой ставки
 * в БД пока не появилось.
 * <p>
 * Ставки расчитываются с помощью калькулятора {@link RelevanceMatchAutoPricesCalculator}
 */
public class RelevanceMatchUpdateOperation extends SimpleAbstractUpdateOperation<RelevanceMatch, Long> {
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;
    private final RelevanceMatchValidationService relevanceMatchValidationService;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final AdGroupRepository adGroupRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final DslContextProvider dslContextProvider;
    private RelevanceMatchAutoPricesCalculator autoPricesCalculator;

    private final boolean autoPrices;
    private final int shard;
    private final ClientId clientId;
    private final Long clientUid;
    private final Currency clientCurrency;
    private final Long operatorUid;
    private final RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer;

    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;

    public RelevanceMatchUpdateOperation(Applicability applicability,
                                         List<ModelChanges<RelevanceMatch>> relevanceMatchModelChangesList,
                                         LogPriceService logPriceService, MailNotificationEventService mailNotificationEventService,
                                         RelevanceMatchValidationService relevanceMatchValidationService,
                                         RelevanceMatchRepository relevanceMatchRepository,
                                         AdGroupRepository adGroupRepository,
                                         AggregatedStatusesRepository aggregatedStatusesRepository,
                                         DslContextProvider dslContextProvider,
                                         KeywordService keywordService,
                                         boolean autoPrices, @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
                                         RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
                                         int shard, ClientId clientId, Long clientUid,
                                         Currency clientCurrency,
                                         Long operatorUid) {
        super(applicability, relevanceMatchModelChangesList, id -> new RelevanceMatch().withId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.clientCurrency = clientCurrency;
        this.operatorUid = operatorUid;

        this.relevanceMatchUpdateOperationContainer = relevanceMatchUpdateOperationContainer;

        this.mailNotificationEventService = mailNotificationEventService;
        this.logPriceService = logPriceService;
        this.adGroupRepository = adGroupRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.relevanceMatchValidationService = relevanceMatchValidationService;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.dslContextProvider = dslContextProvider;
        this.autoPrices = autoPrices;

        if (autoPrices) {
            checkArgument(showConditionAutoPriceParams != null,
                    "showConditionAutoPriceParams must be specified in autoPrices mode");
            autoPricesCalculator = new RelevanceMatchAutoPricesCalculator(
                    showConditionAutoPriceParams.getKeywordRecentStatisticsProvider(),
                    keywordService,
                    clientCurrency,
                    clientId
            );
        }
    }

    @Override
    protected ValidationResult<List<ModelChanges<RelevanceMatch>>, Defect> validateModelChanges(
            List<ModelChanges<RelevanceMatch>> modelChanges) {
        return relevanceMatchValidationService.preValidateUpdateRelevanceMatches(
                modelChanges, relevanceMatchUpdateOperationContainer);
    }

    @Override
    protected Collection<RelevanceMatch> getModels(Collection<Long> ids) {
        return mapList(ids, relevanceMatchUpdateOperationContainer.getRelevanceMatchesByIds()::get);
    }

    @Override
    protected ValidationResult<List<RelevanceMatch>, Defect> validateAppliedChanges(
            ValidationResult<List<RelevanceMatch>, Defect> validationResult) {
        boolean mustHavePrices = !autoPrices;
        return relevanceMatchValidationService
                .validateUpdateRelevanceMatches(validationResult, relevanceMatchUpdateOperationContainer,
                        mustHavePrices);
    }

    @Override
    protected void beforeExecution(ExecutionStep<RelevanceMatch> executionStep) {
        Collection<AppliedChanges<RelevanceMatch>> validAppliedChanges = executionStep.getValidAppliedChanges();

        for (AppliedChanges<RelevanceMatch> validAppliedChange : validAppliedChanges) {
            Campaign campaign = relevanceMatchUpdateOperationContainer
                    .getCampaignById(validAppliedChange.getModel().getCampaignId());
            if (!isExtendedRelevanceMatchAllowedForCampaign(campaign)) {
                validAppliedChange.modify(RelevanceMatch.PRICE_CONTEXT,
                        validAppliedChange.getOldValue(RelevanceMatch.PRICE_CONTEXT));
            }
        }
        // автоматические ставки нужно обязательно считать на этапе apply(),
        // т.к. они расчитываются на основе ставок ключевых фраз из той же
        // группы, которых на этапе prepare() может еще не быть. Например
        // в комплексной операции создания/обновления группы
        if (autoPrices) {
            Map<Long, Campaign> campaignsByIds = relevanceMatchUpdateOperationContainer.getCampaignsByIds();
            autoPricesCalculator.calcAutoPricesInUpdate(validAppliedChanges, campaignsByIds);
        }
        validAppliedChanges.forEach(this::prepareStatuses);
        computeTransactionalTask(validAppliedChanges);
        computeAdditionalTask(validAppliedChanges);
    }

    private void computeAdditionalTask(Collection<AppliedChanges<RelevanceMatch>> validAppliedChanges) {
        List<RelevanceMatch> relevanceMatches = StreamEx.of(validAppliedChanges)
                .filter(changes ->
                        changes.changed(RelevanceMatch.PRICE)
                                || changes.changed(RelevanceMatch.PRICE_CONTEXT)
                ).map(AppliedChanges::getModel)
                .toList();

        List<LogPriceData> priceDataList = computeLogPriceDataList(relevanceMatches);
        List<KeywordEvent<BigDecimal>> events = computeMailEvents(validAppliedChanges);

        additionalTask = () -> {
            if (!priceDataList.isEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid);
            }
            if (!events.isEmpty()) {
                mailNotificationEventService.queueEvents(operatorUid, clientId, events);
            }
        };
    }

    private void prepareStatuses(AppliedChanges<RelevanceMatch> appliedChanges) {
        LocalDateTime now = LocalDateTime.now();
        if (isBsSyncStatusChanged(appliedChanges)) {
            appliedChanges.modify(RelevanceMatch.STATUS_BS_SYNCED, StatusBsSynced.NO);
        }

        if (appliedChanges.hasActuallyChangedProps()) {
            appliedChanges.modify(RelevanceMatch.LAST_CHANGE_TIME, now);
        }
    }

    private void computeTransactionalTask(Collection<AppliedChanges<RelevanceMatch>> validAppliedChanges) {
        Set<Long> adGroupIdsToResend = StreamEx.of(validAppliedChanges)
                .filter(this::isAdGroupBsSyncStatusChanged)
                .map(AppliedChanges::getModel)
                .map(RelevanceMatch::getAdGroupId)
                .toSet();

        //Если на группе включился АТ и она не промодерирована - переотправляем на модерацию
        //т.к. с АТ, возможно, она будет пропущена
        List<Long> adGroupIdsToModerate = StreamEx.of(validAppliedChanges)
                .filter(isChanged(RelevanceMatch.IS_SUSPENDED))
                .map(AppliedChanges::getModel)
                .map(RelevanceMatch::getAdGroupId)
                .toList();

        Set<Long> adGroupIdsToUpdateLastChange = new HashSet<>();
        adGroupIdsToUpdateLastChange.addAll(adGroupIdsToModerate);
        adGroupIdsToUpdateLastChange.addAll(adGroupIdsToResend);

        List<Long> relevanceMatchIdsToResetStatuses = StreamEx.of(validAppliedChanges)
                .filter(isChanged(RelevanceMatch.IS_SUSPENDED))
                .map(AppliedChanges::getModel)
                .map(RelevanceMatch::getId)
                .toList();

        transactionalAdditionalTask = conf -> {
            if (!adGroupIdsToResend.isEmpty()) {
                adGroupRepository.updateStatusBsSyncedExceptNew(conf, adGroupIdsToResend, StatusBsSynced.NO);
            }
            if (!adGroupIdsToModerate.isEmpty()) {
                adGroupRepository.dropStatusModerateExceptDraftsAndModerated(conf, adGroupIdsToModerate);
            }

            adGroupRepository.updateLastChange(conf, adGroupIdsToUpdateLastChange);

            aggregatedStatusesRepository.markKeywordStatusesAsObsolete(conf.dsl(), null,
                    relevanceMatchIdsToResetStatuses);
        };
    }

    private boolean isBsSyncStatusChanged(AppliedChanges<RelevanceMatch> appliedChanges) {
        return appliedChanges.changed(RelevanceMatch.PRICE) ||
                appliedChanges.changed(RelevanceMatch.AUTOBUDGET_PRIORITY) ||
                appliedChanges.changed(RelevanceMatch.PRICE_CONTEXT);
    }

    private boolean isAdGroupBsSyncStatusChanged(AppliedChanges<RelevanceMatch> appliedChanges) {
        return appliedChanges.changed(RelevanceMatch.IS_DELETED) ||
                appliedChanges.changed(RelevanceMatch.IS_SUSPENDED) ||
                appliedChanges.changed(RelevanceMatch.HREF_PARAM1) ||
                appliedChanges.changed(RelevanceMatch.HREF_PARAM2) ||
                appliedChanges.changed(RelevanceMatch.RELEVANCE_MATCH_CATEGORIES);
    }

    private List<LogPriceData> computeLogPriceDataList(Collection<RelevanceMatch> newRelevanceMatch) {
        Function<RelevanceMatch, LogPriceData> relevanceMatchToLogFn = relevanceMatch -> {
            Double priceSearch = relevanceMatch.getPrice() != null ? relevanceMatch.getPrice().doubleValue() : 0;
            Double priceContext =
                    relevanceMatch.getPriceContext() != null ? relevanceMatch.getPriceContext().doubleValue() : 0;
            return new LogPriceData(
                    relevanceMatch.getCampaignId(),
                    relevanceMatch.getAdGroupId(),
                    relevanceMatch.getId(),
                    priceContext,
                    priceSearch,
                    clientCurrency.getCode(),
                    LogPriceData.OperationType.UPDATE_1);
        };
        return mapList(newRelevanceMatch, relevanceMatchToLogFn);
    }

    private List<KeywordEvent<BigDecimal>> computeMailEvents(Collection<AppliedChanges<RelevanceMatch>> changes) {
        Set<Long> affectedAdGroupIds = StreamEx.of(changes)
                .map(AppliedChanges::getModel)
                .map(RelevanceMatch::getAdGroupId)
                .toSet();
        Map<Long, AdGroupName> adGroupIdToAdGroup = getAdGroupIdToAdGroupNameWithCheck(affectedAdGroupIds);
        List<KeywordEvent<BigDecimal>> addKeywordEvents = StreamEx.of(changes)
                .filter(change -> change.changed(RelevanceMatch.PRICE)
                        && change.getOldValue(RelevanceMatch.PRICE) != null
                        && change.getNewValue(RelevanceMatch.PRICE) != null)
                .map(change -> {
                    Long adGroupId = change.getModel().getAdGroupId();
                    long campaignId = change.getModel().getCampaignId();
                    String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();
                    BigDecimal oldValue =
                            checkNotNull(change.getOldValue(RelevanceMatch.PRICE), "Old price value can't be null");
                    BigDecimal newValue =
                            checkNotNull(change.getNewValue(RelevanceMatch.PRICE), "New price value can't be null");
                    return changedSearchPriceEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName,
                            oldValue, newValue);
                }).toList();
        return addKeywordEvents;
    }

    private Map<Long, AdGroupName> getAdGroupIdToAdGroupNameWithCheck(Set<Long> adGroupIds) {
        Map<Long, AdGroupName> adGroupIdToAdGroup = adGroupRepository.getAdGroupNames(shard, clientId, adGroupIds);
        Set<Long> notFoundAdGroupIds = Sets.difference(adGroupIds, adGroupIdToAdGroup.keySet());
        checkState(notFoundAdGroupIds.isEmpty(), "can't get adGroups for adGroup ids: " + notFoundAdGroupIds);
        return adGroupIdToAdGroup;
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<RelevanceMatch>> applicableAppliedChanges) {
        for (AppliedChanges<RelevanceMatch> appliedChanges : applicableAppliedChanges) {
            RelevanceMatch relevanceMatch = appliedChanges.getModel();
            Campaign campaign = relevanceMatchUpdateOperationContainer
                    .getCampaignById(relevanceMatch.getCampaignId());
            if (!isExtendedRelevanceMatchAllowedForCampaign(campaign)) {
                RelevanceMatchUtils.resetExtendedAttributes(relevanceMatch);
            }
        }
        TransactionalRunnable saveFn = conf -> {
            Set<Long> affectedAdGroupIds = StreamEx.of(applicableAppliedChanges)
                    .map(AppliedChanges::getModel)
                    .map(RelevanceMatch::getAdGroupId)
                    .toSet();
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

            relevanceMatchRepository.update(conf, applicableAppliedChanges);
            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current()
                .profile("relevanceMatchUpdate:write", "", applicableAppliedChanges.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }
        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<RelevanceMatch> executionStep) {
        additionalTask.run();
    }
}
