package ru.yandex.direct.core.entity.relevancematch.valdiation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bids.validation.AutobudgetValidator;
import ru.yandex.direct.core.entity.bids.validation.BidPriceValidator2;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects;
import ru.yandex.direct.core.entity.relevancematch.container.AdGroupInfoForRelevanceMatchAdd;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchAddContainer;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchUpdateContainer;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.contextPriceNotNullForManualStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextWontApplyNetIsSwitchedOff;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.searchPriceNotNullForManualStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.strategyIsSet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.relevancematch.Constants.CAMPAIGN_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES;
import static ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchUtils.isExtendedRelevanceMatchAllowedForCampaign;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.maxRelevanceMatchesInAdGroup;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.relevanceMatchAlreadyDeleted;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.relevanceMatchAlreadySuspended;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.relevanceMatchCantBeUsedInAutoBudgetCompany;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.relevanceMatchCantBeUsedWhenSearchIsStopped;
import static ru.yandex.direct.core.entity.relevancematch.valdiation.RelevanceMatchDefects.relevanceMatchNotSuspended;
import static ru.yandex.direct.core.validation.constraints.Constraints.allowedBannerLetters;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class RelevanceMatchValidationService {
    private static final CampaignAccessDefects ACCESS_DEFECTS =
            AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
                    .withTypeNotAllowable(objectNotFound())
                    .withNotVisible(objectNotFound())
                    .withTypeNotSupported(campaignTypeNotSupported())
                    .withNoRights(noRights())
                    .build();

    public static final int MAX_RELEVANCE_MATCHES_IN_GROUP = 1;
    private static final Integer MAX_ELEMENTS_PER_OPERATION = 10_000;
    public static final Integer MAX_HREF_PARAM_LENGTH = 255;

    private final ModelChangesValidationTool modelChangesPreValidationTool;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public RelevanceMatchValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.modelChangesPreValidationTool = ModelChangesValidationTool.builder()
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION).build();
    }

    public ValidationResult<List<Long>, Defect> validateDeleteRelevanceMatches(List<Long> relevanceMatchIds,
                                                                               Map<Long, RelevanceMatch> relevanceMatchesByIds, Long operatorUid, ClientId clientId) {
        //из операции модификации эта валидация должна проходить всегда, т.к. удаляемые ids мы вычисляем сами
        Set<Long> existClientsRelevanceMatchIds = relevanceMatchesByIds.keySet();

        Constraint<Long, Defect> accessConstraint = campaignSubObjectAccessCheckerFactory
                .newRelevanceMatchChecker(operatorUid, clientId, relevanceMatchIds)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();

        ListValidationBuilder<Long, Defect> vb =
                ListValidationBuilder.<Long, Defect>of(relevanceMatchIds)
                        .checkEach(unique())
                        .checkEach(inSet(existClientsRelevanceMatchIds), objectNotFound())
                        .checkEach(relevanceMatchIsNotDeleted(relevanceMatchesByIds), When.isValid())
                        .checkEach(accessConstraint, When.isValid());
        return vb.getResult();
    }

    private Constraint<Long, Defect> relevanceMatchIsNotDeleted(
            Map<Long, RelevanceMatch> clientsRelevanceMatches) {
        Predicate<Long> isNotDeleted = id -> !clientsRelevanceMatches.get(id).getIsDeleted();
        return Constraint.fromPredicate(isNotDeleted, relevanceMatchAlreadyDeleted());
    }

    public ValidationResult<List<ModelChanges<RelevanceMatch>>, Defect> preValidateUpdateRelevanceMatches(
            List<ModelChanges<RelevanceMatch>> modelChanges,
            RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer) {
        ValidationResult<List<ModelChanges<RelevanceMatch>>, Defect> validationResult =
                modelChangesPreValidationTool.validateModelChangesList(modelChanges,
                        relevanceMatchUpdateOperationContainer.getRelevanceMatchesByIds().keySet());

        Map<Long, RelevanceMatch> existingRelevanceMatches =
                relevanceMatchUpdateOperationContainer.getRelevanceMatchesByIds();

        Set<Long> suspendedIds = existingRelevanceMatches.values().stream()
                .filter(RelevanceMatch::getIsSuspended)
                .map(RelevanceMatch::getId)
                .collect(toSet());

        ListValidationBuilder<ModelChanges<RelevanceMatch>, Defect> vb =
                new ListValidationBuilder<>(validationResult);
        vb.checkEachBy(changes -> preValidateUpdateRelevanceMatch(relevanceMatchUpdateOperationContainer, changes,
                        suspendedIds),
                When.isValid());

        return vb.getResult();
    }

    private ValidationResult<ModelChanges<RelevanceMatch>, Defect> preValidateUpdateRelevanceMatch(
            RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
            ModelChanges<RelevanceMatch> modelChanges,
            Set<Long> suspendedIds) {
        ItemValidationBuilder<ModelChanges<RelevanceMatch>, Defect> vb =
                ItemValidationBuilder.of(modelChanges);


        Campaign campaign = relevanceMatchUpdateOperationContainer.getCampaignByRelevanceMatchId(modelChanges.getId());

        Boolean isAutobudget = campaign.getAutobudget();
        Boolean isSearchStop = campaign.getStrategy().isSearchStop();

        vb.check(relevanceMatchPriceIsNotChanged(relevanceMatchCantBeUsedInAutoBudgetCompany(),
                        isExtendedRelevanceMatchAllowedForCampaign(campaign)),
                When.isTrue(isAutobudget));
        //Показы на поиске выключены, расширенный автотаргетинг недоступен изменение условий бесфразного таргетинга
        // невозможно
        vb.check(relevanceMatchPriceIsNotChanged(relevanceMatchCantBeUsedWhenSearchIsStopped(),
                        isExtendedRelevanceMatchAllowedForCampaign(campaign)),
                When.isTrue(isSearchStop && !isExtendedRelevanceMatchAllowedForCampaign(campaign)));

        Long id = modelChanges.getId();
        Boolean suspendChange = modelChanges.getPropIfChanged(RelevanceMatch.IS_SUSPENDED);

        vb.item(id, RelevanceMatch.ID.name())
                .checkBy(setSameValueSuspendValidator(suspendChange, suspendedIds), When.isValid());

        return vb.getResult();
    }

    /**
     * Проверка на измненение флага c тем же значением
     */
    private Validator<Long, Defect> setSameValueSuspendValidator(Boolean suspendChange,
                                                                 Set<Long> suspendedIds) {
        return id -> {
            ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(id);
            if (suspendChange != null && suspendChange) {
                vb.weakCheck(notInSet(suspendedIds), relevanceMatchAlreadySuspended());
            } else if (suspendChange != null) {
                vb.weakCheck(inSet(suspendedIds), relevanceMatchNotSuspended());
            }
            return vb.getResult();
        };
    }


    private Constraint<ModelChanges<RelevanceMatch>, Defect> relevanceMatchPriceIsNotChanged(
            Defect<Void> defect,
            boolean extendedRelevanceMatchEnabled) {
        Predicate<ModelChanges<RelevanceMatch>> priceIsNotChanged =
                changes -> {

                    boolean searchPriceIsNotChanged = !changes.isPropChanged(RelevanceMatch.PRICE)
                            || changes.getChangedProp(RelevanceMatch.PRICE) == null;
                    boolean contextPriceIsNotChanged = !changes.isPropChanged(RelevanceMatch.PRICE_CONTEXT)
                            || changes.getChangedProp(RelevanceMatch.PRICE_CONTEXT) == null;

                    return searchPriceIsNotChanged && (!extendedRelevanceMatchEnabled || contextPriceIsNotChanged);
                };
        return Constraint.fromPredicate(priceIsNotChanged, defect);
    }

    /**
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    public ValidationResult<List<RelevanceMatch>, Defect> validateUpdateRelevanceMatches(
            ValidationResult<List<RelevanceMatch>, Defect> preValidationResult,
            RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
            boolean mustHavePrices) {
        List<Long> adGroupIds = StreamEx.of(preValidationResult.getValue())
                .map(RelevanceMatch::getAdGroupId)
                .nonNull()
                .toList();
        CampaignSubObjectAccessChecker checker =
                campaignSubObjectAccessCheckerFactory.newAdGroupChecker(
                        relevanceMatchUpdateOperationContainer.getOperatorUid(),
                        relevanceMatchUpdateOperationContainer.getClientId(), adGroupIds);
        CampaignSubObjectAccessValidator adGroupAccessValidator = checker.createValidator(
                CampaignAccessType.READ_WRITE, ACCESS_DEFECTS);

        ListValidationBuilder<RelevanceMatch, Defect> vb = new ListValidationBuilder<>(preValidationResult);
        vb.checkEachBy(
                x -> validateUpdateRelevanceMatch(x, relevanceMatchUpdateOperationContainer, adGroupAccessValidator,
                        mustHavePrices),
                When.isValid());
        return vb.getResult();
    }

    /**
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    private ValidationResult<RelevanceMatch, Defect> validateUpdateRelevanceMatch(
            RelevanceMatch relevanceMatch,
            RelevanceMatchUpdateContainer relevanceMatchUpdateOperationContainer,
            CampaignSubObjectAccessValidator adGroupAccessValidator,
            boolean mustHavePrices) {
        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);
        vb.item(RelevanceMatch.AD_GROUP_ID)
                .check(notNull())
                .checkBy(adGroupAccessValidator, When.isValid());
        vb.checkBy(x -> validateStrategy(x,
                        relevanceMatchUpdateOperationContainer.getCampaignByRelevanceMatchId(x.getId()),
                        mustHavePrices), When.isValid())
                .checkBy(this::validateHrefParams)
                .checkBy(rm -> validateRelevanceMatchCategories(rm,
                        relevanceMatchUpdateOperationContainer.getCampaignByRelevanceMatchId(rm.getId())));
        return vb.getResult();
    }

    /**
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    public ValidationResult<List<RelevanceMatch>, Defect> validateAddRelevanceMatches(
            ValidationResult<List<RelevanceMatch>, Defect> preValidationResult,
            RelevanceMatchAddContainer relevanceMatchAddOperationContainer,
            boolean mustHavePrices) {
        List<RelevanceMatch> relevanceMatchList = preValidationResult.getValue();
        Map<Long, Integer> newRelevanceMatchCountByAdGroupIds =
                StreamEx.of(relevanceMatchList)
                        .mapToEntry(RelevanceMatch::getAdGroupId)
                        .invert()
                        .collapseKeys()
                        .mapValues(List::size)
                        .toMap();

        Set<Long> adGroupIds = newRelevanceMatchCountByAdGroupIds.keySet();
        CampaignSubObjectAccessChecker checker =
                campaignSubObjectAccessCheckerFactory.newAdGroupChecker(
                        relevanceMatchAddOperationContainer.getOperatorUid(),
                        relevanceMatchAddOperationContainer.getClientId(), adGroupIds);
        CampaignSubObjectAccessValidator adGroupAccessValidator = checker.createValidator(
                CampaignAccessType.READ_WRITE, ACCESS_DEFECTS);

        ListValidationBuilder<RelevanceMatch, Defect> vb =
                new ListValidationBuilder<>(preValidationResult)
                        .checkEachBy(relevanceMatch -> validateAdGroupId(relevanceMatch,
                                relevanceMatchAddOperationContainer, adGroupAccessValidator))
                        .checkEachBy(relevanceMatch -> {
                            Campaign campaign = relevanceMatchAddOperationContainer
                                    .getCampaignByAdGroupId(relevanceMatch.getAdGroupId());
                            Integer relevanceMatchesCount =
                                    newRelevanceMatchCountByAdGroupIds.get(relevanceMatch.getAdGroupId());
                            return validateCommon(relevanceMatch, campaign, relevanceMatchesCount,
                                    mustHavePrices);
                        }, When.isValid());
        return vb.getResult();
    }

    /**
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    public ValidationResult<List<RelevanceMatch>, Defect> validateAddRelevanceMatchesWithNonexistentAdGroups(
            ValidationResult<List<RelevanceMatch>, Defect> preValidationResult,
            Map<Integer, AdGroupInfoForRelevanceMatchAdd> relevanceMatchInfos,
            boolean mustHavePrices) {
        Map<Integer, Integer> newRelevanceMatchCountByAdGroupIndex =
                EntryStream.of(relevanceMatchInfos)
                        .mapValues(AdGroupInfoForRelevanceMatchAdd::getAdGroupIndex)
                        .invert()
                        .collapseKeys()
                        .mapValues(List::size)
                        .toMap();

        ListValidationBuilder<RelevanceMatch, Defect> vb = new ListValidationBuilder<>(preValidationResult)
                .checkEachBy((index, relevanceMatch) -> {
                    Campaign campaign = relevanceMatchInfos.get(index).getCampaign();
                    Integer adGroupIndex = relevanceMatchInfos.get(index).getAdGroupIndex();
                    Integer relevanceMatchesCount = newRelevanceMatchCountByAdGroupIndex.get(adGroupIndex);
                    return validateCommon(relevanceMatch, campaign, relevanceMatchesCount,
                            mustHavePrices);
                });
        return vb.getResult();
    }

    /**
     * Общие проверки для добавления бесфразных таргетингов с группами или без
     *
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    private ValidationResult<RelevanceMatch, Defect> validateCommon(RelevanceMatch relevanceMatch,
                                                                    Campaign campaign, Integer relevanceMatchesCount,
                                                                    boolean mustHavePrices) {

        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);
        vb.checkBy(rm -> validateStrategy(relevanceMatch, campaign, mustHavePrices),
                When.isValid());
        vb.checkBy(this::validateHrefParams);
        vb.check(relevanceMatchesNoMoreThanMax(relevanceMatchesCount), When.isValid());
        vb.check(campaignTypeIsSupported(campaign), When.isValid());
        vb.checkBy(rm -> validateRelevanceMatchCategories(rm, campaign));
        return vb.getResult();
    }

    private ValidationResult<RelevanceMatch, Defect> validateAdGroupId(RelevanceMatch relevanceMatch,
                                                                       RelevanceMatchAddContainer relevanceMatchAddOperationContainer,
                                                                       CampaignSubObjectAccessValidator adGroupAccessValidator) {
        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);
        vb.item(RelevanceMatch.AD_GROUP_ID)
                .check(notNull())
                .checkBy(adGroupAccessValidator, When.isValid())
                .check(inSet(relevanceMatchAddOperationContainer.getCampaignIdsByAdGroupIds().keySet()),
                        objectNotFound(), When.isValid());
        return vb.getResult();
    }


    private Constraint<RelevanceMatch, Defect> campaignTypeIsSupported(Campaign campaign) {
        Predicate<RelevanceMatch> validCampaignType = relevanceMatch -> {
            CampaignType type = campaign.getType();
            return type == CampaignType.TEXT || type == CampaignType.MOBILE_CONTENT
                    || type == CampaignType.CONTENT_PROMOTION;
        };
        return Constraint.fromPredicate(validCampaignType, CampaignDefects.campaignTypeNotSupported());
    }

    /**
     * @param mustHavePrices нужно проверить, что присутствуют ставки, нужные в текущей стратегии
     */
    private ValidationResult<RelevanceMatch, Defect> validateStrategy(RelevanceMatch relevanceMatch,
                                                                      Campaign campaign, boolean mustHavePrices) {
        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);

        DbStrategy strategy = campaign.getStrategy();
        vb.check(strategyIsSet(strategy));

        Currency currency = campaign.getCurrency().getCurrency();

        vb.item(RelevanceMatch.AUTOBUDGET_PRIORITY)
                .check(notNull(), When.isTrue(strategy != null && mustHavePrices && strategy.isAutoBudget()))
                .checkBy(new AutobudgetValidator(), When.notNull());
        vb.item(RelevanceMatch.PRICE)
                .check(searchPriceNotNullForManualStrategy(strategy), When.isTrue(strategy != null && mustHavePrices))
                .checkBy(new BidPriceValidator2(currency), When.notNull())
                .weakCheck(priceSearchIsAcceptedForStrategy(strategy), When.isTrue(strategy != null));

        if (isExtendedRelevanceMatchAllowedForCampaign(campaign)) {
            vb.item(RelevanceMatch.PRICE_CONTEXT)
                    .check(contextPriceNotNullForManualStrategy(strategy), When.isTrue(strategy != null && mustHavePrices))
                    .checkBy(new BidPriceValidator2(currency), When.notNull())
                    .weakCheck(priceContextIsAcceptedForStrategy(strategy), When.isTrue(strategy != null));
        } else {
            vb.item(RelevanceMatch.PRICE_CONTEXT)
                    .weakCheck(priceContextWontApplyNetIsSwitchedOff());
        }

        return vb.getResult();
    }

    private ValidationResult<RelevanceMatch, Defect> validateHrefParams(RelevanceMatch relevanceMatch) {
        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);

        vb.item(RelevanceMatch.HREF_PARAM1)
                .check(maxStringLength(MAX_HREF_PARAM_LENGTH))
                .check(allowedBannerLetters(), When.isValid());
        vb.item(RelevanceMatch.HREF_PARAM2)
                .check(maxStringLength(MAX_HREF_PARAM_LENGTH))
                .check(allowedBannerLetters(), When.isValid());
        return vb.getResult();
    }

    private ValidationResult<RelevanceMatch, Defect> validateRelevanceMatchCategories(RelevanceMatch relevanceMatch,
                                                                                      Campaign campaign) {
        ModelItemValidationBuilder<RelevanceMatch> vb = ModelItemValidationBuilder.of(relevanceMatch);
        vb.item(RelevanceMatch.RELEVANCE_MATCH_CATEGORIES)
                .check(campaignTypeSupportsRelevanceMatchCategories(campaign.getType()),
                        When.isFalse(isEmpty(relevanceMatch.getRelevanceMatchCategories())));
        return vb.getResult();
    }

    private Constraint<RelevanceMatch, Defect> relevanceMatchesNoMoreThanMax(Integer count) {
        return Constraint.fromPredicate(rm -> count <= MAX_RELEVANCE_MATCHES_IN_GROUP, maxRelevanceMatchesInAdGroup());
    }

    private Constraint<Set<RelevanceMatchCategory>, Defect> campaignTypeSupportsRelevanceMatchCategories(CampaignType campaignType) {
        return Constraint.fromPredicate(rm -> CAMPAIGN_TYPES_ALLOWED_FOR_RELEVANCE_MATCH_CATEGORIES.contains(campaignType),
                CampaignDefects.inconsistentCampaignType());
    }
}
