package ru.yandex.direct.core.entity.retargeting.converter;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.CollectionUtils;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ExperimentConverter {
    private ExperimentConverter() {
    }

    public static RetargetingCondition createRetargetingConditionForAbSegmentStatistic(
            LocalDateTime now,
            ClientId clientId,
            List<Long> sectionIds,
            Map<Long, List<Long>> clientAbSegmentIdsBySectionId) {

        if (sectionIds == null) {
            return null;
        }

        List<Long> sortedSectionIds = StreamEx.of(sectionIds)
                .sorted()
                .toList();

        return (RetargetingCondition) new RetargetingCondition()
                .withName("")
                .withDescription("")
                .withDeleted(false)
                .withLastChangeTime(now)
                .withClientId(clientId.asLong())
                .withType(ConditionType.ab_segments)
                .withRules(mapList(sortedSectionIds,
                        sectionId -> ruleForExperiments(sectionId, clientAbSegmentIdsBySectionId.get(sectionId))));
    }

    @Nullable
    public static RetargetingCondition createRetargetingConditionForAbSegment(
            LocalDateTime now,
            ClientId clientId,
            @Nullable List<Long> abSegmentIds,
            Map<Long, List<Long>> clientAbSegmentIdsBySectionId) {

        if (abSegmentIds == null) {
            return null;
        }

        Map<Long, Long> sectionIdByAbSegmentId = EntryStream.of(clientAbSegmentIdsBySectionId)
                .flatMapValues(Collection::stream)
                .invert()
                .toMap();
        Map<Long, List<Long>> abSegmentsBySectionId = StreamEx.of(abSegmentIds)
                .mapToEntry(sectionIdByAbSegmentId::get)
                .invert()
                .grouping();

        List<Long> sectionIds = StreamEx.of(abSegmentsBySectionId.keySet())
                .sorted()
                .toList();

        return (RetargetingCondition) new RetargetingCondition()
                .withName("")
                .withDescription("")
                .withLastChangeTime(now)
                .withDeleted(false)
                .withClientId(clientId.asLong())
                .withType(ConditionType.ab_segments)
                .withRules(mapList(sectionIds,
                        sectionId -> ruleForExperiments(sectionId, abSegmentsBySectionId.get(sectionId))));
    }

    private static Rule ruleForExperiments(Long sectionId, List<Long> goalIds) {
        List<Goal> goals = StreamEx.of(goalIds)
                .sorted()
                .map(ExperimentConverter::goalFromId)
                .toList();
        return new Rule()
                .withType(RuleType.OR)
                .withSectionId(sectionId)
                .withGoals(goals);
    }

    private static Goal goalFromId(Long goalId) {
        return (Goal) new Goal().withId(goalId);
    }

    /**
     * Возвращает ids если список не пуст, иначе null
     * <p>
     * Нужно до того, как ручка ru.yandex.direct.grid.processing.service.showcondition.
     * RetargetingMutationsGraphQlService#findOrCreateExperimentsConditions
     * исчезнет в рамках <a href="DIRECT-112091">DIRECT-112091</a>
     * <p>
     * Соответствующая проверка переедет будет в
     * ru.yandex.direct.core.entity.retargeting.converter.ExperimentConverter#createRetargetingConditionForAbSegment
     *
     * @param ids - входной список ids
     * @return возвращает ids если список не пуст, иначе null
     */
    @Nullable
    public static List<Long> getExperimentIdsIfNotEmpty(@Nullable List<Long> ids) {
        if (!CollectionUtils.isEmpty(ids)) {
            return ids;
        }
        return null;
    }
}
