package ru.yandex.direct.core.entity.retargeting.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.SelectJoinStep;
import org.jooq.exception.DataAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingConditionValidationData;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionGoal;
import ru.yandex.direct.core.entity.retargeting.service.InterestLinkFactory;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbschema.ppc.tables.records.RetargetingConditionsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.util.RepositoryUtils.FALSE;
import static ru.yandex.direct.common.util.RepositoryUtils.TRUE;
import static ru.yandex.direct.core.entity.retargeting.Constants.DEFAULT_LIMIT;
import static ru.yandex.direct.core.entity.retargeting.Constants.DEFAULT_OFFSET;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition.calcNegativeByRules;
import static ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings.conditionTypeToDb;
import static ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepositoryMapperProvider.createRetargetingConditionMapper;
import static ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType.geo_segments;
import static ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.MediaplanBidsRetargeting.MEDIAPLAN_BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class RetargetingConditionRepository {

    public static final String PROPERTY_INTEREST = "interest";
    public static final String PROPERTY_NEGATIVE = "negative";
    public static final String PROPERTY_AUTO_RETARGETING = "autoretargeting";

    private static final Set<RetargetingConditionsRetargetingConditionsType> RET_COND_TYPES_FOR_DEFAULT_OPERATIONS =
            ImmutableSet.of(RetargetingConditionsRetargetingConditionsType.metrika_goals,
                    RetargetingConditionsRetargetingConditionsType.dmp);

    private static final Set<RetargetingConditionsRetargetingConditionsType> RET_COND_TYPES_INCLUDED_IN_MAX_AMOUNT =
            ImmutableSet.of(RetargetingConditionsRetargetingConditionsType.metrika_goals,
                    RetargetingConditionsRetargetingConditionsType.dmp,
                    RetargetingConditionsRetargetingConditionsType.shortcuts);

    private static final Set<RetargetingConditionsRetargetingConditionsType> IGNORED_RET_COND_TYPES =
            ImmutableSet.of(RetargetingConditionsRetargetingConditionsType.brandsafety);

    private static final Logger logger = LoggerFactory.getLogger(RetargetingConditionRepository.class);

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final RetargetingGoalsRepository retGoalsRepository;
    private final InterestLinkFactory interestLinkFactory;
    private final JooqMapperWithSupplier<RetargetingCondition> retargetingConditionMapper;

    @Autowired
    public RetargetingConditionRepository(
            DslContextProvider dslContextProvider, ShardHelper shardHelper,
            RetargetingGoalsRepository retGoalsRepository,
            InterestLinkFactory interestLinkFactory) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.retGoalsRepository = retGoalsRepository;
        this.interestLinkFactory = interestLinkFactory;
        this.retargetingConditionMapper = createRetargetingConditionMapper();
    }

    public List<RetargetingCondition> getFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                        LimitOffset limitOffset) {
        return getFromRetargetingConditionsTable(shard, clientId, null, null, null, null, limitOffset);
    }

    public List<RetargetingCondition> getFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                        Collection<Long> ids, LimitOffset limitOffset) {
        return getFromRetargetingConditionsTable(shard, clientId, ids, null, null, null, limitOffset);
    }

    public List<RetargetingCondition> getFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                        @Nullable Collection<Long> ids,
                                                                        Collection<Long> adGroupIds, String filter,
                                                                        Collection<RetargetingConditionsRetargetingConditionsType> types,
                                                                        LimitOffset limitOffset) {
        List<RetargetingCondition> retConditions =
                getRetConditionsFromRetargetingConditionsTable(shard, clientId, ids, adGroupIds, filter, types,
                        limitOffset);
        populateRetConditionsFromRetargetingGoalsTable(shard, retConditions);
        return retConditions;
    }

    public List<RetargetingCondition> getFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                        Collection<Long> ids) {
        return getFromRetargetingConditionsTable(shard, clientId, ids, maxLimited());
    }

    /**
     * This method is used to get <b>only</b> IDs of existing RetargetingConditions searched by given {@code ids}
     *
     * @param shard    shard
     * @param clientId client ID
     * @param ids      {@link Collection} of IDs to search
     * @return {@link List} of those ids which are present or all RetargetingConditions ids id {@code ids} is {@code
     * null}
     */
    public List<Long> getExistingIds(int shard, ClientId clientId, Collection<Long> ids) {
        List<RetargetingCondition> retConditions = getRetConditionsFromRetargetingConditionsTable(shard, clientId, ids);
        return retConditions
                .stream()
                .map(RetargetingCondition::getId)
                .collect(toList());
    }

    /**
     * Получаем список всех {@link RetargetingCondition}, которые являются ссылками на категории интересов.
     * <p>
     * У таких {@link RetargetingCondition} установлено {@code PROPERTIES=`interest`}
     *
     * @return список {@link RetargetingCondition}, являющихся ссылками на категории интересов
     */
    public List<InterestLink> getExistingInterest(int shard, ClientId clientId) {
        return mapList(getInterestFromRetargetingConditionsTable(shard, clientId, null,
                limited((int) DEFAULT_LIMIT, (int) DEFAULT_OFFSET)), interestLinkFactory::from);
    }

    /**
     * Получаем список {@link RetargetingCondition}, которые являются ссылками на категории интересов, по списку Id.
     * <p>
     * У таких {@link RetargetingCondition} установлено {@code PROPERTIES=`interest`}
     *
     * @return список {@link RetargetingCondition}, являющихся ссылками на категории интересов
     */
    public List<InterestLink> getInterestByIds(int shard, ClientId clientId, Collection<Long> ids) {
        return mapList(getInterestFromRetargetingConditionsTable(shard, clientId, ids,
                limited(ids.size(), (int) DEFAULT_OFFSET)), interestLinkFactory::from);
    }

    /**
     * То же, что и в {@link #add}, только без создания retargeting_goals
     */
    public List<Long> addInterests(int shard, List<InterestLink> interestLinks) {
        List<RetargetingCondition> retConditionsWithInterest =
                mapList(interestLinks, InterestLink::asRetargetingCondition);
        generateIdsForRetargetingConditions(retConditionsWithInterest);
        addRetConditionsToRetargetingConditionsTable(shard, retConditionsWithInterest);
        return mapList(retConditionsWithInterest, RetargetingCondition::getId);
    }

    public List<Long> add(int shard, Collection<RetargetingCondition> retConditions) {
        generateIdsForRetargetingConditions(retConditions);
        addRetConditionsToRetargetingConditionsTable(shard, retConditions);
        addRetConditionsToRetargetingGoalsTable(shard, retConditions);
        return mapList(retConditions, RetargetingCondition::getId);
    }

    /**
     * Получение списка {@link RetargetingConditionValidationData} по клиенту
     * с типами 'metrika_goals' и 'dmp', исключая удаленные и авторетаргетинги
     */
    public List<RetargetingConditionValidationData> getValidationData(Integer shard, ClientId clientId) {
        return dslContextProvider.ppc(shard).
                select(RETARGETING_CONDITIONS.RET_COND_ID,
                        RETARGETING_CONDITIONS.CONDITION_NAME, RETARGETING_CONDITIONS.CONDITION_JSON).
                from(RETARGETING_CONDITIONS).
                where(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong())).
                and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.isNull()
                        .or(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE
                                .in(RET_COND_TYPES_FOR_DEFAULT_OPERATIONS))).
                and(RETARGETING_CONDITIONS.IS_DELETED.eq(FALSE)).
                and(RETARGETING_CONDITIONS.PROPERTIES.notContains(PROPERTY_AUTO_RETARGETING)).
                fetch(r -> new RetargetingConditionValidationData(r.value1(), r.value2(), r.value3()));
    }

    /**
     * Получение числа созданных условий ретаргетинга у клиента
     * с типами 'metrika_goals', 'dmp' и 'shortcuts', исключая удаленные и авторетаргетинги
     */
    public int getExistingRetargetingConditionsCount(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .selectCount()
                .from(RETARGETING_CONDITIONS)
                .where(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.isNull()
                        .or(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE
                                .in(RET_COND_TYPES_INCLUDED_IN_MAX_AMOUNT)))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(FALSE))
                .and(RETARGETING_CONDITIONS.PROPERTIES.notContains(PROPERTY_AUTO_RETARGETING))
                .fetchOne()
                .value1();
    }

    public List<Long> delete(int shard, @NotNull ClientId clientId, List<Long> idsToDelete) {
        if (clientId == null) {
            return Collections.emptyList();
        }
        Condition condition = RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong());
        var dslContext = dslContextProvider.ppc(shard);
        return deleteInner(dslContext, condition, idsToDelete);
    }

    public List<Long> deleteHyperGeo(DSLContext dslContext, @Nullable ClientId clientId, List<Long> idsToDelete) {
        Condition condition = RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(geo_segments);
        if (clientId != null) {
            condition = condition.and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()));
        }
        return deleteInner(dslContext, condition, idsToDelete);
    }

    private List<Long> deleteInner(DSLContext dslContext, Condition condition, List<Long> idsToDelete) {
        if (idsToDelete.isEmpty()) {
            logger.debug("Skip deletion as there are no ids given");
            return Collections.emptyList();
        }

        logger.trace("Try to delete retargeting conditions with next ids: {}", idsToDelete);
        deleteRetConditionsFromRetargetingConditionsTable(dslContext, condition, idsToDelete);
        retGoalsRepository.deleteByRetConditionIds(dslContext, idsToDelete);

        return idsToDelete;
    }

    /**
     * Return IDs of RetargetingLists used by some other objects
     *
     * @param shard shard
     * @param ids   {@link Collection} of IDs to check for usage
     * @return {@link Collection} of IDs which are in use
     */
    public Collection<Long> getUsedRetargetingConditionsIds(int shard, Collection<Long> ids) {
        DSLContext context = dslContextProvider.ppc(shard);

        SelectConditionStep<Record1<Long>> bidsRetargetings = context
                .select(BIDS_RETARGETING.RET_COND_ID)
                .from(BIDS_RETARGETING)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_RETARGETING.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(BIDS_RETARGETING.RET_COND_ID.in(ids))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));
        SelectConditionStep<Record1<Long>> mediaplanBidsRetargetings = context
                .select(MEDIAPLAN_BIDS_RETARGETING.RET_COND_ID)
                .from(MEDIAPLAN_BIDS_RETARGETING)
                .where(MEDIAPLAN_BIDS_RETARGETING.RET_COND_ID.in(ids));
        SelectConditionStep<Record1<Long>> retargetingMultiplierValues = context
                .select(RETARGETING_MULTIPLIER_VALUES.RET_COND_ID)
                .from(RETARGETING_MULTIPLIER_VALUES)
                .where(RETARGETING_MULTIPLIER_VALUES.RET_COND_ID.in(ids));

        Result<Record1<Long>> result =
                bidsRetargetings
                        .union(mediaplanBidsRetargetings)
                        .union(retargetingMultiplierValues)
                        .fetch();
        return result.stream()
                .map(record -> record.getValue(record.field1()))
                .collect(Collectors.toList());
    }

    /**
     * Получение списка существующих условий ретаргетинга,
     * кроме условий с типами, которые пользователям не показываем (например brandsafety)
     *
     * @param shard                   шард для запроса
     * @param clientId                id клиента
     * @param retargetingConditionIds список ид условий
     * @return список существующих ид условий
     */
    public Set<Long> getExistingRetargetingConditionIdsExceptIgnoredTypes(int shard, ClientId clientId,
                                                                          Collection<Long> retargetingConditionIds) {
        return dslContextProvider.ppc(shard)
                .select(RETARGETING_CONDITIONS.RET_COND_ID)
                .from(RETARGETING_CONDITIONS)
                .where(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .and(RETARGETING_CONDITIONS.RET_COND_ID.in(retargetingConditionIds))
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.notIn(IGNORED_RET_COND_TYPES))
                .fetchSet(RETARGETING_CONDITIONS.RET_COND_ID);
    }

    /**
     * Метод получения условий ретаргетинга по их id
     */
    public List<RetargetingCondition> getConditions(int shard, List<Long> retCondIds) {
        return getConditions(dslContextProvider.ppc(shard), retCondIds);
    }

    /**
     * Метод получения условий ретаргетинга по их id
     */
    public List<RetargetingCondition> getConditions(DSLContext dslContext, List<Long> retCondIds) {
        return StreamEx.of(dslContext
                .select(retargetingConditionMapper.getFieldsToRead())
                .from(RETARGETING_CONDITIONS)
                .where(RETARGETING_CONDITIONS.RET_COND_ID.in(retCondIds))
                .fetch())
                .map(retargetingConditionMapper::fromDb)
                .toList();
    }

    /**
     * @param dslContext transaction context
     * @param condition  condition
     * @param ids        {@link List} of RetargetingCondition IDs to be deleted
     * @throws DataAccessException      if some error occurred on SQL execution
     * @throws IllegalArgumentException if {@code ids} is {@code null}
     */
    private void deleteRetConditionsFromRetargetingConditionsTable(DSLContext dslContext, Condition condition,
                                                                   List<Long> ids)
            throws DataAccessException, IllegalArgumentException {
        condition = condition.and(RETARGETING_CONDITIONS.RET_COND_ID.in(ids));

        int updatedRowsCount =
                dslContext.update(RETARGETING_CONDITIONS)
                        .set(RETARGETING_CONDITIONS.IS_DELETED, TRUE)
                        .set(RETARGETING_CONDITIONS.MODTIME, now())
                        .where(condition)
                        .execute();

        if (updatedRowsCount != ids.size()) {
            logger.warn(
                    "Number of deleted retargeting conditions ({}) differs from size of given collection ({}). IDs: {}",
                    updatedRowsCount, ids.size(), ids
            );
        }
    }

    protected LocalDateTime now() {
        return LocalDateTime.now();
    }

    private List<RetargetingCondition> getRetConditionsFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                                      Collection<Long> ids) {
        return getRetConditionsFromRetargetingConditionsTable(shard, clientId, ids, null, null, null,
                limited(ids.size(), (int) DEFAULT_OFFSET));
    }

    private List<RetargetingCondition> getRetConditionsFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                                      Collection<Long> ids,
                                                                                      Collection<Long> adGroupIds,
                                                                                      String filter,
                                                                                      Collection<RetargetingConditionsRetargetingConditionsType> types, LimitOffset limitOffset) {
        Set<RetargetingConditionsRetargetingConditionsType> filteredRetCondTypes = filterIgnoredRetCondTypes(types);
        List<RetargetingCondition> retConditions = get(shard, clientId, ids, adGroupIds,
                filter, filteredRetCondTypes, limitOffset);
        return filterList(retConditions, r -> !r.getInterest());
    }

    private Set<RetargetingConditionsRetargetingConditionsType> filterIgnoredRetCondTypes(
            Collection<RetargetingConditionsRetargetingConditionsType> types) {
        Collection<RetargetingConditionsRetargetingConditionsType> availableTypes =
                types == null ? Set.of(RetargetingConditionsRetargetingConditionsType.values()) : types;
        return availableTypes.stream().filter(type -> !IGNORED_RET_COND_TYPES.contains(type)).collect(toSet());
    }

    private List<RetargetingCondition> getInterestFromRetargetingConditionsTable(int shard, ClientId clientId,
                                                                                 Collection<Long> ids,
                                                                                 LimitOffset limitOffset) {
        List<RetargetingCondition> retConditions =
                get(shard, clientId, ids, null,
                        null, null, limitOffset);
        return filterList(retConditions, RetargetingCondition::getInterest);
    }

    public Map<Long, RetargetingCondition> getRetConditionsByCampaignIds(int shard, List<Long> campaignIds,
                                                                         ConditionType conditionType) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Field<?>> fieldsToRead = new ArrayList<>(singletonList(CAMPAIGNS.CID));
        fieldsToRead.addAll(retargetingConditionMapper.getFieldsToRead());

        SelectConditionStep<Record> selectConditionStep = dslContext
                .select(fieldsToRead)
                .from(RETARGETING_CONDITIONS)
                .join(CAMPAIGNS).on(RETARGETING_CONDITIONS.RET_COND_ID.eq(CAMPAIGNS.AB_SEGMENT_RET_COND_ID))
                .where(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(conditionTypeToDb(conditionType)))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .and(CAMPAIGNS.CID.in(campaignIds));
        return selectConditionStep
                .orderBy(RETARGETING_CONDITIONS.RET_COND_ID)
                .fetchMap(CAMPAIGNS.CID, retargetingConditionMapper::fromDb);
    }

    public Map<Long, RetargetingCondition> getStatisticRetConditionsByCampaignIds(int shard, List<Long> campaignIds,
                                                                                  ConditionType conditionType) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Field<?>> fieldsToRead = new ArrayList<>(singletonList(CAMPAIGNS.CID));
        fieldsToRead.addAll(retargetingConditionMapper.getFieldsToRead());

        SelectConditionStep<Record> selectConditionStep = dslContext
                .select(fieldsToRead)
                .from(RETARGETING_CONDITIONS)
                .join(CAMPAIGNS).on(RETARGETING_CONDITIONS.RET_COND_ID.eq(CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID))
                .where(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(conditionTypeToDb(conditionType)))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .and(CAMPAIGNS.CID.in(campaignIds));
        return selectConditionStep
                .orderBy(RETARGETING_CONDITIONS.RET_COND_ID)
                .fetchMap(CAMPAIGNS.CID, retargetingConditionMapper::fromDb);
    }

    public Map<Long, RetargetingCondition> getBrandSafetyRetConditionsByCampaignIds(int shard, List<Long> campaignIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Field<?>> fieldsToRead = new ArrayList<>(singletonList(CAMPAIGNS.CID));
        fieldsToRead.addAll(retargetingConditionMapper.getFieldsToRead());

        SelectConditionStep<Record> selectConditionStep = dslContext
                .select(fieldsToRead)
                .from(RETARGETING_CONDITIONS)
                .join(CAMPAIGNS).on(RETARGETING_CONDITIONS.RET_COND_ID.eq(CAMPAIGNS.BRANDSAFETY_RET_COND_ID))
                .where(CAMPAIGNS.CID.in(campaignIds));
        return selectConditionStep
                .fetchMap(CAMPAIGNS.CID, retargetingConditionMapper::fromDb);
    }

    public Map<Long, String> getRetargetingConditionNameByIds(int shard, Long campaignId, ConditionType conditionType) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_NAME)
                .from(RETARGETING_CONDITIONS)
                .join(BIDS_RETARGETING).on(BIDS_RETARGETING.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .where(BIDS_RETARGETING.CID.eq(campaignId))
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(conditionTypeToDb(conditionType)))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .fetchMap(RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_NAME);
    }

    public List<Long> getCampaignIdsByBrandSafetyRetConditions(int shard, List<Long> brandsafetyRetCondIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.BRANDSAFETY_RET_COND_ID.in(brandsafetyRetCondIds))
                .fetch(CAMPAIGNS.CID);
    }

    public List<RetargetingCondition> getBrandSafetyRetConditionsByClient(int shard, ClientId clientId) {
        return StreamEx.of(dslContextProvider.ppc(shard)
                .select(retargetingConditionMapper.getFieldsToRead())
                .from(RETARGETING_CONDITIONS)
                .where(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(conditionTypeToDb(ConditionType.brandsafety)))
                .fetch())
                .map(retargetingConditionMapper::fromDb)
                .toList();
    }

    private Map<Long, List<RetargetingCondition>> getRetConditionsByAdGroupIds(int shard, @Nullable Long clientId,
                                                                               Collection<Long> adGroupIds) {
        if (adGroupIds == null || adGroupIds.isEmpty()) {
            return Collections.emptyMap();
        }

        DSLContext dslContext = dslContextProvider.ppc(shard);
        List<Field<?>> fieldsToRead = new ArrayList<>(singletonList(BIDS_RETARGETING.PID));
        fieldsToRead.addAll(retargetingConditionMapper.getFieldsToRead());

        SelectConditionStep<Record> selectConditionStep = dslContext
                .select(fieldsToRead)
                .from(RETARGETING_CONDITIONS)
                .join(BIDS_RETARGETING).on(RETARGETING_CONDITIONS.RET_COND_ID.eq(BIDS_RETARGETING.RET_COND_ID))
                .where(BIDS_RETARGETING.PID.in(adGroupIds))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE));

        if (clientId != null) {
            selectConditionStep = selectConditionStep.and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId));
        }

        return selectConditionStep
                .orderBy(RETARGETING_CONDITIONS.RET_COND_ID)
                .limit(maxLimited().limit())
                .offset(maxLimited().offset())
                .fetchGroups(BIDS_RETARGETING.PID, retargetingConditionMapper::fromDb);
    }

    /**
     * Получения условий ретаргетинга, связанных с группами данного клиента.
     *
     * @param shard      номер шарда
     * @param clientId   идентификатор клиента
     * @param adGroupIds список идентификаторов групп
     */
    public Map<Long, List<RetargetingCondition>> getRetConditionsByAdGroupIds(int shard, ClientId clientId,
                                                                              Collection<Long> adGroupIds) {
        return getRetConditionsByAdGroupIds(shard, clientId.asLong(), adGroupIds);
    }

    /**
     * Получения условий ретаргетинга, связанных с данными группами.
     *
     * @param shard      номер шарда
     * @param adGroupIds список идентификаторов групп
     */
    public Map<Long, List<RetargetingCondition>> getRetConditionsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return getRetConditionsByAdGroupIds(shard, (Long) null, adGroupIds);
    }

    /**
     * Получить список условий нацеливания
     *
     * @param shard       шард
     * @param clientId    идентификатор клиента
     * @param ids         идентификаторы условий нацеливания
     * @param adGroupIds  идентификаторы групп объявлений
     * @param filter      фильтр по наименованию
     * @param types       тип условий нацеливания
     * @param limitOffset ограничение количества записей и смещение от начала выборки
     * @return {@link List<RetargetingCondition>} список условий нацеливания
     */
    @Nonnull
    List<RetargetingCondition> get(int shard, @Nonnull ClientId clientId, @Nullable Collection<Long> ids,
                                   @Nullable Collection<Long> adGroupIds,
                                   @Nullable String filter,
                                   @Nullable Collection<RetargetingConditionsRetargetingConditionsType> types,
                                   @Nonnull LimitOffset limitOffset) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        SelectJoinStep<Record> selectJoinStep = dslContext
                .select(retargetingConditionMapper.getFieldsToRead())
                .from(RETARGETING_CONDITIONS);

        SelectConditionStep<Record> selectConditionStep = selectJoinStep
                .where(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .and(RETARGETING_CONDITIONS.PROPERTIES.notContains(PROPERTY_AUTO_RETARGETING));

        if (adGroupIds != null) {
            selectConditionStep
                    .and(RETARGETING_CONDITIONS.RET_COND_ID.in(
                            dslContext.select(BIDS_RETARGETING.RET_COND_ID)
                                    .from(BIDS_RETARGETING)
                                    .where(BIDS_RETARGETING.PID.in(adGroupIds)))
                    );
        }

        if (ids != null) {
            selectConditionStep = selectConditionStep.and(RETARGETING_CONDITIONS.RET_COND_ID.in(ids));
        }

        if (filter != null) {
            selectConditionStep = selectConditionStep.and(RETARGETING_CONDITIONS.CONDITION_NAME.contains(filter));
        }

        if (types != null) {
            selectConditionStep = selectConditionStep.and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.in(types));
        }

        Result<Record> result = selectConditionStep
                .orderBy(RETARGETING_CONDITIONS.RET_COND_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();

        return result.stream().map(retargetingConditionMapper::fromDb).collect(toList());
    }


    private void populateRetConditionsFromRetargetingGoalsTable(int shard, List<RetargetingCondition> retConditions) {
        Set<Long> retConditionsIds = retConditions
                .stream()
                .map(RetargetingCondition::getId)
                .collect(toSet());
        Set<Long> accessible = retGoalsRepository.getAccessibleRetConditionIds(shard, retConditionsIds);
        for (RetargetingCondition retCondition : retConditions) {
            retCondition.setAvailable(accessible.contains(retCondition.getId()));
        }
    }

    private void generateIdsForRetargetingConditions(Collection<RetargetingCondition> retConditions) {
        List<Long> clientIds = mapList(retConditions, RetargetingConditionBase::getClientId);
        Iterator<Long> ids = shardHelper.generateRetargetingConditionIds(clientIds).iterator();
        retConditions.forEach(retCond -> retCond.setId(ids.next()));
    }

    private void addRetConditionsToRetargetingConditionsTable(int shard,
                                                              Collection<RetargetingCondition> retConditions) {
        if (retConditions.isEmpty()) {
            return;
        }

        InsertHelper<RetargetingConditionsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), RETARGETING_CONDITIONS);

        for (RetargetingCondition retargetingCondition : retConditions) {
            insertHelper.add(retargetingConditionMapper, retargetingCondition).newRecord();
        }

        int rowsInserted = insertHelper.execute();
        logger.debug("Rows inserted: {}", rowsInserted);
    }

    private void addRetConditionsToRetargetingGoalsTable(int shard, Collection<RetargetingCondition> retConditions) {
        Multimap<Long, RetargetingConditionGoal> goals = HashMultimap.create();
        for (RetargetingCondition retCondition : retConditions) {
            for (RetargetingConditionGoal goal : retCondition.collectGoals()) {
                goals.put(retCondition.getId(), goal);
            }
        }

        retGoalsRepository.add(shard, goals);
    }

    /**
     * Обновление условий нацеливания
     *
     * @param shard          - шард
     * @param appliedChanges - набор {@link AppliedChanges}, описывающих, какие изменения были применены
     */
    public void update(int shard, Collection<AppliedChanges<RetargetingCondition>> appliedChanges) {
        JooqUpdateBuilder<RetargetingConditionsRecord, RetargetingCondition> ub =
                new JooqUpdateBuilder<>(RETARGETING_CONDITIONS.RET_COND_ID, appliedChanges);
        ub.processProperty(RetargetingCondition.NAME, RETARGETING_CONDITIONS.CONDITION_NAME);
        ub.processProperty(RetargetingCondition.DESCRIPTION, RETARGETING_CONDITIONS.CONDITION_DESC);
        ub.processProperty(RetargetingCondition.LAST_CHANGE_TIME, RETARGETING_CONDITIONS.MODTIME, Function.identity());

        ub.processProperty(RetargetingCondition.RULES, RETARGETING_CONDITIONS.CONDITION_JSON,
                RetargetingConditionMappings::rulesToJson);
        ub.processProperty(RetargetingCondition.RULES, RETARGETING_CONDITIONS.PROPERTIES,
                rules -> calcNegativeByRules(rules) ? PROPERTY_NEGATIVE : "");

        dslContextProvider.ppc(shard)
                .update(RETARGETING_CONDITIONS)
                .set(ub.getValues())
                .where(RETARGETING_CONDITIONS.RET_COND_ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Получить список групп объявлений, которые каким-либо образом используют указанные условия нацеливания
     *
     * @param shard      - шард
     * @param retCondIds - список id условий нацеливания
     * @return - словарь: id условия нацеливания -> список связанных id групп
     */
    public Map<Long, List<Long>> getAdGroupIds(int shard, Collection<Long> retCondIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_COND_ID, BIDS_RETARGETING.PID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_COND_ID.in(retCondIds))
                .and(BIDS_RETARGETING.IS_SUSPENDED.eq(0L))
                .fetchGroups(BIDS_RETARGETING.RET_COND_ID, BIDS_RETARGETING.PID);
    }

    /**
     * Получить список кампаний, которые каким-либо образом используют указанные условия нацеливания
     *
     * @param shard      - шард
     * @param retCondIds - список id условий нацеливания
     * @return - словарь: id условия нацеливания -> список связанных id кампаний
     */
    public Map<Long, List<Long>> getCampaignIds(int shard, Collection<Long> retCondIds) {
        return getRetConditionIdWithCampaignId(shard, retCondIds)
                .fetchGroups(BIDS_RETARGETING.RET_COND_ID, BIDS_RETARGETING.CID);
    }

    /**
     * Получить словарь кампаний, которые каким-либо образом используют указанные условия нацеливания
     *
     * @param shard      - шард
     * @param retCondIds - список id условий нацеливания
     * @return - словарь: id кампании -> список связанных id условий нацеливания
     */
    public Map<Long, List<Long>> getRetConditionIdsByCampaignId(int shard, Collection<Long> retCondIds) {
        return getRetConditionIdWithCampaignId(shard, retCondIds)
                .fetchGroups(BIDS_RETARGETING.CID, BIDS_RETARGETING.RET_COND_ID);
    }

    private SelectConditionStep<Record2<Long, Long>> getRetConditionIdWithCampaignId(int shard,
                                                                                     Collection<Long> retCondIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_COND_ID, BIDS_RETARGETING.CID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_COND_ID.in(retCondIds))
                .and(BIDS_RETARGETING.IS_SUSPENDED.eq(0L));
    }

}
