package ru.yandex.direct.core.entity.retargeting.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx;
import ru.yandex.direct.core.entity.retargeting.model.ConversionLevel;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.core.entity.retargeting.model.RawMetrikaSegmentPreset;
import ru.yandex.direct.dbschema.ppcdict.enums.MetrikaGoalsConversionLevel;
import ru.yandex.direct.dbschema.ppcdict.tables.records.MetrikaGoalsRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.MetrikaSegmentsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.util.RepositoryUtils.intToLong;
import static ru.yandex.direct.dbschema.ppcdict.tables.MetrikaGoals.METRIKA_GOALS;
import static ru.yandex.direct.dbschema.ppcdict.tables.MetrikaSegmentPresets.METRIKA_SEGMENT_PRESETS;
import static ru.yandex.direct.dbschema.ppcdict.tables.MetrikaSegments.METRIKA_SEGMENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.PassportUtils.normalizeLogin;

@Repository
public class RetargetingGoalsPpcDictRepository {

    private final DslContextProvider dslContextProvider;

    private static final Logger logger = LoggerFactory.getLogger(RetargetingGoalsPpcDictRepository.class);

    private JooqMapperWithSupplier<Goal> retargetingConditionGoalMapper;
    private JooqMapperWithSupplier<Goal> retargetingConditionGoalMapperForInsert;
    private JooqReaderWithSupplier<RawMetrikaSegmentPreset> metrikaSegmentPresetReader;

    @Autowired
    public RetargetingGoalsPpcDictRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        retargetingConditionGoalMapper = createGoalMapper(true);
        retargetingConditionGoalMapperForInsert = createGoalMapper(false);

        metrikaSegmentPresetReader = JooqReaderWithSupplierBuilder.builder(RawMetrikaSegmentPreset::new)
                .readProperty(RawMetrikaSegmentPreset.PRESET_ID,
                        ReaderBuildersEx.fromLongFieldToInteger(METRIKA_SEGMENT_PRESETS.PRESET_ID))
                .readProperty(RawMetrikaSegmentPreset.NAME, fromField(METRIKA_SEGMENT_PRESETS.NAME))
                .readProperty(RawMetrikaSegmentPreset.EXPRESSION, fromField(METRIKA_SEGMENT_PRESETS.EXPRESSION))
                .readProperty(RawMetrikaSegmentPreset.TANKER_NAME_KEY,
                        fromField(METRIKA_SEGMENT_PRESETS.TANKER_NAME_KEY))
                .build();
    }

    private JooqMapperWithSupplier<Goal> createGoalMapper(boolean withGoalStatus) {
        JooqMapperWithSupplierBuilder<Goal> builder = JooqMapperWithSupplierBuilder.builder(Goal::new)
                .map(property(Goal.ID, METRIKA_GOALS.GOAL_ID))
                .map(property(Goal.PARENT_ID, METRIKA_GOALS.PARENT_GOAL_ID))
                .map(property(Goal.NAME, METRIKA_GOALS.NAME))
                .map(convertibleProperty(Goal.METRIKA_COUNTER_GOAL_TYPE, METRIKA_GOALS.GOAL_TYPE,
                        MetrikaCounterGoalType::fromSource, MetrikaCounterGoalType::toSource))
                .map(convertibleProperty(Goal.CONVERSION_LEVEL, METRIKA_GOALS.CONVERSION_LEVEL,
                        ConversionLevel::fromSource, this::conversionLevelToDb));
        if (withGoalStatus) {
            builder.map(convertibleProperty(Goal.STATUS, METRIKA_GOALS.GOAL_STATUS,
                    GoalStatus::fromSource, GoalStatus::toSource));
        }
        return builder.build();
    }

    private MetrikaGoalsConversionLevel conversionLevelToDb(ConversionLevel value) {
        return value == null ? MetrikaGoalsConversionLevel.unknown : ConversionLevel.toSource(value);
    }

    /**
     * Получение данных целей по их ids из PPCDICT без подцелей
     *
     * @param goalIds {@link Collection} id целей
     * @return {@link Set} целей.
     */
    public List<Goal> getMetrikaGoalsFromPpcDict(Collection<Long> goalIds) {
        return getMetrikaGoalsFromPpcDict(goalIds, false);
    }

    /**
     * Получение данных целей по их ids из PPCDICT с подцелями или без
     *
     * @param goalIds      {@link Collection} id целей
     * @param withSubgoals boolean признак для выборки подцелей
     * @return {@link Set} целей.
     */
    public List<Goal> getMetrikaGoalsFromPpcDict(Collection<Long> goalIds, boolean withSubgoals) {
        if (goalIds.isEmpty()) {
            return emptyList();
        }

        Condition whereCondition = METRIKA_GOALS.GOAL_ID.in(goalIds);

        if (withSubgoals) {
            // отбрасываем '0' из списка потенциальных parentGoalId
            List<Long> parentGoalIds = filterList(goalIds, id -> id != null && id != 0L);
            whereCondition = whereCondition.or(METRIKA_GOALS.PARENT_GOAL_ID.in(parentGoalIds));
        }

        Result<Record> result = dslContextProvider.ppcdict()
                .select(retargetingConditionGoalMapper.getFieldsToRead())
                .from(METRIKA_GOALS).where(whereCondition)
                .orderBy(METRIKA_GOALS.GOAL_ID)
                .fetch();
        return result.stream().map(retargetingConditionGoalMapper::fromDb).collect(toList());
    }

    public void addMetrikaGoalsToPpcDict(Set<Goal> goals) {
        if (goals.isEmpty()) {
            return;
        }

        InsertHelper<MetrikaGoalsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), METRIKA_GOALS);
        for (Goal goal : goals) {
            insertHelper.add(retargetingConditionGoalMapperForInsert, goal).newRecord();
        }
        int rowsInserted = insertHelper.execute();
        logger.debug("Rows inserted: {}", rowsInserted);
    }

    public void updateMetrikaGoals(List<AppliedChanges<Goal>> appliedChanges) {
        UpdateHelper<MetrikaGoalsRecord> updateHelper = new UpdateHelper<>(dslContextProvider.ppcdict(),
                METRIKA_GOALS.GOAL_ID);
        updateHelper.processUpdateAll(retargetingConditionGoalMapper, appliedChanges)
                .execute();
    }

    public void deleteMetrikaGoalsFromPpcDict(Collection<Long> ids) {
        DSLContext dslContext = dslContextProvider.ppcdict();
        int rowsDeleted = dslContext.deleteFrom(METRIKA_GOALS)
                .where(METRIKA_GOALS.GOAL_ID.in(ids)).execute();
        logger.debug("Rows delete: {}", rowsDeleted);
    }

    /**
     * Возвращает пресеты сегментов из базы ppcdict
     */
    public List<RawMetrikaSegmentPreset> getSegmentPresets() {
        return dslContextProvider.ppcdict()
                .select(metrikaSegmentPresetReader.getFieldsToRead())
                .from(METRIKA_SEGMENT_PRESETS)
                .orderBy(METRIKA_SEGMENT_PRESETS.PRESET_ID)
                .fetch(metrikaSegmentPresetReader::fromDb);
    }

    public void addMetrikaSegmentsCreatedByPresets(Map<Integer, String> idToOwnerLogin) {
        InsertHelper<MetrikaSegmentsRecord> helper = new InsertHelper<>(dslContextProvider.ppcdict(), METRIKA_SEGMENTS);
        idToOwnerLogin.forEach((id, login) -> helper
                .set(METRIKA_SEGMENTS.SEGMENT_ID, intToLong(id))
                .set(METRIKA_SEGMENTS.OWNER_LOGIN, normalizeLogin(login))
                .newRecord());
        var rowsInserted = helper.onDuplicateKeyIgnore().executeIfRecordsAdded();

        logger.debug("Rows inserted: {}", rowsInserted);
    }

}
