package ru.yandex.direct.core.entity.retargeting.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Multimap;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStep5;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.types.UInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.IdModFilter;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionGoal;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingGoalsGoalSource;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingGoalsGoalType;
import ru.yandex.direct.dbschema.ppc.tables.records.RetargetingGoalsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingGoals.RETARGETING_GOALS;

@Repository
public class RetargetingGoalsRepository {

    private static final Logger logger = LoggerFactory.getLogger(RetargetingGoalsRepository.class);

    private final DslContextProvider dslContextProvider;


    @Autowired
    public RetargetingGoalsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Условие ретаргетинга доступно (isAvailable), если все его цели is_accessible.
     *
     * @param shard           шард
     * @param retConditionIds список id условий ретаргетинга, которые требуется проверить.
     * @return подмножество доступных условий ретаргетинга из переданного списка.
     */
    public Set<Long> getAccessibleRetConditionIds(Integer shard, Collection<Long> retConditionIds) {

        Set<UInteger> retConditionsIdsUint = retConditionIds.stream().
                map(UInteger::valueOf).
                collect(toSet());

        Result<Record1<Long>> inaccessibleRecords = dslContextProvider.ppc(shard).
                select(RETARGETING_GOALS.RET_COND_ID).
                from(RETARGETING_GOALS).
                where(RETARGETING_GOALS.RET_COND_ID.in(retConditionsIdsUint)).
                and(RETARGETING_GOALS.IS_ACCESSIBLE.eq(0L)).
                fetch();

        Set<Long> retCondIdsWithInaccessibleGoals = inaccessibleRecords
                .stream()
                .map(Record1::value1)
                .collect(toSet());

        return retConditionIds
                .stream()
                .filter(id -> !retCondIdsWithInaccessibleGoals.contains(id))
                .collect(toSet());
    }

    public void add(int shard, Multimap<Long, RetargetingConditionGoal> goalsByRetCondId) {
        if (goalsByRetCondId.isEmpty()) {
            return;
        }

        DSLContext context = dslContextProvider.ppc(shard);
        InsertValuesStep5<RetargetingGoalsRecord, Long, Long, LocalDateTime, RetargetingGoalsGoalType,
                RetargetingGoalsGoalSource> retGoalInsert = context.insertInto(
                        RETARGETING_GOALS,
                        RETARGETING_GOALS.RET_COND_ID,
                        RETARGETING_GOALS.GOAL_ID,
                        RETARGETING_GOALS.MODTIME,
                        RETARGETING_GOALS.GOAL_TYPE,
                        RETARGETING_GOALS.GOAL_SOURCE);

        final LocalDateTime timeNow = now();
        for (Long retCondId : goalsByRetCondId.keySet()) {
            for (RetargetingConditionGoal goal : goalsByRetCondId.get(retCondId)) {
                retGoalInsert.values(
                        retCondId,
                        goal.getId(),
                        timeNow,
                        RetargetingGoalsGoalType.valueOf(goal.getType().name().toLowerCase()),
                        RetargetingGoalsGoalSource.valueOf(goal.getType().getSource().name().toLowerCase()));
            }
        }

        retGoalInsert
                .onDuplicateKeyIgnore()
                .execute();
    }

    void deleteByRetConditionIds(DSLContext dslContext, Collection<Long> conditionIds) {
        int rowsDeleted = dslContext.deleteFrom(RETARGETING_GOALS)
                .where(RETARGETING_GOALS.RET_COND_ID.in(conditionIds))
                .execute();
        logger.trace("Amount of RetargetingGoals deleted: {}. RetargetingConditions ids: {}",
                rowsDeleted, conditionIds);
    }


    /**
     * Удаление целей
     *
     * @param shard             - шард
     * @param retCondIdToGoalId - мульти-словарь: id условия нацеливания -> набор целей
     * @return - количество удалённых строк
     */
    public int delete(int shard, Multimap<Long, Long> retCondIdToGoalId) {
        Condition condition = retCondIdToGoalId.entries().stream()
                .map(e -> RETARGETING_GOALS.RET_COND_ID.eq(e.getKey()).and(RETARGETING_GOALS.GOAL_ID.eq(e.getValue())))
                .reduce(Condition::or)
                .orElse(null);

        if (condition == null) {
            return 0;
        }

        return dslContextProvider.ppc(shard).deleteFrom(RETARGETING_GOALS)
                .where(condition)
                .execute();
    }

    public List<Long> getGoalIds(int shard, long conditionId) {
        Result<Record1<Long>> result = dslContextProvider.ppc(shard)
                .select(RETARGETING_GOALS.GOAL_ID)
                .from(RETARGETING_GOALS)
                .where(RETARGETING_GOALS.RET_COND_ID.eq(conditionId))
                .fetch();
        return result.stream().map(Record1::value1).collect(toList());
    }

    /**
     * Получение доступных (is_accessible=1) целей по шарду с фильтрацией
     *
     * @param shard  номер шарда
     * @param filter состоит из 2х полей: divisor и remainder
     *               цели фильтруются следующим образом:
     *               where goal_id mod divisor = remainder
     *               в случае, если filter = null, фильтрация по id не применится
     * @return коллекция goal_id
     */
    public List<Long> getAccessibleGoalIds(Integer shard, IdModFilter filter) {
        SelectConditionStep<Record1<Long>> selectConditionStep = dslContextProvider.ppc(shard)
                .selectDistinct(RETARGETING_GOALS.GOAL_ID)
                .from(RETARGETING_GOALS)
                .where(RETARGETING_GOALS.IS_ACCESSIBLE.eq(RepositoryUtils.TRUE));

        if (filter != null) {
            selectConditionStep = selectConditionStep
                    .and(RETARGETING_GOALS.GOAL_ID.mod(filter.getDivisor()).eq(filter.getRemainder()));
        }

        return selectConditionStep.fetch(RETARGETING_GOALS.GOAL_ID);
    }

    protected LocalDateTime now() {
        return LocalDateTime.now();
    }

}
