package ru.yandex.direct.core.entity.retargeting.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCampaignInfo;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingSimple;
import ru.yandex.direct.core.entity.retargeting.model.StatusAggregationRetargeting;
import ru.yandex.direct.dbschema.ppc.enums.BidsRetargetingStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsRetargetingRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting.BIDS_RETARGETING;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class RetargetingRepository {
    private static final Logger logger = LoggerFactory.getLogger(RetargetingRepository.class);

    private final DslContextProvider ppcDslContextProvider;
    private final JooqMapperWithSupplier<Retargeting> retargetingMapper;
    private final JooqReaderWithSupplier<RetargetingCampaignInfo> retargetingCampaignInfoReader;
    private final ShardHelper shardHelper;

    private final Collection<Field<?>> statusAggregationFieldsToRead;

    @Autowired
    public RetargetingRepository(DslContextProvider ppcDslContextProvider, ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
        this.retargetingMapper = JooqMapperWithSupplierBuilder.builder(Retargeting::new)
                .map(property(Retargeting.ID, BIDS_RETARGETING.RET_ID))
                .map(property(Retargeting.RETARGETING_CONDITION_ID, BIDS_RETARGETING.RET_COND_ID))
                .map(property(Retargeting.AD_GROUP_ID, BIDS_RETARGETING.PID))
                .map(convertibleProperty(Retargeting.PRICE_CONTEXT, BIDS_RETARGETING.PRICE_CONTEXT,
                        RetargetingMappings::priceFromDbFormat,
                        RetargetingMappings::priceToDbFormat))
                .map(integerProperty(Retargeting.AUTOBUDGET_PRIORITY, BIDS_RETARGETING.AUTOBUDGET_PRIORITY))
                .map(convertibleProperty(Retargeting.STATUS_BS_SYNCED, BIDS_RETARGETING.STATUS_BS_SYNCED,
                        RetargetingMappings::statusBsSyncedFromDb,
                        RetargetingMappings::statusBsSyncedToDb))
                .map(booleanProperty(Retargeting.IS_SUSPENDED, BIDS_RETARGETING.IS_SUSPENDED))
                .map(property(Retargeting.LAST_CHANGE_TIME, BIDS_RETARGETING.MODTIME))
                .map(property(Retargeting.CAMPAIGN_ID, BIDS_RETARGETING.CID))
                .build();

        this.retargetingCampaignInfoReader = JooqReaderWithSupplierBuilder.builder(RetargetingCampaignInfo::new)
                .readProperty(RetargetingCampaignInfo.RETARGETING_ID, fromField(BIDS_RETARGETING.RET_ID))
                .readProperty(RetargetingCampaignInfo.RETARGETING_CONDITION_ID, fromField(BIDS_RETARGETING.RET_COND_ID))
                .readProperty(RetargetingCampaignInfo.AD_GROUP_ID, fromField(BIDS_RETARGETING.PID))
                .readProperty(RetargetingCampaignInfo.CAMPAIGN_ID, fromField(BIDS_RETARGETING.CID))
                .readProperty(RetargetingCampaignInfo.CAMPAIGN_IS_ARCHIVED,
                        fromField(CAMPAIGNS.ARCHIVED).by(CampaignMappings::archivedFromDb))
                .readProperty(RetargetingCampaignInfo.CAMPAIGN_TYPE,
                        fromField(CAMPAIGNS.TYPE).by(CampaignType::fromSource))
                .build();

        this.statusAggregationFieldsToRead =
                retargetingMapper.getFieldsToRead(StatusAggregationRetargeting.allModelProperties());
    }

    /**
     * возвращает информацию, необходимую в процессе удаления ретаргетинга:
     *
     * @param shard  номер шарда
     * @param retIds id требуемых ретаргетингов
     */
    public Map<Long, RetargetingCampaignInfo> getRetargetingToCampaignMappingForDelete(int shard,
                                                                                       Collection<Long> retIds) {
        if (retIds.isEmpty()) {
            return emptyMap();
        }

        return ppcDslContextProvider.ppc(shard)
                .select(retargetingCampaignInfoReader.getFieldsToRead())
                .from(BIDS_RETARGETING)
                .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                .where(BIDS_RETARGETING.RET_ID.in(retIds).and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .fetchMap(BIDS_RETARGETING.RET_ID, retargetingCampaignInfoReader::fromDb);
    }

    public List<Long> add(int shard, List<Retargeting> retargetings) {
        generateIdsForRetargetings(retargetings);
        addRetargetingsToBidsRetargetingTable(shard, retargetings);
        return mapList(retargetings, Retargeting::getId);
    }

    private void addRetargetingsToBidsRetargetingTable(int shard, List<Retargeting> retargetings) {
        if (retargetings.isEmpty()) {
            return;
        }
        InsertHelper<BidsRetargetingRecord> insertHelper =
                new InsertHelper<>(ppcDslContextProvider.ppc(shard), BIDS_RETARGETING);
        for (Retargeting retargeting : retargetings) {
            insertHelper.add(retargetingMapper, retargeting).newRecord();
        }
        int rowsInserted = insertHelper.execute();
        logger.debug("Rows inserted: {}", rowsInserted);
    }

    private void generateIdsForRetargetings(List<Retargeting> retargetings) {
        List<Long> ids = shardHelper.generateRetargetingIds(retargetings.size());
        StreamEx.of(retargetings).zipWith(ids.stream())
                .forKeyValue(Retargeting::setId);
    }

    /**
     * Получить все Retargeting'и по {@code adGroupIds}.
     * <p>
     * Используется только в валидации.
     * <p>
     * <b>Note!</b> Не осуществляет проверку, что retargeting'и видны оператору.
     *
     * @param shard      shard
     * @param adGroupIds список ID-шников AdGroup
     * @return {@link List}&lt;{@link Retargeting}&gt; для указанных {@code adGroupIds}
     */
    public List<Retargeting> getRetargetingsByAdGroups(int shard, Collection<Long> adGroupIds) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);
        Result<Record> result = dslContext
                .select(retargetingMapper.getFieldsToRead())
                .from(BIDS_RETARGETING)
                .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                .where(BIDS_RETARGETING.PID.in(adGroupIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .orderBy(BIDS_RETARGETING.RET_ID)
                .fetch();

        return result.stream().map(retargetingMapper::fromDb).collect(toList());
    }

    public List<Retargeting> getRetargetingsByRetCondIds(int shard, Collection<Long> retCondIds) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);
        Result<Record> result = dslContext
                .select(retargetingMapper.getFieldsToRead())
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_COND_ID.in(retCondIds))
                .fetch();
        return result.stream().map(retargetingMapper::fromDb).collect(toList());
    }

    public List<Retargeting> getRetargetingsByCampaigns(int shard, List<Long> campaignIds) {
        Result<Record> result = ppcDslContextProvider.ppc(shard)
                .select(retargetingMapper.getFieldsToRead())
                .from(BIDS_RETARGETING)
                .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .orderBy(BIDS_RETARGETING.RET_ID)
                .fetch();

        return result.stream().map(retargetingMapper::fromDb).collect(toList());
    }

    /**
     * Получение ретаргетингов групп определенного типа принадлежащих списку кампаний
     */
    public List<Retargeting> getRetargetingsByCampaignIdsAndAdGroupType(
            int shard,
            Collection<Long> campaignIds,
            Collection<AdGroupType> adGroupTypes) {
        return ppcDslContextProvider.ppc(shard)
                .select(retargetingMapper.getFieldsToRead())
                .from(BIDS_RETARGETING)
                .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                .join(PHRASES).on(BIDS_RETARGETING.PID.eq(PHRASES.PID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(PHRASES.ADGROUP_TYPE.in(mapList(adGroupTypes, AdGroupType::toSource)))
                .orderBy(BIDS_RETARGETING.RET_ID)
                .fetch(retargetingMapper::fromDb);
    }

    /**
     * Включает/выключает ретаргетинги в соответствии с параметрами, выставленными в запросе
     *
     * @param shard   шард
     * @param changes набор запросов на обновление ретаргетингов
     * @return количество обновленных сущностей
     */
    public int setSuspended(int shard, List<AppliedChanges<Retargeting>> changes) {
        JooqUpdateBuilder<BidsRetargetingRecord, Retargeting> ub =
                new JooqUpdateBuilder<>(BIDS_RETARGETING.RET_ID, changes);
        ub.processProperty(Retargeting.LAST_CHANGE_TIME, BIDS_RETARGETING.MODTIME, identity());
        ub.processProperty(Retargeting.IS_SUSPENDED, BIDS_RETARGETING.IS_SUSPENDED, x -> x ? 1L : 0L);

        ppcDslContextProvider.ppc(shard)
                .update(BIDS_RETARGETING)
                .set(ub.getValues())
                .where(BIDS_RETARGETING.RET_ID.in(ub.getChangedIds()))
                .execute();
        int updatedRowsCount = ub.getChangedIds().size();
        if (updatedRowsCount != changes.size()) {
            logger.warn(
                    "Number of updated retargeting ({}) differs from size of given collection ({}). IDs: {}",
                    updatedRowsCount, changes.size(), mapList(changes, x -> x.getModel().getId())
            );
        }
        return updatedRowsCount;
    }

    /**
     * Обновляет ретаргетинги в соответствии с параметрами, выставленными в запросе
     *
     * @param shard   шард
     * @param changes набор запросов на обновление ретаргетингов
     * @return количество обновленных сущностей
     */
    public int setBids(int shard, List<AppliedChanges<Retargeting>> changes) {
        JooqUpdateBuilder<BidsRetargetingRecord, Retargeting> ub =
                new JooqUpdateBuilder<>(BIDS_RETARGETING.RET_ID, changes);
        ub.processProperty(Retargeting.AUTOBUDGET_PRIORITY, BIDS_RETARGETING.AUTOBUDGET_PRIORITY, Integer::longValue);
        ub.processProperty(Retargeting.PRICE_CONTEXT, BIDS_RETARGETING.PRICE_CONTEXT);
        ub.processProperty(Retargeting.LAST_CHANGE_TIME, BIDS_RETARGETING.MODTIME, identity());
        ub.processProperty(Retargeting.STATUS_BS_SYNCED, BIDS_RETARGETING.STATUS_BS_SYNCED,
                status -> BidsRetargetingStatusbssynced.valueOf(status.toDbFormat()));

        ppcDslContextProvider.ppc(shard)
                .update(BIDS_RETARGETING)
                .set(ub.getValues())
                .where(BIDS_RETARGETING.RET_ID.in(ub.getChangedIds()))
                .execute();
        return ub.getChangedIds().size();
    }


    public Map<Long, Long> getAdGroupIdByRetargetingIds(int shard, Collection<Long> retIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_ID, BIDS_RETARGETING.PID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_ID.in(retIds))
                .fetchMap(BIDS_RETARGETING.RET_ID, BIDS_RETARGETING.PID);
    }

    public Map<Long, Long> getRetIdWithCidWithoutLimit(int shard, RetargetingSelection selection,
                                                       List<InterestLink> existingInterests) {
        Set<Long> queryInterestIds = listToSet(nvl(selection.getInterestIds(), emptyList()), id -> id);
        List<InterestLink> filteredInterests =
                filterList(existingInterests, i -> queryInterestIds.contains(i.getInterestId()));

        List<Long> interestRetCondIds = mapList(filteredInterests, InterestLink::getRetargetingConditionId);

        SelectConditionStep<Record2<Long, Long>> step = ppcDslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_ID, BIDS_RETARGETING.CID)
                .from(BIDS_RETARGETING).where();

        // перестраховываемся, так как ошибочное формирование запроса с пустым WHERE болезненно
        boolean isSelectiveCriteriaSpecified = false;
        if (!isEmpty(selection.getIds())) {
            step = step.and(BIDS_RETARGETING.RET_ID.in(selection.getIds()));
            isSelectiveCriteriaSpecified = true;
        }
        if (!isEmpty(selection.getAdGroupIds())) {
            step = step.and(BIDS_RETARGETING.PID.in(selection.getAdGroupIds()));
            isSelectiveCriteriaSpecified = true;
        }
        if (!isEmpty(selection.getRetargetingListIds())) {
            step = step.and(BIDS_RETARGETING.RET_COND_ID.in(selection.getRetargetingListIds()));
            isSelectiveCriteriaSpecified = true;
        }
        if (!isEmpty(selection.getInterestIds())) {
            step = step.and(BIDS_RETARGETING.RET_COND_ID.in(interestRetCondIds));
            isSelectiveCriteriaSpecified = true;
        }
        if (!isEmpty(selection.getCampaignIds())) {
            step = step.and(BIDS_RETARGETING.CID.in(selection.getCampaignIds()));
            isSelectiveCriteriaSpecified = true;
        }
        checkState(isSelectiveCriteriaSpecified, "At least one selective criteria should be specified");

        if (selection.getSuspended() != null) {
            step = step.and(BIDS_RETARGETING.IS_SUSPENDED.eq(selection.getSuspended() ? 1L : 0L));
        }
        Result<Record2<Long, Long>> result = step.fetch();
        return result.stream().collect(toMap(Record2::value1, Record2::value2));
    }

    /**
     * Получить соответствие идентификаторов ретаргетингов с номерами кампаний
     */
    public Map<Long, Long> getRetIdWithCid(int shard, Collection<Long> retargetingIds) {

        Result<Record2<Long, Long>> result = ppcDslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_ID, BIDS_RETARGETING.CID)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_ID.in(retargetingIds))
                .fetch();
        return result.stream().collect(toMap(Record2::value1, Record2::value2));
    }

    public List<Retargeting> getRetargetingsByIds(int shard, List<Long> ids, LimitOffset limitOffset) {
        return getPartialRetargetings(shard, ids, limitOffset, retargetingMapper.getFieldsToRead());
    }

    public List<RetargetingSimple> getRetargetingsSimple(int shard, Collection<Long> ids, LimitOffset limitOffset) {
        List<Retargeting> retargetings =
                getPartialRetargetings(shard, ids, limitOffset,
                        retargetingMapper.getFieldsToRead(RetargetingSimple.allModelProperties()));
        return mapList(retargetings, retargeting -> retargeting);
    }

    public List<Retargeting> getRetargetingsForStatusAggregationByIds(int shard, List<Long> ids,
                                                                      LimitOffset limitOffset) {
        return getPartialRetargetings(shard, ids, limitOffset, statusAggregationFieldsToRead);
    }

    private List<Retargeting> getPartialRetargetings(int shard, Collection<Long> ids, LimitOffset limitOffset,
                                                     Collection<Field<?>> fieldsToRead) {
        return ppcDslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(BIDS_RETARGETING)
                .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                .where(BIDS_RETARGETING.RET_ID.in(ids))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .orderBy(BIDS_RETARGETING.RET_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .map(retargetingMapper::fromDb);
    }

    public List<Long> delete(int shard, List<Long> idsToDelete) {
        if (idsToDelete.isEmpty()) {
            return emptyList();
        }

        ppcDslContextProvider.ppc(shard)
                .deleteFrom(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_ID.in(idsToDelete))
                .execute();
        return idsToDelete;
    }

    public void updateRetargetings(int shard, Collection<AppliedChanges<Retargeting>> appliedChanges) {
        JooqUpdateBuilder<BidsRetargetingRecord, Retargeting> updateBuilder =
                new JooqUpdateBuilder<>(BIDS_RETARGETING.RET_ID, appliedChanges);

        updateBuilder.processProperty(Retargeting.RETARGETING_CONDITION_ID, BIDS_RETARGETING.RET_COND_ID);
        updateBuilder.processProperty(Retargeting.PRICE_CONTEXT, BIDS_RETARGETING.PRICE_CONTEXT);
        updateBuilder.processProperty(Retargeting.AUTOBUDGET_PRIORITY, BIDS_RETARGETING.AUTOBUDGET_PRIORITY,
                RepositoryUtils::intToLong);
        updateBuilder.processProperty(Retargeting.LAST_CHANGE_TIME, BIDS_RETARGETING.MODTIME);
        updateBuilder.processProperty(Retargeting.IS_SUSPENDED, BIDS_RETARGETING.IS_SUSPENDED,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(Retargeting.STATUS_BS_SYNCED, BIDS_RETARGETING.STATUS_BS_SYNCED,
                status -> BidsRetargetingStatusbssynced.valueOf(status.toDbFormat()));

        ppcDslContextProvider.ppc(shard)
                .update(BIDS_RETARGETING)
                .set(updateBuilder.getValues())
                .where(BIDS_RETARGETING.RET_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
