package ru.yandex.direct.core.entity.retargeting.repository;

import java.math.BigInteger;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import com.google.common.collect.BiMap;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;

import static com.google.common.collect.ImmutableBiMap.toImmutableBiMap;

/**
 * Кэш для элементов таблицы категорий интересов (интересы для РМП). Является надстройкой
 * над {@link TargetingCategoriesRepository}, кэширует записи и строит над ними два отображения идентификаторов:
 * прямое и обратное для удобства транслирования.
 */
@Service
@ParametersAreNonnullByDefault
public class TargetingCategoriesCache {
    private static final Duration RELOAD_INTERVAL = Duration.ofMinutes(15);

    private final TargetingCategoriesRepository targetingCategoriesRepository;

    private Supplier<DataContainer> dataContainerSupplier;

    public TargetingCategoriesCache(TargetingCategoriesRepository targetingCategoriesRepository) {
        this.targetingCategoriesRepository = targetingCategoriesRepository;
        invalidate();
    }

    /**
     * Инвалидировать кэш. Нужно в случае добавления новых категорий интересов в базу данных.
     */
    public void invalidate() {
        dataContainerSupplier = Suppliers.memoizeWithExpiration(
                this::computeData, RELOAD_INTERVAL.getSeconds(), TimeUnit.SECONDS);
    }

    /**
     * Получить кэшированный список категорий интересов из базы.
     */
    public List<TargetingCategory> getTargetingCategories() {
        return dataContainerSupplier.get().targetingCategories;
    }

    /**
     * Получить ID в источнике (например номер цели в метрике, для интересов РМП) по категории
     * интересов (InterestId в Dictionaries.get в API)
     */
    public Long getCategoryByImportId(BigInteger importId) {
        return dataContainerSupplier.get().categoryToImportId.inverse().get(importId);
    }

    /**
     * Получить категорию интересов (InterestId в Dictionaries.get в API) по ID в источнике.
     */
    public BigInteger getImportIdByCategory(Long category) {
        return dataContainerSupplier.get().categoryToImportId.get(category);
    }

    private DataContainer computeData() {
        DataContainer dataContainer = new DataContainer();
        dataContainer.targetingCategories = targetingCategoriesRepository.getAll();
        dataContainer.categoryToImportId = dataContainer.targetingCategories.stream()
                .collect(toImmutableBiMap(TargetingCategory::getTargetingCategoryId, TargetingCategory::getImportId));
        return dataContainer;
    }

    private static class DataContainer {
        private List<TargetingCategory> targetingCategories;
        private BiMap<Long, BigInteger> categoryToImportId;
    }
}
