package ru.yandex.direct.core.entity.retargeting.repository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.Record5;
import org.jooq.Result;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.dbschema.ppcdict.enums.TargetingCategoriesState;
import ru.yandex.direct.dbschema.ppcdict.enums.TargetingCategoriesTargetingType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppcdict.tables.TargetingCategories.TARGETING_CATEGORIES;

@Repository
public class TargetingCategoriesRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public TargetingCategoriesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить список категорий интересов из базы. Для получения кэшированного списка можно
     * использовать {@link TargetingCategoriesCache}
     *
     * @return Список ID всех существующих в словаре категорий интересов
     */
    public List<TargetingCategory> getAll() {
        // для выбора доступны только те категории, которые являются листьями (не имеют дочерних категорий)
        DSLContext ctx = dslContextProvider.ppcdict();
        Result<Record5<Long, ULong, Long, String, String>> result =
                ctx.select(TARGETING_CATEGORIES.CATEGORY_ID,
                        TARGETING_CATEGORIES.IMPORT_ID,
                        TARGETING_CATEGORIES.PARENT_CATEGORY_ID,
                        TARGETING_CATEGORIES.NAME,
                        TARGETING_CATEGORIES.ORIGINAL_NAME)
                        .from(TARGETING_CATEGORIES)
                        .where(TARGETING_CATEGORIES.TARGETING_TYPE.eq(TargetingCategoriesTargetingType.rmp_interest))
                        .and(TARGETING_CATEGORIES.STATE.eq(TargetingCategoriesState.Submitted))
                        .orderBy(TARGETING_CATEGORIES.ORDER_NUM, TARGETING_CATEGORIES.CATEGORY_ID)
                        .fetch();

        Set<Long> parentIds = result.stream()
                .map(record -> record.getValue(TARGETING_CATEGORIES.PARENT_CATEGORY_ID))
                .collect(toSet());

        return result.stream()
                .map(r ->
                        new TargetingCategory(r.getValue(TARGETING_CATEGORIES.CATEGORY_ID),
                                r.getValue(TARGETING_CATEGORIES.PARENT_CATEGORY_ID),
                                r.getValue(TARGETING_CATEGORIES.NAME),
                                r.getValue(TARGETING_CATEGORIES.ORIGINAL_NAME),
                                r.getValue(TARGETING_CATEGORIES.IMPORT_ID).toBigInteger(),
                                !parentIds.contains(r.getValue(TARGETING_CATEGORIES.CATEGORY_ID))
                        )
                )
                .collect(toList());
    }

    /**
     * Проверить, изменялся ли справочник интересов после указанного момента времени.
     * использовать {@link TargetingCategoriesCache}
     *
     * @return true - если справочник изменялся.
     */
    public boolean existsNewRecordsAfter(LocalDateTime checkingTime) {

        DSLContext ctx = dslContextProvider.ppcdict();
        Record1<Long> result = ctx
                .select(TARGETING_CATEGORIES.CATEGORY_ID)
                .from(TARGETING_CATEGORIES)
                .where(TARGETING_CATEGORIES.TARGETING_TYPE.eq(TargetingCategoriesTargetingType.rmp_interest))
                .and(TARGETING_CATEGORIES.STATE.eq(TargetingCategoriesState.Submitted))
                .and(TARGETING_CATEGORIES.LAST_CHANGE.ge(checkingTime))
                .limit(1)
                .fetchAny();

        return result != null;
    }
}
