package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Predicate;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.copyentity.translations.RenameProcessor;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingConditionValidationData;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalInterest;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.helper.RetargetingConditionWithLalSegmentHelper;
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingConditionValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.ExistingRetargetingConditionsInfo;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingConditionsValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.extractLalSegmentIds;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.ExistingRetargetingConditionsInfo.fromRetargetingConditionsValidationData;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция добавления условий ретаргетинга
 */
public class AddRetargetingConditionsOperation extends SimpleAbstractAddOperation<RetargetingCondition, Long> {

    private final AddRetargetingConditionValidationService2 addValidationService;
    private final RenameProcessor renameProcessor;
    private final Locale locale;
    private final RetargetingConditionRepository retConditionRepository;
    private final LalSegmentRepository lalSegmentRepository;
    private final RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper;
    private final ClientId clientId;
    private final int shard;
    private final boolean forceSkipGoalExistenceCheck;
    private final boolean mergeConditionsIfNeeded;
    private final Map<RetargetingCondition, Long> modelExistingIds;

    private List<RetargetingConditionValidationData> existingConditionsData;

    public AddRetargetingConditionsOperation(Applicability applicability,
                                             List<RetargetingCondition> models,
                                             RetargetingConditionRepository retConditionRepository,
                                             RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper,
                                             LalSegmentRepository lalSegmentRepository,
                                             AddRetargetingConditionValidationService2 addValidationService,
                                             RenameProcessor renameProcessor,
                                             Locale locale,
                                             ClientId clientId,
                                             int shard,
                                             boolean forceSkipGoalExistenceCheck,
                                             boolean mergeConditionsIfNeeded) {
        super(applicability, models);

        this.retConditionRepository = retConditionRepository;
        this.retConditionWithLalSegmentHelper = retConditionWithLalSegmentHelper;
        this.lalSegmentRepository = lalSegmentRepository;
        this.addValidationService = addValidationService;
        this.renameProcessor = renameProcessor;
        this.locale = locale;
        this.clientId = clientId;
        this.shard = shard;
        this.forceSkipGoalExistenceCheck = forceSkipGoalExistenceCheck;
        this.mergeConditionsIfNeeded = mergeConditionsIfNeeded;
        modelExistingIds = new IdentityHashMap<>();
    }

    public AddRetargetingConditionsOperation(Applicability applicability,
                                             List<RetargetingCondition> models,
                                             RetargetingConditionRepository retConditionRepository,
                                             RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper,
                                             LalSegmentRepository lalSegmentRepository,
                                             AddRetargetingConditionValidationService2 addValidationService,
                                             ClientId clientId,
                                             int shard) {
        this(applicability, models, retConditionRepository, retConditionWithLalSegmentHelper, lalSegmentRepository,
                addValidationService, null, null, clientId, shard,
                false, false);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<RetargetingCondition> modelsPreValidatedStep) {
        existingConditionsData = retConditionRepository.getValidationData(shard, clientId);
        if (mergeConditionsIfNeeded) {
            Collection<RetargetingCondition> models = modelsPreValidatedStep.getPreValidModelsMap().values();
            tryToMergeRetargetingConditions(models);
        }
        super.onPreValidated(modelsPreValidatedStep);
    }

    private void tryToMergeRetargetingConditions(Collection<RetargetingCondition> retargetingConditions) {
        mergeConditionsByRules(retargetingConditions);
        tryToUnifyConditionsNames(retargetingConditions);
    }

    private void mergeConditionsByRules(Collection<RetargetingCondition> retargetingConditions) {
        List<RetargetingCondition> candidatesToMerge = StreamEx.of(retargetingConditions)
                .filter(rc -> RetargetingConditionsValidator
                        .conditionNotAllowedDuplicateRules(rc.getType(), rc.getAutoRetargeting()))
                .toList();
        if (candidatesToMerge.isEmpty()) {
            return;
        }

        Map<List<Rule>, Long> existingIdsByRule = StreamEx.of(existingConditionsData)
                .toMap(rc -> RetargetingConditionMappings.rulesFromJson(rc.getRulesJson()),
                        RetargetingConditionValidationData::getId);

        for (RetargetingCondition retCondition: candidatesToMerge) {
            Long existingId = existingIdsByRule.get(retCondition.getRules());
            if (existingId != null) {
                modelExistingIds.put(retCondition, existingId);
            }
        }
    }

    private void tryToUnifyConditionsNames(Collection<RetargetingCondition> retargetingConditions) {
        List<RetargetingCondition> candidatesToCreateUniqueName = StreamEx.of(retargetingConditions)
                .filter(rc -> !modelExistingIds.containsKey(rc))
                .filter(RetargetingConditionsValidator::conditionNotAllowedDuplicateName)
                .toList();

        ExistingRetargetingConditionsInfo existingRetargetingConditionsInfo =
                fromRetargetingConditionsValidationData(existingConditionsData);

        renameProcessor.tryToUnifyRetargetingCopyNames(
                existingRetargetingConditionsInfo.getExistingNames(),
                candidatesToCreateUniqueName,
                locale);
    }

    @Override
    protected void validate(ValidationResult<List<RetargetingCondition>, Defect> preValidationResult) {
        // Если нужно смержить условия ретаргетенга с уже существующими в БД, то передаем валидатору пустые списки
        // существующих имен и правил, чтобы он не проверял их уникальность. Имена мы постарались уникализировать
        // в tryToUnifyConditionsNames, и если не вышло (что крайне маловероятно), то все равно лучше сохранить
        // условия ретаргетинга с дублирующимися именами, чем не сохранить их вовсе. Особенно это важно при копировании.
        // А уникальность правил проверять валидатору в этом случае не нужно, так как мы не будем сохранять условия
        // ретаргетинга с дублирующимися правилами, а возьмем идентификаторы уже существующих условий ретаргетинга.
        // При этом мы укажем корректное количество существующих условий ретаргетинга, чтобы не создать
        // больше чем можно.
        ExistingRetargetingConditionsInfo existingRetargetingConditionsInfo = mergeConditionsIfNeeded ?
            new ExistingRetargetingConditionsInfo(existingConditionsData.size() - modelExistingIds.size()) :
            fromRetargetingConditionsValidationData(existingConditionsData);

        List<RetargetingCondition> retConditionList = preValidationResult.getValue();

        validationResult = addValidationService.validateWithOptionalForceSkipGoalExistenceCheck(
                retConditionList, clientId, shard, existingRetargetingConditionsInfo, forceSkipGoalExistenceCheck);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<RetargetingCondition> modelsValidatedStep) {
        modelsValidatedStep.getValidModelsMap().values().forEach(this::preprocessRetConditionForAdd);
    }

    private void preprocessRetConditionForAdd(RetargetingCondition retCondition) {
        if (retCondition.getInterest()) {
            List<Rule> rules = StreamEx.of(retCondition.getRules())
                    .mapToEntry(Rule::getGoals)
                    .mapValues(this::convertToGoalInterest)
                    .mapKeyValue((rule, goals) -> (Rule) new RuleInterest(rule.getType(), goals))
                    .toList();
            retCondition.setRules(rules);
            retCondition.setName("");
            retCondition.setDescription("");
        }
        retCondition.setDeleted(false);
        retCondition.setLastChangeTime(LocalDateTime.now());
        if (retCondition.getDescription() == null) {
            retCondition.setDescription("");
        }
        retConditionWithLalSegmentHelper.updateLalSegmentsOnRules(retCondition.getRules());
    }

    private List<Goal> convertToGoalInterest(Collection<Goal> goals) {
        return mapList(goals, g -> new GoalInterest(g.getId(), g.getTime()));
    }

    @Override
    protected List<Long> execute(List<RetargetingCondition> validRetConditions) {
        List<RetargetingCondition> notExistsRetConditions =
                filterList(validRetConditions, Predicate.not(modelExistingIds::containsKey));
        retConditionRepository.add(shard, notExistsRetConditions);
        lalSegmentRepository.activateLalSegments(extractLalSegmentIds(notExistsRetConditions));
        modelExistingIds.forEach(RetargetingCondition::setId);
        return mapList(validRetConditions, RetargetingCondition::getId);
    }
}
