package ru.yandex.direct.core.entity.retargeting.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageGeoTree;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingAdGroupInfo;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingValidationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.CPM_PRICE;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.getPackagePriceFunctionByCampaignId;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.intRange;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция добавления ретаргетингов
 * <p>
 * Параметр {@code autoPrices} включает режим автоматического выставления
 * ставок, если их нет в запросе.
 * При этом параметр {@code fixedAutoPrices} должен быть не {@code null}.
 * <p>
 * В режиме {@code autoPrices} сначала пробуем достать ставки из контейнера
 * {@code fixedAutoPrices}, но если там нет, то выставляется минимальная ставка
 * в валюте клиента.
 * <p>
 * Если режим {@code autoPrices} выключен, всем ретаргетингам без ставки
 * выставляется минимальная в валюте клиента.
 * <p>
 * Если у ретаргетинга не был выставлен приоритет автобюджета, проставляется
 * приоритет по умолчанию.
 * <p>
 * Параметры {@code autoPrices} и {@code fixedAutoPrices} игнорируются прайсовыми
 * кампаниями.
 */
public class AddRetargetingsOperation extends SimpleAbstractAddOperation<TargetInterest, Long> {

    private final RetargetingService retargetingService;
    private final AddRetargetingValidationService validationService;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;
    private final PricePackageGeoTree pricePackageGeoTree;
    private final MarkupConditionRepository markupConditionRepository;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final AddTargetInterestService targetInterestService;
    private final ClientGeoService clientGeoService;
    /**
     * Включен ли режим {@code autoPrices}. См. коммент к классу.
     * Игнорируется в прайсовых кампаниях.
     */
    private final boolean autoPrices;
    /**
     * Контейнер с фиксированными ставками, которые нужно выставить у
     * ретаргетингов, если ставка не была явно указана. Ставки могут быть
     * не для всех ретаргетингов. Должен быть не {@code null},
     * если {@code autoPrices == true}.
     */
    @Nullable
    private final ShowConditionFixedAutoPrices fixedAutoPrices;

    private final boolean adGroupsNonexistentOnPrepare;
    private final boolean retargetingConditionsNonexistentOnPrepare;
    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;
    private final Currency clientCurrency;
    private final boolean isCopy;

    private boolean existentAdGroupsDataReady;
    private Map<Long, AdGroupSimple> existentAdGroupsInfo;
    private Map<Long, AdGroup> preparedAdGroupModels;
    private Map<Long, List<Long>> existentProjectParamConditions;
    private Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> existentFrontpageAdGroupsPriceRestrictions;
    private List<TargetInterest> existingTargetInterests;
    private Map<Integer, RetargetingAdGroupInfo> nonexistentAdGroupsInfo;
    private Map<Long, CampaignType> campaignsType;

    private List<RetargetingCondition> retargetingConditions;

    /**
     * @param autoPrices      включает режим {@code autoPrices}, см. коммент к классу
     * @param fixedAutoPrices контейнер со ставками для ретаргетингов, у которых нет своих ставок.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}. См коммент к классу.
     */
    public AddRetargetingsOperation(Applicability applicability,
                                    boolean adGroupsNonexistentOnPrepare,
                                    boolean retargetingConditionsNonexistentOnPrepare,
                                    List<TargetInterest> models,
                                    RetargetingService retargetingService,
                                    AddRetargetingValidationService validationService,
                                    ClientService clientService,
                                    ClientGeoService clientGeoService,
                                    RetargetingConditionRepository retargetingConditionRepository,
                                    AdGroupRepository adGroupRepository,
                                    CampaignRepository campaignRepository,
                                    CampaignTypedRepository campaignTypedRepository,
                                    PricePackageRepository pricePackageRepository,
                                    PricePackageGeoTree pricePackageGeoTree,
                                    MarkupConditionRepository markupConditionRepository,
                                    CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
                                    AddTargetInterestService targetInterestService,
                                    boolean autoPrices,
                                    @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                    long operatorUid, ClientId clientId, long clientUid, int shard, boolean isCopy) {
        super(applicability, models);
        this.adGroupsNonexistentOnPrepare = adGroupsNonexistentOnPrepare;
        this.retargetingConditionsNonexistentOnPrepare = retargetingConditionsNonexistentOnPrepare;
        this.retargetingService = retargetingService;
        this.validationService = validationService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.pricePackageGeoTree = pricePackageGeoTree;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.markupConditionRepository = markupConditionRepository;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.targetInterestService = targetInterestService;
        this.clientGeoService = clientGeoService;
        this.autoPrices = autoPrices;
        this.fixedAutoPrices = fixedAutoPrices;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
        this.isCopy = isCopy;

        if (autoPrices) {
            checkArgument(fixedAutoPrices != null, "fixedAutoPrices must be specified in autoPrices mode");
        }

        clientCurrency = clientService.getWorkCurrency(clientId);
    }

    public void setNonexistentAdGroupsInfo(Map<Integer, RetargetingAdGroupInfo> nonexistentAdGroupsInfo) {
        checkState(adGroupsNonexistentOnPrepare, "ad group info can't be set if groups exist");
        checkState(!isPrepared(), "operation is already prepared");
        this.nonexistentAdGroupsInfo = nonexistentAdGroupsInfo;
        setFakeIds();
    }

    private void setFakeIds() {
        List<TargetInterest> targetInterests = getModels();
        for (int i = 0; i < targetInterests.size(); ++i) {
            Long adGroupFakeId = nonexistentAdGroupsInfo.get(i).getAdGroupFakeId();
            targetInterests.get(i).setAdGroupId(adGroupFakeId);
        }
    }

    public void setAdGroupsIds(Map<Integer, Long> adGroupIds) {
        checkState(adGroupsNonexistentOnPrepare, "ids can be set only when ad groups non existent on prepare");
        checkState(isPrepared(), "operation is not prepared yet");
        checkState(!isExecuted(), "operation is already executed");
        List<TargetInterest> retargetings = getModels();
        for (int i = 0; i < retargetings.size(); ++i) {
            retargetings.get(i).setAdGroupId(adGroupIds.get(i));
        }
    }

    public void setExistentAdGroupsInfo(Map<Long, AdGroupSimple> existentAdGroupsInfo,
                                        List<TargetInterest> existingTargetInterests,
                                        Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> existentFrontpageAdGroupsPriceRestrictions,
                                        Map<Long, AdGroup> preparedAdGroupModels) {
        checkState(!adGroupsNonexistentOnPrepare, "ad groups nonexistent on prepare");
        checkArgument(existentAdGroupsInfo != null, "existentAdGroupsInfo is required");
        checkArgument(existingTargetInterests != null, "existingTargetInterests is required");
        checkState(!isPrepared(), "data must be set before prepare");
        checkState(!existentAdGroupsDataReady, "data for prepare-stage has been already set");
        this.existentAdGroupsInfo = existentAdGroupsInfo;
        this.preparedAdGroupModels = preparedAdGroupModels;
        this.existentFrontpageAdGroupsPriceRestrictions = existentFrontpageAdGroupsPriceRestrictions;
        this.existingTargetInterests = existingTargetInterests;
        existentAdGroupsDataReady = true;
    }

    public void setNonexistentRetargetingCondition(List<RetargetingCondition> nonexistentRetargetingConditions) {
        checkState(retargetingConditionsNonexistentOnPrepare,
                "retargeting conditions can't be set if conditions exist");
        checkState(!isPrepared(), "operation is already prepared");
        this.retargetingConditions = nonexistentRetargetingConditions;

        setRetargetingConditionIdInTargetInterests(nonexistentRetargetingConditions);
    }

    public void setRetargetingConditionsIds(List<RetargetingCondition> retargetingConditions) {
        checkState(retargetingConditionsNonexistentOnPrepare,
                "retargeting conditions can't be set if conditions exist");
        checkState(isPrepared(), "operation is not prepared yet");
        checkState(!isExecuted(), "operation is already executed");

        setRetargetingConditionIdInTargetInterests(retargetingConditions);
    }

    private void setRetargetingConditionIdInTargetInterests(List<RetargetingCondition> retargetingConditions) {
        EntryStream.of(getModels()).forKeyValue((index, targetInterest) -> {
            Long retConditionId = retargetingConditions.get(index).getId();
            targetInterest.setRetargetingConditionId(retConditionId);
        });
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<TargetInterest> modelsPreValidatedStep) {
        initDataForPrepare(modelsPreValidatedStep.getPreValidModelsMap().values());
        initRetargetingConditionsForPrepare(modelsPreValidatedStep.getPreValidModelsMap().values());
    }

    private void initDataForPrepare(Collection<TargetInterest> preValidTargetInterests) {
        if (!adGroupsNonexistentOnPrepare && !existentAdGroupsDataReady) {
            Set<Long> adGroupIds = StreamEx.of(preValidTargetInterests)
                    .filter(Objects::nonNull)
                    .map(TargetInterest::getAdGroupId)
                    .filter(Objects::nonNull)
                    .collect(toSet());

            existentAdGroupsInfo = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);
            existentProjectParamConditions = adGroupRepository.getAdGroupsProjectParamConditions(shard, adGroupIds);
            existentFrontpageAdGroupsPriceRestrictions = cpmYndxFrontpageCurrencyService
                    .getAdGroupIdsToPriceDataMapByAdGroups(existentAdGroupsInfo.values(), shard, clientCurrency);
            existingTargetInterests = retargetingService
                    .getTargetInterestsWithInterestByAdGroupIds(existentAdGroupsInfo.keySet(), clientId, shard);
        }
        initCampaignsTypeAndPricePackagePriceForPrepare();
    }

    private void initCampaignsTypeAndPricePackagePriceForPrepare() {
        Set<Long> campaignIds = adGroupsNonexistentOnPrepare
                ? nonexistentAdGroupsInfo.values().stream().map(RetargetingAdGroupInfo::getCampaignId).collect(toSet())
                : existentAdGroupsInfo.values().stream().map(AdGroupSimple::getCampaignId).collect(toSet());
        campaignsType = campaignRepository.getCampaignsTypeMap(shard, campaignIds);
    }

    private void initRetargetingConditionsForPrepare(Collection<TargetInterest> preValidTargetInterests) {
        if (retargetingConditionsNonexistentOnPrepare) {
            return;
        }

        Set<Long> retCondIds = StreamEx.of(preValidTargetInterests)
                .filter(Objects::nonNull)
                .map(TargetInterest::getRetargetingConditionId)
                .filter(Objects::nonNull)
                .collect(toSet());

        retargetingConditions =
                retargetingConditionRepository.getFromRetargetingConditionsTable(shard, clientId, retCondIds);
    }

    @Override
    protected void validate(ValidationResult<List<TargetInterest>, Defect> preValidationResult) {
        if (adGroupsNonexistentOnPrepare) {
            validationService
                    .validateWithNonExistentAdGroups(validationResult, nonexistentAdGroupsInfo, retargetingConditions,
                            retargetingConditionsNonexistentOnPrepare, clientId, shard, isCopy);
        } else {
            validationService.validate(validationResult, existingTargetInterests,
                    existentAdGroupsInfo, operatorUid, clientId, shard, isCopy);
        }
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<TargetInterest> modelsValidatedStep) {
        Map<Integer, TargetInterest> validModelsMap = modelsValidatedStep.getValidModelsMap();
        fillPriceContext(validModelsMap);
        fillSystemFields(validModelsMap);
    }

    private void fillPriceContext(Map<Integer, TargetInterest> validModelsMap) {
        Map<Long, List<Goal>> goalsByRetCondId = retargetingConditions.stream()
                .collect(toMap(ModelWithId::getId, RetargetingCondition::collectGoals));
        var packagePriceFunctionByCampaignId =
                getPackagePriceFunctionByCampaignId(shard, campaignTypedRepository, pricePackageRepository,
                        markupConditionRepository, campaignsType);

        GeoTree geoTree = pricePackageGeoTree.getGeoTree();

        validModelsMap.forEach((index, ti) -> {
            Long campaignId = getCampaignId(index, ti);
            CampaignType campaignType = campaignsType.get(campaignId);
            if (campaignType == CPM_PRICE) {
                checkState(ti.getPriceContext() == null);
                List<Long> projectParamConditions = getAdGroupProjectParamConditions(index, ti);
                List<Long> goalIds = mapList(goalsByRetCondId.get(ti.getRetargetingConditionId()), ModelWithId::getId);
                List<Long> geo = clientGeoService.convertForSave(getAdGroupGeo(index, ti), geoTree);
                var priceCalculator = packagePriceFunctionByCampaignId.get(campaignId);
                var priceCalculation = priceCalculator.apply(geo, goalIds, projectParamConditions);
                BigDecimal pricePackagePrice = (priceCalculation != null) ? priceCalculation.getPrice() :
                        priceCalculator.getDefaultPricePackagePrice();
                ti.setPriceContext(pricePackagePrice);
            } else {
                if (autoPrices) {
                    setFixedAutoPrice(ti);
                }
                calcMissingPrice(index, ti);
            }
        });
    }

    /**
     * Пытаемся выставить фиксированные ставки из контейнера {@code fixedAutoPrices},
     * если нужно.
     */
    private void setFixedAutoPrice(TargetInterest ti) {
        checkNotNull(fixedAutoPrices);
        if (ti.getPriceContext() != null) {
            return;
        }

        BigDecimal fixedPrice = null;
        if (fixedAutoPrices.hasGlobalFixedPrice()) {
            fixedPrice = fixedAutoPrices.getGlobalFixedPrice();
        } else if (!adGroupsNonexistentOnPrepare && fixedAutoPrices.hasAdGroupFixedPrice(ti.getAdGroupId())) {
            fixedPrice = fixedAutoPrices.getAdGroupFixedPrice(ti.getAdGroupId());
        }

        if (fixedPrice != null) {
            ti.setPriceContext(fixedPrice);
        }
    }

    /**
     * Выставляем ставки и приоритеты автобюджета по умолчанию везде,
     * где их нет.
     */
    private void calcMissingPrice(Integer index, TargetInterest ti) {
        BigDecimal defaultPrice = clientCurrency.getMinPrice();

        AdGroupType adGroupType = getAdGroupType(index, ti);
        switch (adGroupType) {
            case BASE:
            case MOBILE_CONTENT:
                if (ti.getPriceContext() == null) {
                    ti.setPriceContext(defaultPrice);
                }

                if (ti.getAutobudgetPriority() == null) {
                    ti.setAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY);
                }
                break;
            case CPM_VIDEO:
            case CPM_BANNER:
            case CPM_OUTDOOR:
            case CPM_INDOOR:
            case CPM_AUDIO:
            case CPM_GEO_PIN:
            case CPM_GEOPRODUCT:
                if (ti.getPriceContext() == null) {
                    ti.setPriceContext(clientCurrency.getMinCpmPrice());
                }
                break;
            case CPM_YNDX_FRONTPAGE:
                calcFrontpageMissingPrice(index, ti);
                break;
            default:
                throw new UnsupportedOperationException("Ad group type not supported");
        }
    }

    private AdGroupType getAdGroupType(int index, TargetInterest ti) {
        return adGroupsNonexistentOnPrepare
                ? nonexistentAdGroupsInfo.get(index).getAdGroupType()
                : existentAdGroupsInfo.get(ti.getAdGroupId()).getType();
    }

    private List<Long> getAdGroupGeo(int index, TargetInterest ti) {
        return adGroupsNonexistentOnPrepare
                ? nonexistentAdGroupsInfo.get(index).getAdGroup().getGeo()
                : nvl(preparedAdGroupModels, existentAdGroupsInfo).get(ti.getAdGroupId()).getGeo();
    }

    private List<Long> getAdGroupProjectParamConditions(int index, TargetInterest ti) {
        return adGroupsNonexistentOnPrepare
                ? nonexistentAdGroupsInfo.get(index).getAdGroup().getProjectParamConditions()
                : preparedAdGroupModels != null
                        ? preparedAdGroupModels.get(ti.getAdGroupId()).getGeo()
                        : existentProjectParamConditions.get(ti.getAdGroupId());
    }

    private Long getCampaignId(int index, TargetInterest ti) {
        return adGroupsNonexistentOnPrepare
                ? nonexistentAdGroupsInfo.get(index).getCampaignId()
                : existentAdGroupsInfo.get(ti.getAdGroupId()).getCampaignId();
    }

    private void calcFrontpageMissingPrice(Integer index, TargetInterest targetInterest) {
        if (targetInterest.getPriceContext() != null) {
            return;
        }
        BigDecimal cpmYndxFrontpageDefaultPrice;
        if (adGroupsNonexistentOnPrepare) {
            cpmYndxFrontpageDefaultPrice = ifNotNull(
                    nonexistentAdGroupsInfo.get(index).getCpmYndxFrontpageAdGroupPriceRestrictions(),
                    CpmYndxFrontpageAdGroupPriceRestrictions::getCpmYndxFrontpageMinPrice);
        } else {
            cpmYndxFrontpageDefaultPrice = ifNotNull(
                    existentFrontpageAdGroupsPriceRestrictions.get(targetInterest.getAdGroupId()),
                    CpmYndxFrontpageAdGroupPriceRestrictions::getCpmYndxFrontpageMinPrice);
        }
        checkState(cpmYndxFrontpageDefaultPrice != null, "Frontpage default price not found correctly");
        targetInterest.setPriceContext(cpmYndxFrontpageDefaultPrice);

    }

    private void fillSystemFields(Map<Integer, TargetInterest> targetInterests) {
        LocalDateTime now = LocalDateTime.now();
        targetInterests.forEach((index, ti) -> {
            if (adGroupsNonexistentOnPrepare) {
                Long campaignId = nonexistentAdGroupsInfo.get(index).getCampaignId();
                ti.setCampaignId(campaignId);
            } else {
                AdGroupSimple adGroupSimple = existentAdGroupsInfo.get(ti.getAdGroupId());
                checkState(adGroupSimple != null,
                        "there is no AdGroup for id = %s in existentAdGroupsInfo", ti.getAdGroupId());
                ti.setCampaignId(adGroupSimple.getCampaignId());
            }
        });
        targetInterests.forEach((index, ti) -> ti.setLastChangeTime(now));
        targetInterests.forEach((index, ti) -> ti.setStatusBsSynced(StatusBsSynced.NO));
        targetInterests.forEach((index, ti) -> ti.setIsSuspended(false));
    }

    @Override
    protected void beforeExecution(Map<Integer, TargetInterest> validModelsMapToApply) {
        if (adGroupsNonexistentOnPrepare) {
            checkAllAdGroupIdsAreSet();
        }
    }

    /**
     * Проверяет, что у всех ретаргетингов были выставлены корректные (положительные) id групп,
     * если их не было в {@link #prepare()}
     */
    private void checkAllAdGroupIdsAreSet() {
        List<TargetInterest> models = getModels();
        Set<Integer> retargetingsWithoutAdGroupId = intRange(0, models.size())
                .stream()
                .filter(index -> {
                    Long adGroupId = models.get(index).getAdGroupId();
                    return adGroupId == null || adGroupId <= 0;
                })
                .collect(Collectors.toSet());
        checkState(retargetingsWithoutAdGroupId.isEmpty(),
                "no ad group id for retargetings with indexes: " + retargetingsWithoutAdGroupId);
    }

    @Override
    protected List<Long> execute(List<TargetInterest> validTargetInterestsToApply) {
        Map<Long, AdGroupType> adGroupIdToAdGroupType = new HashMap<>();
        IntStream.range(0, validTargetInterestsToApply.size())
                .forEach(i -> {
                    TargetInterest targetInterest = validTargetInterestsToApply.get(i);
                    if (adGroupIdToAdGroupType.containsKey(targetInterest.getAdGroupId())) {
                        return;
                    }
                    AdGroupType adGroupType = getAdGroupType(i, validTargetInterestsToApply.get(i));
                    adGroupIdToAdGroupType.put(targetInterest.getAdGroupId(), adGroupType);
                });

        return targetInterestService.addValidTargetInterests(
                shard, operatorUid, UidAndClientId.of(clientUid, clientId), clientCurrency,
                validTargetInterestsToApply, adGroupIdToAdGroupType);
    }
}
