package ru.yandex.direct.core.entity.retargeting.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent.addRetargetingEvent;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.convertTargetInterestsToRetargetings;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Добавляет валидные ретаргетинги, выполняя необходимое логирование и добавление нотификаций
 */
@Service
public class AddTargetInterestService {
    private final RetargetingRepository retargetingRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final InterestLinkFactory interestLinkFactory;
    private final AdGroupRepository adGroupRepository;
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;

    public AddTargetInterestService(RetargetingRepository retargetingRepository,
                                    RetargetingConditionRepository retargetingConditionRepository,
                                    InterestLinkFactory interestLinkFactory, AdGroupRepository adGroupRepository,
                                    LogPriceService logPriceService,
                                    MailNotificationEventService mailNotificationEventService) {
        this.retargetingRepository = retargetingRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.interestLinkFactory = interestLinkFactory;
        this.adGroupRepository = adGroupRepository;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
    }

    public List<Long> addValidTargetInterests(int shard, long operatorUid, UidAndClientId owner,
                                              Currency clientCurrency, List<TargetInterest> validTargetInterests) {
        return addValidTargetInterests(shard, operatorUid, owner, clientCurrency, validTargetInterests, emptyMap());
    }

    /**
     * Добавляем недостающие интересы в retargeting_conditions (и в retargeting_goals для РМП) и в условия
     * ретаргетинга bids_retargeting. Возвращаем id ретаргетингов (bids_retargeting)
     *
     * @param adGroupIdToAdGroupType - тип группы по pid, для групп РМП добавляем еще цель в retargeting_goals
     *                               (для недостающих retargeting_conditions)
     */
    public List<Long> addValidTargetInterests(int shard, long operatorUid, UidAndClientId owner,
                                              Currency clientCurrency, List<TargetInterest> validTargetInterests,
                                              Map<Long, AdGroupType> adGroupIdToAdGroupType) {
        List<TargetInterest> targetInterestsWithInterest =
                filterList(validTargetInterests, r -> r.getInterestId() != null);

        List<TargetInterest> mobileContentTargetInterests = StreamEx.of(targetInterestsWithInterest)
                .filter(ti -> AdGroupType.MOBILE_CONTENT == adGroupIdToAdGroupType.get(ti.getAdGroupId()))
                .toList();
        List<TargetInterest> targetInterests = StreamEx.of(targetInterestsWithInterest)
                .filter(ti -> AdGroupType.MOBILE_CONTENT != adGroupIdToAdGroupType.get(ti.getAdGroupId()))
                .toList();

        List<InterestLink> existingInterestLinks = new ArrayList<>();
        if (!targetInterests.isEmpty()) {
            existingInterestLinks.addAll(
                    createMissedInterests(targetInterests, owner.getClientId(), shard));
        }
        if (!mobileContentTargetInterests.isEmpty()) {
            existingInterestLinks.addAll(
                    createMissedInterestsWithRetargetingGoals(mobileContentTargetInterests, owner.getClientId(), shard));
        }

        setInterestIdForRetargetings(targetInterestsWithInterest, existingInterestLinks);

        List<Retargeting> retargetings =
                convertTargetInterestsToRetargetings(validTargetInterests, existingInterestLinks);
        List<Long> ids = retargetingRepository.add(shard, retargetings);

        // устанавливаем id на исходных target interest
        EntryStream.zip(validTargetInterests, retargetings)
                .forKeyValue(((targetInterest, retargeting) -> targetInterest.setId(retargeting.getId())));

        Set<Long> adGroupIds = new HashSet<>(mapList(validTargetInterests, TargetInterest::getAdGroupId));
        adGroupRepository.actualizeAdGroupsOnChildModification(shard, adGroupIds);

        logPriceChanges(operatorUid, clientCurrency, retargetings);
        sendMailNotifications(shard, operatorUid, owner, retargetings);

        return ids;
    }

    /**
     * Создает в базе недостающие ссылки на интересы
     * ({@link InterestLink}: {@link RetargetingCondition} с интересом).
     *
     * @return возвращает все имеющиеся у клиента ссылки на интересы (InterestLink).
     */
    List<InterestLink> createMissedInterests(List<TargetInterest> retargetingsWithInterest,
                                             ClientId clientId, int shard) {
        return createMissedInterests(retargetingsWithInterest, clientId, shard,
                (missedCategories) -> {
                    LocalDateTime now = LocalDateTime.now();
                    List<InterestLink> interestLinksToAdd = StreamEx.of(missedCategories)
                            .map(catId -> interestLinkFactory.construct(catId, clientId, now))
                            .toList();

                    // вызов addInterests приводит к проставлению Id'шников у создаваемых сущностей
                    retargetingConditionRepository.addInterests(shard, interestLinksToAdd);
                    return interestLinksToAdd;
                });
    }

    /**
     * Создает в базе недостающие ссылки на интересы для РМП (добавляется еще цель в retargeting_goals)
     * У интересов РМП поле condition_json в retargeting_conditions записывается в 'урезанном' виде.
     * Более подробно как хранится интерес РМП в retargeting_conditions в
     * <a href="https://st.yandex-team.ru/DIRECT-110200#5f4631147f480955cb3002ab">комменте</a>
     *
     * @return возвращает все имеющиеся у клиента ссылки на интересы (InterestLink).
     */
    List<InterestLink> createMissedInterestsWithRetargetingGoals(List<TargetInterest> retargetingsWithInterest,
                                                                 ClientId clientId, int shard) {
        return createMissedInterests(retargetingsWithInterest, clientId, shard,
                (missedCategories) -> {
                    LocalDateTime now = LocalDateTime.now();
                    List<InterestLink> interestLinksToAdd = StreamEx.of(missedCategories)
                            .map(catId -> interestLinkFactory.constructForMobileContent(catId, clientId, now))
                            .toList();

                    List<RetargetingCondition> retConditionsWithInterest =
                            mapList(interestLinksToAdd, InterestLink::asRetargetingCondition);
                    // вызов add приводит к проставлению Id'шников у создаваемых сущностей
                    retargetingConditionRepository.add(shard, retConditionsWithInterest);
                    return interestLinksToAdd;
                });
    }

    private List<InterestLink> createMissedInterests(List<TargetInterest> retargetingsWithInterest,
                                                     ClientId clientId, int shard,
                                                     Function<Set<Long>, List<InterestLink>> methodToAddInterestLinks) {
        List<InterestLink> existingInterestLinks = retargetingConditionRepository.getExistingInterest(shard, clientId);
        Set<Long> existingCategories = listToSet(existingInterestLinks, InterestLink::getInterestId);

        Set<Long> missedCategories = StreamEx.of(retargetingsWithInterest)
                .map(TargetInterest::getInterestId)
                .remove(existingCategories::contains)
                .toSet();

        existingInterestLinks.addAll(methodToAddInterestLinks.apply(missedCategories));
        return existingInterestLinks;
    }

    /**
     * У переданных {@link Retargeting}'ов проставляет {@code interestId},
     * если находится соответсвие в {@code existingInterestLink}
     *
     * @param targetInterests      {@link Retargeting}'и
     * @param existingInterestLink набор {@link InterestLink}
     */
    private void setInterestIdForRetargetings(Collection<TargetInterest> targetInterests,
                                              Collection<InterestLink> existingInterestLink) {
        Multimap<Long, InterestLink> interestById = Multimaps.index(existingInterestLink, InterestLink::getInterestId);

        targetInterests.forEach(
                r -> r.setRetargetingConditionId(
                        // игнорируем дубликаты -- берём первый из списка Interest по заданному interestId
                        Iterables.getOnlyElement(interestById.get(r.getInterestId())).getRetargetingConditionId())
        );
    }

    /**
     * Логгирование изменения цен
     */
    private void logPriceChanges(long operatorUid, Currency clientCurrency, List<Retargeting> retargetings) {
        Function<Retargeting, LogPriceData> retargetingToLog = r -> new LogPriceData(
                r.getCampaignId(),
                r.getAdGroupId(),
                r.getId(),
                nvl(r.getPriceContext(), BigDecimal.ZERO).doubleValue(),
                null,
                clientCurrency.getCode(),
                LogPriceData.OperationType.RET_ADD);

        List<LogPriceData> logPriceList = mapList(retargetings, retargetingToLog);
        logPriceService.logPrice(logPriceList, operatorUid);
    }

    private void sendMailNotifications(int shard, Long operatorUid, UidAndClientId owner,
                                       List<Retargeting> retargetings) {
        Set<Long> affectedAdGroupIds = listToSet(retargetings, Retargeting::getAdGroupId);
        Map<Long, AdGroupName> adGroupIdToAdGroup =
                adGroupRepository.getAdGroupNames(shard, owner.getClientId(), affectedAdGroupIds);
        checkState(affectedAdGroupIds.size() == adGroupIdToAdGroup.keySet().size(),
                "All ad groups of deleted retargetings must exists.");

        List<RetargetingEvent> events = new ArrayList<>();

        retargetings.forEach(r -> {
            Long newValue = r.getRetargetingConditionId();
            checkNotNull(newValue, "retargeting condition id must be not null");

            long adGroupId = r.getAdGroupId();
            long campaignId = adGroupIdToAdGroup.get(adGroupId).getCampaignId();
            String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();

            events.add(addRetargetingEvent(operatorUid, owner.getUid(), campaignId, adGroupId, adGroupName,
                    newValue));
        });
        mailNotificationEventService.queueEvents(operatorUid, owner.getClientId(), events);
    }
}
