package ru.yandex.direct.core.entity.retargeting.service;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal;
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoalId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Сервис работы с сущностями "целим на кампании" (camp_metrika_goals)
 */
@Component
@ParametersAreNonnullByDefault
public class CampMetrikaGoalService implements EntityService<CampMetrikaGoal, CampMetrikaGoalId> {
    private final ShardHelper shardHelper;
    private final MetrikaCampaignRepository repository;

    public CampMetrikaGoalService(ShardHelper shardHelper, MetrikaCampaignRepository repository) {
        this.shardHelper = shardHelper;
        this.repository = repository;
    }

    @Override
    public List<CampMetrikaGoal> get(ClientId clientId, Long operatorUid, Collection<CampMetrikaGoalId> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return repository.getCampMetrikaGoalsByIds(shard, clientId, List.copyOf(ids));
    }

    private MassResult<CampMetrikaGoal> add(int shard, List<CampMetrikaGoal> entities) {
        ValidationResult<List<CampMetrikaGoal>, Defect> vr = new ValidationResult<>(entities);

        repository.addCampMetrikaGoals(shard, entities);

        entities.forEach(campMetrikaGoal -> campMetrikaGoal
                .withId(new CampMetrikaGoalId()
                        .withCampaignId(campMetrikaGoal.getCampaignId())
                        .withGoalId(campMetrikaGoal.getGoalId())));

        return MassResult.successfulMassAction(entities, vr);
    }

    @Override
    public MassResult<CampMetrikaGoal> add(ClientId clientId, Long operatorUid, List<CampMetrikaGoal> entities,
                                           Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return add(shard, entities);
    }

    @Override
    public MassResult<CampMetrikaGoal> copy(CopyOperationContainer copyContainer,
                                            List<CampMetrikaGoal> entities, Applicability applicability) {
        return add(copyContainer.getShardTo(), entities);
    }
}
