package ru.yandex.direct.core.entity.retargeting.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.RetargetingCampaignInfo;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.audienceTargetNotFound;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedRetargetingId;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.deleteIsAllowed;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.nonArchivedCampaignOnDelete;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.operatorHasRightsToChange;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.retargetingExistsAndVisible;
import static ru.yandex.direct.core.validation.BaseDeleteValidation.idsToDeleteValidate;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRightsCantWrite;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

@Service
public class DeleteRetargetingValidationService {

    private final RbacService rbacService;

    @Autowired
    public DeleteRetargetingValidationService(RbacService rbacService) {
        this.rbacService = rbacService;
    }

    /**
     * Основной метод валидации при удалении ретаргетингов, невалидные идентификаторы уже должны быть помечены как ошибочные
     *
     * @param ids         список идентификаторов ретаргетингов для валидации удаления
     * @param operatorUid uid оператора
     * @return результат валидации
     */
    public ValidationResult<List<Long>, Defect> validate(List<Long> ids,
                                                         Map<Long, RetargetingCampaignInfo> retargetingsInfoMap, long operatorUid) {
        Map<Long, Long> campaignIdByRetargetingId = EntryStream.of(retargetingsInfoMap)
                .mapValues(RetargetingCampaignInfo::getCampaignId)
                .toMap();

        Collection<Long> campaignIds = campaignIdByRetargetingId.values();
        Set<Long> visibleCampaigns = rbacService.getVisibleCampaigns(operatorUid, campaignIds);
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, campaignIds);

        ValidationResult<List<Long>, Defect> baseIdsDeleteValidation = idsToDeleteValidate(ids);

        return new ListValidationBuilder<>(baseIdsDeleteValidation)
                .checkEach(unique(), duplicatedRetargetingId())
                .checkEach(retargetingExistsAndVisible(campaignIdByRetargetingId, visibleCampaigns),
                        audienceTargetNotFound(), When.isValid())
                .checkEach(operatorHasRightsToChange(campaignIdByRetargetingId, writableCampaignIds),
                        noRightsCantWrite(), When.isValid())
                .checkEach(nonArchivedCampaignOnDelete(retargetingsInfoMap), When.isValid())
                .checkEach(deleteIsAllowed(retargetingsInfoMap), When.isValid())
                .getResult();
    }
}
