package ru.yandex.direct.core.entity.retargeting.service;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.AddedModelId;

import static java.util.stream.Collectors.collectingAndThen;
import static ru.yandex.direct.utils.FunctionalUtils.batchDispatch;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Добавляет клиенту retargeting conditions, но если у клиента уже есть retargeting condition с такими же как в запросе
 * правилами, то его не добавляет, а просто возвращает ID существующего. При этом, нет задачи сделать все записи
 * на клиенте строго уникальными. Если всё же, в результате гонок или добавления другими механизмами, у клиента
 * будет больше одного retargeting condition с одинаковыми правилами, то будет получаться ID с минимальным значением.
 */
@Service
public class FindOrCreateRetargetingConditionService {
    public static final Comparator<RetargetingCondition> RETARGETING_CONDITION_COMPARATOR =
            Comparator.comparing(RetargetingCondition::getType)
                    .thenComparing(RetargetingCondition::getRules,
                            Comparator.comparing(RetargetingConditionMappings::rulesToJson));

    private final RetargetingConditionRepository retargetingConditionRepository;

    public FindOrCreateRetargetingConditionService(RetargetingConditionRepository retargetingConditionRepository) {
        this.retargetingConditionRepository = retargetingConditionRepository;
    }

    public List<AddedModelId> findOrCreateRetargetingConditions(int shard, ClientId clientId,
                                                                List<RetargetingCondition> validRetargetingConditions) {
        var retargetingConditions = retargetingConditionRepository.getFromRetargetingConditionsTable(
                shard, clientId, LimitOffset.maxLimited());
        var existing = indexExistingRetConditions(retargetingConditions);

        return batchDispatch(validRetargetingConditions,
                existing::containsKey,
                retCondList -> mapList(retCondList, retCond -> AddedModelId.ofExisting(existing.get(retCond))),
                retCondList -> mapList(retargetingConditionRepository.add(shard, retCondList), AddedModelId::ofNew)
        );
    }

    private Map<RetargetingCondition, Long> indexExistingRetConditions(List<RetargetingCondition> retargetingConditions) {
        return StreamEx.of(retargetingConditions)
                .mapToEntry(Function.identity(), RetargetingConditionBase::getId)
                // если условия показа уже неуникальны, мы не расстроимся, а возмём старейший (по ID)
                .grouping(() -> new TreeMap<>(RETARGETING_CONDITION_COMPARATOR),
                        collectingAndThen(Collectors.minBy(Long::compareTo), Optional::orElseThrow));
    }
}
