package ru.yandex.direct.core.entity.retargeting.service;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalInterest;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleInterest;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.retargeting.Constants.INTEREST_LINK_TIME_VALUE;

/**
 * Фабрика {@link InterestLink} (абстракция по работе с категорями нацеливания). Позволяет создавать
 * {@link InterestLink} из {@link RetargetingCondition}, использую информацио о словарных категориях интересов
 * из {@link TargetingCategoriesCache}.
 */
@Component
public class InterestLinkFactory {
    private final TargetingCategoriesCache targetingCategoriesCache;

    public InterestLinkFactory(TargetingCategoriesCache targetingCategoriesCache) {
        this.targetingCategoriesCache = targetingCategoriesCache;
    }

    /**
     * Создаёт {@link InterestLink} с соответствующим {@link RetargetingCondition}
     *
     * @param interestId     ID категории нацеливания
     * @param clientId       clientId
     * @param lastChangeTime lastChangeTime
     * @return экземпляр {@link InterestLink}
     * @see ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesRepository
     */
    public InterestLink construct(Long interestId, ClientId clientId, LocalDateTime lastChangeTime) {
        Long goalId = targetingCategoriesCache.getImportIdByCategory(interestId).longValue();

        Goal goal = new Goal();
        goal.withId(goalId)
                .withTime(INTEREST_LINK_TIME_VALUE);
        Rule rule = new Rule();
        rule.withType(RuleType.ALL)
                .withGoals(Collections.singletonList(goal));
        RetargetingCondition retargetingCondition = new RetargetingCondition();
        retargetingCondition.withClientId(clientId.asLong())
                .withType(ConditionType.metrika_goals)
                .withName("")
                .withDescription("")
                .withDeleted(false)
                .withInterest(true)
                .withLastChangeTime(lastChangeTime)
                .withRules(Collections.singletonList(rule));

        return new InterestLink(retargetingCondition, interestId);
    }

    /**
     * Собираем InterestLink для интереса, c использованием GoalInterest и RuleInterest чтобы убрать лишние поля при
     * записи в БД. Используется для интересов РМП
     */
    InterestLink constructForMobileContent(Long interestId, ClientId clientId, LocalDateTime lastChangeTime) {
        BigInteger goalId = targetingCategoriesCache.getImportIdByCategory(interestId);
        checkState(goalId != null,
                "Could not find importId of the interest by category %s", interestId);

        GoalInterest goal = new GoalInterest(goalId.longValue(), INTEREST_LINK_TIME_VALUE);
        RuleInterest rule = new RuleInterest(RuleType.ALL, Collections.singletonList(goal));
        RetargetingCondition retargetingCondition = new RetargetingCondition();
        retargetingCondition.withClientId(clientId.asLong())
                .withType(ConditionType.metrika_goals)
                .withName("")
                .withDescription("")
                .withDeleted(false)
                .withInterest(true)
                .withLastChangeTime(lastChangeTime)
                .withRules(Collections.singletonList(rule));

        return new InterestLink(retargetingCondition, interestId);
    }

    /**
     * Создаёт {@link InterestLink} как обёртку над переданным {@link RetargetingCondition}
     *
     * @param retargetingCondition {@link RetargetingCondition}
     * @return экземпляр {@link InterestLink}
     */
    public InterestLink from(RetargetingCondition retargetingCondition) {
        List<Goal> goals = retargetingCondition.collectGoals();
        checkState(retargetingCondition.getInterest(), "Given RetargetingCondition has no INTEREST property");
        checkState(goals.size() == 1,
                "Given RetargetingCondition with id %s has wrong amount of goals (%s). Can't be interest",
                retargetingCondition.getId(),
                goals.size());
        Long interestId = targetingCategoriesCache.getCategoryByImportId(BigInteger.valueOf(goals.get(0).getId()));
        checkState(interestId != null,
                "Given RetargetingCondition with id %s has unknown INTEREST category (importId %s)",
                retargetingCondition.getId(),
                goals.get(0).getId());
        return new InterestLink(retargetingCondition, interestId);
    }
}
