package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingInternal;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.creator.GroupOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.convertRetargetingCondition;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.createTargetInterest;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class RetargetingConditionInternalService implements EntityService<RetargetingInternal, Long> {

    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;
    private final RetargetingConditionService retargetingConditionService;
    private final AddTargetInterestService addTargetInterestService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;
    private final AdGroupService adGroupService;
    private final RetargetingService retargetingService;

    @Autowired
    public RetargetingConditionInternalService(
            RetargetingConditionOperationFactory retargetingConditionOperationFactory,
            RetargetingConditionService retargetingConditionService,
            AddTargetInterestService addTargetInterestService,
            ClientService clientService,
            ShardHelper shardHelper,
            AdGroupService adGroupService,
            RetargetingService retargetingService
    ) {
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;
        this.retargetingConditionService = retargetingConditionService;
        this.addTargetInterestService = addTargetInterestService;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
        this.adGroupService = adGroupService;
        this.retargetingService = retargetingService;
    }

    @Override
    public List<RetargetingInternal> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        var retargetingByAdGroupIds = retargetingService.getRetargetingByAdGroupIds(ids, clientId);
        var retargetinConditionsByAdGroupIds =
                retargetingConditionService.getRetargetingConditionsByAdGroupIds(clientId, ids);
        var retargetinConditionMapById = StreamEx.of(retargetinConditionsByAdGroupIds)
                .mapToEntry(RetargetingCondition::getId, identity())
                .grouping();
        var retargetingConditionsByAdGroupId = StreamEx.of(retargetingByAdGroupIds)
                .mapToEntry(Retargeting::getAdGroupId,
                        r -> retargetinConditionMapById.get(r.getRetargetingConditionId()))
                .toMap();
        return StreamEx.of(ids)
                .map(id -> {
                    var retConds = StreamEx.of(retargetingConditionsByAdGroupId.getOrDefault(id, emptyList()))
                            .select(RetargetingConditionBase.class)
                            .toList();
                    return new RetargetingInternal()
                            .withId(id)
                            .withAdGroupId(id)
                            .withRetargetingConditions(retConds);
                })
                .filter(ri -> !ri.getRetargetingConditions().isEmpty())
                .toList();
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid,
                                List<RetargetingInternal> entities, Applicability applicability) {
        var findOrCreateRetargetingConditionsOpCreator = new GroupOperationCreator<>(
                (List<RetargetingCondition> retargetingConditions) ->
                        retargetingConditionOperationFactory.createFindOrCreateOperation(
                                applicability,
                                retargetingConditions,
                                clientId,
                                true));
        List<List<RetargetingCondition>> retargetingConditionGroups = mapList(entities,
                item -> getRetargetingConditions(clientId, item));
        var findOrCreateRetargetingConditionsOp =
                findOrCreateRetargetingConditionsOpCreator.create(retargetingConditionGroups);

        MassResult<List<AddedModelId>> resultForRetConds =
                findOrCreateRetargetingConditionsOp.prepareAndApply();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        // Создаём ретаргетинги
        List<TargetInterest> targetInterests = createTargetInterests(clientId,
                mapList(entities, RetargetingInternal::getAdGroupId),
                resultForRetConds.getResult());
        var clientCurrency = clientService.getWorkCurrency(clientId);
        addTargetInterestService.addValidTargetInterests(shard, operatorUid, UidAndClientId.of(operatorUid, clientId),
                clientCurrency, targetInterests);
        return MassResult.emptyMassAction();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<RetargetingInternal> entities, Applicability applicability) {
        return add(copyContainer.getClientIdTo(), copyContainer.getOperatorUid(), entities, applicability);
    }

    private List<RetargetingCondition> getRetargetingConditions(ClientId clientId, RetargetingInternal item) {
        LocalDateTime now = LocalDateTime.now();
        List<RetargetingConditionBase> retargetingConditions = nvl(item.getRetargetingConditions(), emptyList());
        return mapList(retargetingConditions, rcBase -> convertRetargetingCondition(clientId, rcBase, now));
    }

    private List<TargetInterest> createTargetInterests(ClientId clientId, List<Long> adGroupIds,
                                                       List<Result<List<AddedModelId>>> retCondsResult) {
        var campaignByAdGroup = EntryStream.of(adGroupService.getSimpleAdGroups(clientId, adGroupIds))
                .mapValues(AdGroupSimple::getCampaignId)
                .nonNullKeys()
                .nonNullValues()
                .toMap();

        LocalDateTime now = LocalDateTime.now();

        return EntryStream.zip(adGroupIds, retCondsResult)
                .mapValues(Result::getResult)
                .flatMapValues(Collection::stream)
                .mapKeyValue((adGroupId, rcAddInfo) -> {
                    Long campaignId = checkNotNull(campaignByAdGroup.get(adGroupId));
                    Long retargetingConditionId = rcAddInfo.getId();
                    return createTargetInterest(campaignId, adGroupId, retargetingConditionId, now);
                })
                .toList();
    }
}
