package ru.yandex.direct.core.entity.retargeting.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.multipliers.repository.MultipliersRepository;
import ru.yandex.direct.core.entity.retargeting.container.ReplaceRetargetingConditionGoal;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionDomainFlags;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionGoal;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsRepository;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.entity.retargeting.service.helper.RetargetingConditionWithLalSegmentHelper;
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingConditionValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.DeleteRetargetingConditionValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.ReplaceGoalsInRetargetingValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.UpdateRetargetingConditionValidationService2;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.FunctionOperation;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.containsAny;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RET_CONDITIONS_PER_CLIENT;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionDomainFlags.ADGROUPS_LAST_CHANGE;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionDomainFlags.ADGROUPS_SYNC;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionDomainFlags.MULTIPLIERS_SYNC;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.mergeSuccessfulAndInvalidItems;
import static ru.yandex.direct.validation.result.ValidationResultUtils.splitValidationResult;

@Service
public class RetargetingConditionOperationFactory {
    private static final Logger logger = LoggerFactory.getLogger(RetargetingConditionService.class);

    private final RetargetingConditionRepository retConditionRepository;
    private final LalSegmentRepository lalSegmentRepository;
    private final RetargetingGoalsRepository retGoalsRepository;
    private final AdGroupRepository adGroupRepository;
    private final MultipliersRepository multipliersRepository;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final ShardHelper shardHelper;
    private final GoalUtilsService goalUtilsService;
    private final AddRetargetingConditionValidationService2 addValidationService;
    private final UpdateRetargetingConditionValidationService2 updateValidationService;
    private final DeleteRetargetingConditionValidationService2 deleteValidationService;
    private final ReplaceGoalsInRetargetingValidationService2 replaceGoalsInRetargetingValidationService;
    private final FeatureService featureService;
    private final FindOrCreateRetargetingConditionService findOrCreateRetargetingConditionService;
    private final RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper;

    @Autowired
    public RetargetingConditionOperationFactory(
            RetargetingConditionRepository retConditionRepository,
            LalSegmentRepository lalSegmentRepository,
            RetargetingGoalsRepository retGoalsRepository,
            AdGroupRepository adGroupRepository,
            MultipliersRepository multipliersRepository,
            BsResyncQueueRepository bsResyncQueueRepository,
            ShardHelper shardHelper, GoalUtilsService goalUtilsService,
            AddRetargetingConditionValidationService2 addValidationService,
            UpdateRetargetingConditionValidationService2 updateValidationService,
            DeleteRetargetingConditionValidationService2 deleteValidationService,
            ReplaceGoalsInRetargetingValidationService2 replaceGoalsInRetargetingValidationService,
            FeatureService featureService,
            FindOrCreateRetargetingConditionService findOrCreateRetargetingConditionService,
            RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper) {
        this.retConditionRepository = retConditionRepository;
        this.lalSegmentRepository = lalSegmentRepository;
        this.retGoalsRepository = retGoalsRepository;
        this.adGroupRepository = adGroupRepository;
        this.multipliersRepository = multipliersRepository;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.shardHelper = shardHelper;
        this.goalUtilsService = goalUtilsService;
        this.addValidationService = addValidationService;
        this.updateValidationService = updateValidationService;
        this.deleteValidationService = deleteValidationService;
        this.replaceGoalsInRetargetingValidationService = replaceGoalsInRetargetingValidationService;
        this.featureService = featureService;
        this.findOrCreateRetargetingConditionService = findOrCreateRetargetingConditionService;
        this.retConditionWithLalSegmentHelper = retConditionWithLalSegmentHelper;
    }

    /**
     * Создание операции добавления RetargetingConditions
     */
    public AddRetargetingConditionsOperation createAddOperation(Applicability applicability,
                                                                List<RetargetingCondition> retargetingConditions,
                                                                ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return new AddRetargetingConditionsOperation(applicability,
                retargetingConditions,
                retConditionRepository,
                retConditionWithLalSegmentHelper,
                lalSegmentRepository,
                addValidationService,
                clientId,
                shard);
    }

    public PartiallyApplicableOperation<AddedModelId> createFindOrCreateOperation(
            Applicability applicability, List<RetargetingCondition> retargetingConditions, ClientId clientId,
            boolean skipGoalExistenceCheck) {
        checkArgument(retargetingConditions.stream().allMatch(Objects::nonNull));
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new FunctionOperation<>(
                applicability,
                items -> splitValidationResult(items, addValidationService.validate(
                        items, clientId, skipGoalExistenceCheck)),
                items -> findOrCreateRetargetingConditionService.findOrCreateRetargetingConditions(
                        shard, clientId, items),
                retargetingConditions
        );
    }

    /**
     * Создание операции изменения RetargetingConditions
     */
    public RetargetingConditionsUpdateOperation createUpdateOperation(Applicability applicability,
                                                                      List<ModelChanges<RetargetingCondition>> changesList,
                                                                      ClientId clientId,
                                                                      boolean skipPixelValidation,
                                                                      boolean partOfComplexOperation) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return new RetargetingConditionsUpdateOperation(
                applicability,
                changesList,
                updateValidationService,
                retConditionRepository,
                retConditionWithLalSegmentHelper,
                lalSegmentRepository,
                this,
                skipPixelValidation,
                partOfComplexOperation,
                clientId,
                shard
        );
    }

    /**
     * Отфильтровываем {@link AppliedChanges} из {@code appliedChangesCollection} и затем извлекаем ID'шники элементов
     *
     * @param appliedChangesCollection {@link AppliedChanges}
     * @param filterRule               {@link Predicate} условие фильтрации элементов
     * @return {@link Set} с ID'шками
     */
    private static Set<Long> filterIdsByRule(Collection<AppliedChanges<RetargetingCondition>> appliedChangesCollection,
                                             Predicate<AppliedChanges<RetargetingCondition>> filterRule) {
        // filterList не используется, чтобы не создавать ненужную промежуточную коллекцию
        return appliedChangesCollection
                .stream()
                .filter(filterRule)
                .map(a -> a.getModel().getId())
                .collect(toSet());
    }

    /**
     * Обновление целей условия нацелевания
     *
     * @param retConds  условие нацелвания
     * @param visitFunc функция Goal -> Goal
     */
    private static void updateGoalsByFunction(RetargetingCondition retConds, Function<Goal, Goal> visitFunc) {
        retConds.getRules().forEach(rule -> rule.setGoals(mapList(rule.getGoals(), visitFunc)));
    }

    public MassResult<Long> deleteRetargetingConditions(List<Long> ids, ClientId clientId) {
        return deleteRetargetingConditions(ids, clientId, Applicability.PARTIAL);
    }

    public MassResult<Long> deleteRetargetingConditions(List<Long> ids, ClientId clientId,
                                                        Applicability applicability) {
        Objects.requireNonNull(ids, "ids");

        logger.trace("delete retargeting conditions");
        ValidationResult<List<Long>, Defect> massValidation = deleteValidationService.validate(ids, clientId);

        if (massValidation.hasErrors()) {
            logger.debug("can not delete retargeting conditions: selection criteria contains errors");
            return MassResult.brokenMassAction(ids, massValidation);
        }

        List<Long> validIds = getValidItems(massValidation);

        if (isFull(applicability) && validIds.size() != ids.size()) {
            logger.debug("can not delete retargeting conditions: there are invalid ids");
            return MassResult.brokenMassAction(ids, massValidation);
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Long> deletedRetConditions = retConditionRepository.delete(shard, clientId, validIds);
        List<Long> resultIds = mergeSuccessfulAndInvalidItems(massValidation, deletedRetConditions, rcId -> rcId);

        logger.trace("retargeting conditions deleted successfully");
        return MassResult.successfulMassAction(resultIds, massValidation);
    }

    public MassResult<Long> updateRetargetingConditions(
            ClientId clientId, List<ModelChanges<RetargetingCondition>> changesList) {
        return updateRetargetingConditions(clientId, changesList,
                new UpdateRetargetingConditionsConfig(Applicability.PARTIAL));
    }

    /**
     * Обновление RetargetingCondition
     *
     * @param clientId    -
     * @param changesList -
     */
    public MassResult<Long> updateRetargetingConditions(
            ClientId clientId, List<ModelChanges<RetargetingCondition>> changesList,
            UpdateRetargetingConditionsConfig config) {
        Objects.requireNonNull(changesList, "changesList");

        RetargetingConditionsUpdateOperation updateOperation =
                createUpdateOperation(config.getApplicability(), changesList, clientId, config.isSkipPixelValidation(),
                        false);

        Optional<MassResult<Long>> prepareResult = updateOperation.prepare();
        if (prepareResult.isPresent()) {
            return prepareResult.get();
        }

        if (!config.getValidationOnly()) {
            return updateOperation.apply();
        } else {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(emptyList()));
        }
    }

    public void retConditionPostUpdate(
            Collection<AppliedChanges<RetargetingCondition>> changes, int shard) {
        Map<RetargetingConditionDomainFlags, Set<Long>> flagsToIds = calculateFlags(changes);

        actualizeGoals(changes, shard);
        actualizeAdGroups(flagsToIds, shard);
        actualizeMultipliers(flagsToIds, shard);
    }

    /**
     * Замена id старых целей на новые
     *
     * @param clientId                         id клиента
     * @param replaceRetargetingConditionGoals List изменений id целей
     */
    public Result<List<ReplaceRetargetingConditionGoal>> replaceGoalsInRetargetings(
            ClientId clientId,
            List<ReplaceRetargetingConditionGoal> replaceRetargetingConditionGoals) {
        boolean skipGoalExistenceCheck = featureService.isEnabledForClientId(clientId,
                FeatureName.SKIP_GOAL_EXISTENCE_FOR_AGENCY);

        Set<Long> metrikaGoalsForUids = emptySet();
        if (!skipGoalExistenceCheck) {
            var goalIds = mapList(replaceRetargetingConditionGoals,
                    ReplaceRetargetingConditionGoal::getNewGoalId);
            metrikaGoalsForUids = goalUtilsService.getAvailableMetrikaGoalIds(clientId, goalIds);
        }


        ValidationResult<List<ReplaceRetargetingConditionGoal>, Defect> validation =
                replaceGoalsInRetargetingValidationService
                        .validate(replaceRetargetingConditionGoals, metrikaGoalsForUids, skipGoalExistenceCheck);
        if (validation.hasAnyErrors()) {
            logger.debug("can not replace goals in retargetings: there are operation-level errors");
            return Result.broken(validation);
        }

        Map<Long, Long> oldGoalIdToNewGoalId = StreamEx.of(replaceRetargetingConditionGoals)
                .mapToEntry(
                        ReplaceRetargetingConditionGoal::getOldGoalId,
                        ReplaceRetargetingConditionGoal::getNewGoalId
                )
                .toMap();

        int shard = shardHelper.getShardByClientId(clientId);

        Set<Long> oldGoalIds = listToSet(replaceRetargetingConditionGoals,
                ReplaceRetargetingConditionGoal::getOldGoalId);
        List<RetargetingCondition> allClientRetargetinConditions = retConditionRepository
                .getFromRetargetingConditionsTable(shard, clientId, limited(MAX_RET_CONDITIONS_PER_CLIENT));
        List<RetargetingCondition> retargetingConditionsForUpdate = filterList(allClientRetargetinConditions,
                x -> containsAny(mapList(x.collectGoals(), Goal::getId), oldGoalIds));

        // На этапе валидации проверяется, что все ids целей существуют и принадлежат пользов
        Function<Goal, Goal> substituteWithNewGoalId = goal -> {
            if (oldGoalIdToNewGoalId.containsKey(goal.getId())) {
                goal.setId(oldGoalIdToNewGoalId.get(goal.getId()));
            }
            return goal;
        };

        Collection<AppliedChanges<RetargetingCondition>> changes = StreamEx.of(retargetingConditionsForUpdate)
                .peek(retargetingCondition -> updateGoalsByFunction(retargetingCondition,
                        substituteWithNewGoalId))
                .map(retargetingCondition ->
                        retargetingConditionModelChanges(retargetingCondition.getId())
                                .process(
                                        retargetingCondition.getRules(),
                                        RetargetingCondition.RULES)
                                .applyTo(retargetingCondition)
                )
                .toList();

        actualizeGoals(changes, shard);
        return Result.successful(getValidItems(validation));
    }

    private static ModelChanges<RetargetingCondition> retargetingConditionModelChanges(long id) {
        return new ModelChanges<>(id, RetargetingCondition.class);
    }

    /**
     * Метод позволяет получить отношение флагов к тем элементам, изменения которых привели к взведению соотвествующих
     * флагов.
     *
     * @param appliedChangesCollection коллекция {@link AppliedChanges}, описывающих изменённое состояние объектов
     * @return отношение {@link RetargetingConditionDomainFlags} к набору ID'шников
     */
    private static Map<RetargetingConditionDomainFlags, Set<Long>> calculateFlags(
            Collection<AppliedChanges<RetargetingCondition>> appliedChangesCollection) {
        Map<RetargetingConditionDomainFlags, Set<Long>> flagsToIds = new HashMap<>();
        flagsToIds.put(ADGROUPS_SYNC,
                filterIdsByRule(appliedChangesCollection, a -> a.changed(RetargetingCondition.RULES)));
        flagsToIds.put(MULTIPLIERS_SYNC,
                filterIdsByRule(appliedChangesCollection, a -> a.changed(RetargetingCondition.RULES)));
        flagsToIds.put(ADGROUPS_LAST_CHANGE,
                filterIdsByRule(appliedChangesCollection, a -> !a.getActuallyChangedProps().isEmpty()));
        return flagsToIds;
    }

    // ленивая переотправка в БК групп, у которых обновились какие-то условия нацеливания
    private void actualizeMultipliers(Map<RetargetingConditionDomainFlags, Set<Long>> flagsToIds, int shard) {
        Set<Long> retCondIds = flagsToIds.get(MULTIPLIERS_SYNC);

        Collection<BsResyncItem> items =
                multipliersRepository.getAdgroupIdsByRetargeting(shard, retCondIds, BsResyncPriority.DEFAULT);
        bsResyncQueueRepository.addToResync(shard, items);
    }

    // переотправка в БК групп и performance-banner-ов, затертых обновлением условий нацеливания
    private void actualizeAdGroups(Map<RetargetingConditionDomainFlags, Set<Long>> flagsToIds, int shard) {

        Set<Long> retCondIdsForSync = flagsToIds.get(ADGROUPS_SYNC);
        Set<Long> retCondIdsForLastChange = flagsToIds.get(ADGROUPS_LAST_CHANGE);

        Set<Long> retCondIdsAll = new HashSet<>(retCondIdsForSync);
        retCondIdsAll.addAll(retCondIdsForLastChange);

        // retCondId -> AdGroupIds
        Map<Long, List<Long>> adgroupByRetCondIdMap = retConditionRepository.getAdGroupIds(shard, retCondIdsAll);

        actualizeAdGroupsUpdateLastChange(retCondIdsForLastChange, adgroupByRetCondIdMap, shard);

        actualizeAdGroupsUpdateStatusBsSynced(retCondIdsForSync, adgroupByRetCondIdMap, shard);
    }

    private void actualizeAdGroupsUpdateLastChange(Set<Long> retCondIdsForLastChange,
                                                   Map<Long, List<Long>> adgroupByRetCondIdMap, int shard) {
        Set<Long> adgroupsToLastChange = getAdGroupsIds(retCondIdsForLastChange, adgroupByRetCondIdMap);
        adGroupRepository.updateLastChange(shard, adgroupsToLastChange);
    }

    private void actualizeAdGroupsUpdateStatusBsSynced(Set<Long> retCondIdsForSync,
                                                       Map<Long, List<Long>> adgroupByRetCondIdMap, int shard) {
        Set<Long> adgroupsToSync = getAdGroupsIds(retCondIdsForSync, adgroupByRetCondIdMap);
        adGroupRepository.updateStatusBsSynced(shard, adgroupsToSync, StatusBsSynced.NO);
    }

    private static Set<Long> getAdGroupsIds(Set<Long> retCondIds, Map<Long, List<Long>> adgroupByRetCondIdMap) {
        Set<Long> adgroups = new HashSet<>();
        for (Map.Entry<Long, List<Long>> adgroupEntry : adgroupByRetCondIdMap.entrySet()) {
            if (retCondIds.contains(adgroupEntry.getKey())) {
                adgroups.addAll(adgroupEntry.getValue());
            }
        }
        return adgroups;
    }

    // обновление целей, затронутых изменение условий нацеливания
    private void actualizeGoals(Collection<AppliedChanges<RetargetingCondition>> appliedChangesSet, int shard) {

        Multimap<Long, Long> toDelete = HashMultimap.create();
        Multimap<Long, RetargetingConditionGoal> toInsert = HashMultimap.create();

        for (AppliedChanges<RetargetingCondition> appliedChanges : appliedChangesSet) {
            if (!appliedChanges.changed(RetargetingCondition.RULES)) {
                // Свойство не менялось у этого объекта, идём дальше
                continue;
            }

            List<Rule> oldRules = appliedChanges.getOldValue(RetargetingCondition.RULES);
            Set<RetargetingConditionGoal> oldGoals = oldRules == null ? emptySet() : oldRules.stream()
                    .flatMap(rule -> rule.getGoals().stream())
                    .collect(toSet());

            List<Rule> newRules =
                    checkNotNull(RetargetingCondition.RULES.get(appliedChanges.getModel()));
            Set<RetargetingConditionGoal> newGoals = newRules.stream()
                    .flatMap(rule -> rule.getGoals().stream())
                    .collect(toSet());

            Long rcId = appliedChanges.getModel().getId();
            newGoals.stream()
                    .filter(goal -> !oldGoals.contains(goal))
                    .forEach(goal -> toInsert.put(rcId, goal));
            oldGoals.stream()
                    .filter(goal -> !newGoals.contains(goal))
                    .forEach(goal -> toDelete.put(rcId, goal.getId()));
        }
        retGoalsRepository.delete(shard, toDelete);
        retGoalsRepository.add(shard, toInsert);
    }

}
