package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.copyentity.translations.RenameProcessor;
import ru.yandex.direct.core.entity.IdModFilter;
import ru.yandex.direct.core.entity.campaign.model.CampaignExperiment;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.metrika.service.MetrikaSegmentService;
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsService;
import ru.yandex.direct.core.entity.retargeting.converter.GoalConverter;
import ru.yandex.direct.core.entity.retargeting.converter.RuleConverter;
import ru.yandex.direct.core.entity.retargeting.model.ExperimentRetargetingConditions;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionGoal;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsPpcDictRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsRepository;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.entity.retargeting.service.helper.RetargetingConditionWithLalSegmentHelper;
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingConditionValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingConditionsValidator;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.request.RetargetingGoalGroup;
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_IDS_PER_GET;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RET_CONDITIONS_PER_CLIENT;
import static ru.yandex.direct.core.entity.retargeting.converter.ExperimentConverter.createRetargetingConditionForAbSegment;
import static ru.yandex.direct.core.entity.retargeting.converter.ExperimentConverter.createRetargetingConditionForAbSegmentStatistic;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.ALLOWED_CAMPAIGN_TYPES_FOR_SHORTCUTS;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.NOT_BOUNCE_SHORTCUT_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.SHORTCUT_DEFAULT_ID_BY_NAME;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.SHORTCUT_NAME_BY_DEFAULT_ID;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.extractLalSegmentIds;
import static ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService.toMetrikaGoalType;
import static ru.yandex.direct.metrika.client.model.response.RetargetingCondition.Type.AB_SEGMENT;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class RetargetingConditionService implements EntityService<RetargetingConditionBase, Long> {
    private static final Logger logger = LoggerFactory.getLogger(RetargetingConditionService.class);

    private final RetargetingConditionRepository retConditionRepository;
    private final RetargetingGoalsRepository retGoalsRepository;
    private final RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper;
    private final RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository;
    private final RetargetingConditionShortcutService retargetingConditionShortcutService;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final LalSegmentRepository lalSegmentRepository;
    private final AddRetargetingConditionValidationService2 addValidationService;
    private final GoalUtilsService goalUtilsService;
    private final MetrikaClient metrikaClient;
    private final MetrikaSegmentService metrikaSegmentService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final MetrikaCampaignRepository metrikaCampaignRepository;
    private final CampaignRepository campaignRepository;
    private final MobileGoalsService mobileGoalsService;
    private final FeatureService featureService;
    private final RenameProcessor renameProcessor;

    @Autowired
    public RetargetingConditionService(
            RetargetingConditionRepository retConditionRepository,
            RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper,
            RetargetingGoalsRepository retGoalsRepository,
            RetargetingGoalsPpcDictRepository retargetingGoalsPpcDictRepository,
            RetargetingConditionShortcutService retargetingConditionShortcutService,
            RbacService rbacService,
            ShardHelper shardHelper,
            CryptaSegmentRepository cryptaSegmentRepository,
            LalSegmentRepository lalSegmentRepository,
            AddRetargetingConditionValidationService2 addValidationService,
            MetrikaClient metrikaClient,
            MetrikaSegmentService metrikaSegmentService,
            CampMetrikaCountersService campMetrikaCountersService,
            MetrikaCampaignRepository metrikaCampaignRepository,
            CampaignRepository campaignRepository,
            GoalUtilsService goalUtilsService,
            MobileGoalsService mobileGoalsService,
            FeatureService featureService,
            RenameProcessor renameProcessor
    ) {
        this.retConditionRepository = retConditionRepository;
        this.retConditionWithLalSegmentHelper = retConditionWithLalSegmentHelper;
        this.retGoalsRepository = retGoalsRepository;
        this.retargetingGoalsPpcDictRepository = retargetingGoalsPpcDictRepository;
        this.retargetingConditionShortcutService = retargetingConditionShortcutService;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.lalSegmentRepository = lalSegmentRepository;
        this.addValidationService = addValidationService;
        this.metrikaClient = metrikaClient;
        this.metrikaSegmentService = metrikaSegmentService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.metrikaCampaignRepository = metrikaCampaignRepository;
        this.campaignRepository = campaignRepository;
        this.goalUtilsService = goalUtilsService;
        this.mobileGoalsService = mobileGoalsService;
        this.featureService = featureService;
        this.renameProcessor = renameProcessor;
    }

    /**
     * Обновление целей условия нацелевания
     *
     * @param retConds  условие нацелвания
     * @param visitFunc функция Goal -> Goal
     */
    private static void updateGoalsByFunction(RetargetingCondition retConds, Function<Goal, Goal> visitFunc) {
        retConds.getRules().forEach(rule -> rule.setGoals(mapList(rule.getGoals(), visitFunc)));
    }

    @Override
    public List<RetargetingConditionBase> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        var retConditions = getRetargetingConditions(clientId, ids, new LimitOffset(Integer.MAX_VALUE, 0));
        return StreamEx.of(retConditions).select(RetargetingConditionBase.class).toList();
    }

    public List<RetargetingCondition> getRetargetingConditions(ClientId clientId, @Nullable Collection<Long> ids,
                                                               LimitOffset limitOffset) {
        return getRetargetingConditions(clientId, ids, null, null, null, limitOffset);
    }

    public List<RetargetingCondition> getRetargetingConditions(ClientId clientId,
                                                               @Nullable Collection<Long> ids,
                                                               Collection<RetargetingConditionsRetargetingConditionsType> types,
                                                               LimitOffset limitOffset) {
        return getRetargetingConditions(clientId, ids, null, null, types, limitOffset);
    }

    public List<RetargetingCondition> getRetargetingConditionsByAdGroupIds(ClientId clientId,
                                                                           @Nullable Collection<Long> adGroupIds) {
        return getRetargetingConditions(clientId, null, adGroupIds, null, null, LimitOffset.maxLimited());
    }

    public List<RetargetingCondition> getRetargetingConditions(ClientId clientId,
                                                               @Nullable Collection<Long> ids,
                                                               @Nullable Collection<Long> adGroupIds,
                                                               @Nullable String filter,
                                                               @Nullable Collection<RetargetingConditionsRetargetingConditionsType> types,
                                                               LimitOffset limitOffset) {
        logger.debug("get retargeting conditions");
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return retConditionRepository
                .getFromRetargetingConditionsTable(shard, clientId, ids, adGroupIds, filter, types, limitOffset);
    }

    /**
     * Получение условий нацеливания с полной информацией о их целях.
     *
     * @param retargetingConditionIds ids условий нацелевания
     * @param clientId                id клиента
     * @param adGroupIds              ids групп
     * @param filter                  фильтр по наименованию
     * @param types                   типы условий нацеливания
     * @return условия нацеливания
     */
    public List<RetargetingCondition> getRetargetingConditionsWithFullGoals(
            List<Long> retargetingConditionIds, ClientId clientId, Collection<Long> adGroupIds, String filter,
            Collection<RetargetingConditionsRetargetingConditionsType> types) {
        List<RetargetingCondition> retargetingConditions = getRetargetingConditions(clientId, retargetingConditionIds,
                adGroupIds, filter, types, limited(MAX_IDS_PER_GET));

        var goalIds = StreamEx.of(retargetingConditions)
                .map(RetargetingCondition::collectGoalsSafe)
                .flatMap(Collection::stream)
                .map(GoalBase::getId)
                .nonNull()
                .toSet();
        Map<Long, Goal> goalsById = listToMap(getMetrikaGoalsForRetargetingByIds(clientId, goalIds), Goal::getId);
        goalsById.putAll(cryptaSegmentRepository.getAll());

        Function<Goal, Goal> substituteWithFullGoal = goal -> {
            Goal goalWithInfo = goalsById.get(goal.getId());
            checkNotNull(goalWithInfo, "Can't find Goal by id %d for clientId %s", goal.getId(),
                    clientId);
            goal
                    .withType(goalWithInfo.getType())
                    .withName(goalWithInfo.getName())
                    .withCounterName(goalWithInfo.getCounterName())
                    .withAllowToUse(goalWithInfo.getAllowToUse())
                    .withDomain(goalWithInfo.getDomain())
                    .withOwner(goalWithInfo.getOwner())
                    .withCounterId(goalWithInfo.getCounterId())
                    .withSubtype(goalWithInfo.getSubtype())
                    .withUploadingSourceId(goalWithInfo.getUploadingSourceId())
                    .withParentId(goalWithInfo.getParentId())
                    .withTankerDescriptionKey(goalWithInfo.getTankerDescriptionKey())
                    .withStatus(goalWithInfo.getStatus());
            return goal;
        };

        retargetingConditions.forEach(retargetingCondition ->
                updateGoalsByFunction(retargetingCondition, substituteWithFullGoal));

        return retargetingConditions;
    }

    /**
     * Получение всех доступных шорткатов условий ретаргетинга (без учета используемости, возвращаются дефолтные
     * значения).
     * <p>
     * Используется для отображения списка шорткатов без возможности редактирования, поэтому id условий (ret_cond_id)
     * возвращаются дефолтные, а условия ретаргетинга могут быть незаполненными (нужные цели создадутся при сохранении).
     *
     * @param clientId   id клиента
     * @param campaignId id кампании
     * @return {@link List} шорткатов
     */
    public List<RetargetingCondition> getTruncatedRetargetingConditionShortcuts(ClientId clientId,
                                                                                @Nullable Long campaignId) {
        if (campaignId == null) {
            return List.of();
        }

        int shard = shardHelper.getShardByClientId(clientId);
        var campaignType = campaignRepository.getCampaignsTypeMap(shard, List.of(campaignId)).get(campaignId);
        if (campaignType == null || !ALLOWED_CAMPAIGN_TYPES_FOR_SHORTCUTS.contains(campaignType)) {
            return List.of();
        }

        // не возвращаем доступные шорткаты, если пользователь достиг предела по числу сохраненных условий ретаргетинга;
        // случай очень редкий, на момент написания таких пользователей всего 1 - DIRECT-153161
        int existingConditionsSize = retConditionRepository.getExistingRetargetingConditionsCount(shard, clientId);
        int possibleShortcutsSize = RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS.size();
        if (existingConditionsSize + possibleShortcutsSize > MAX_RET_CONDITIONS_PER_CLIENT) {
            return List.of();
        }

        // проверяем наличие на кампании счетчиков, на которые у клиента есть права редактирования
        List<Long> counterIdsLong = campMetrikaCountersService.getCounterByCampaignIds(clientId, List.of(campaignId))
                .get(campaignId);
        // в counterIdsLong может вернуться null в случае, если счетчик на кампании "не найден"/удален
        List<Integer> counterIds = nvl(mapList(counterIdsLong, Long::intValue), List.of());
        var isNotBounceShortcutsAvailable = !isEmpty(counterIds)
                && metrikaSegmentService.checkAvailabilityOfPresetSegments(clientId, counterIds,
                MetrikaSegmentService.SegmentType.NOT_BOUNCE);

        // проверяем наличие на кампании целей стратегии или ключевых целей
        var campaignGoalIds = metrikaCampaignRepository.getStrategyOrMeaningfulGoalIdsByCampaignId(shard,
                Set.of(campaignId)).get(campaignId);
        // не все цели, сохраненные на кампании, могут быть доступны для создания шортката
        var availableCampaignGoals = getAvailableMetrikaGoalsForRetargeting(clientId, campaignGoalIds);
        var isCampaignGoalsAvailable = !isEmpty(availableCampaignGoals);

        // проверяем наличие или возможность создания еком-сегментов
        List<Integer> ecomCounterIds = getEcomCounterIds(clientId, counterIdsLong);
        Set<MetrikaSegmentService.SegmentType> availableEcomSegments = !isEmpty(ecomCounterIds)
                ? metrikaSegmentService.getAvailableEcomSegments(clientId, ecomCounterIds)
                : Set.of();
        var isEcomPurchaseSegmentsAvailable =
                availableEcomSegments.contains(MetrikaSegmentService.SegmentType.ECOM_PURCHASE);
        var isEcomAbandonedCartSegmentsAvailable =
                availableEcomSegments.contains(MetrikaSegmentService.SegmentType.ECOM_ABANDONED_CART);
        var isEcomViewedWithoutPurchaseSegmentsAvailable =
                availableEcomSegments.contains(MetrikaSegmentService.SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE);

        return retargetingConditionShortcutService.getAvailableTruncatedRetargetingConditionShortcuts(clientId,
                isNotBounceShortcutsAvailable, isCampaignGoalsAvailable, isEcomPurchaseSegmentsAvailable,
                isEcomAbandonedCartSegmentsAvailable, isEcomViewedWithoutPurchaseSegmentsAvailable);
    }

    /**
     * Возвращает только те счетчики, на которых включена опция ecommerce.
     *
     * @param clientId   id клиента
     * @param counterIds список id счетчиков
     * @return {@link List} id ecommerce счетчиков
     */
    private List<Integer> getEcomCounterIds(ClientId clientId, @Nullable List<Long> counterIds) {
        if (isEmpty(counterIds)) {
            return List.of();
        }
        var clientRepresentativesUids = rbacService.getClientRepresentativesUidsForGetMetrikaCounters(clientId);
        var filter = new UserCountersExtendedFilter().withCounterIds(counterIds);
        var userCountersExtended = metrikaClient.getUsersCountersNumExtended2(clientRepresentativesUids, filter)
                .getUsers();
        return StreamEx.of(userCountersExtended)
                .flatCollection(UserCountersExtended::getCounters)
                .filter(CounterInfoDirect::getEcommerce)
                .map(CounterInfoDirect::getId)
                .toList();
    }

    /**
     * Получение для пользователя всех целей для ретаргетинга.
     * <p>
     * Цели состоят из доступных целей, получаемых из метрики, и недоступных: которые удалены в метрике,
     * и получаются из METRIKA_GOALS в PPCDICT или из retargeting_conditions в PPC.
     * <p>
     * Одному рекламодателю может принадлежать не более 2000 условий подбора аудиторий.
     *
     * @param clientId id клиента
     * @return {@link List} целей
     */
    public List<Goal> getMetrikaGoalsForRetargeting(ClientId clientId) {
        List<Goal> availableMetrikaGoals = getAvailableMetrikaGoalsForRetargeting(clientId);
        return getMetrikaGoalsForRetargeting(clientId, availableMetrikaGoals);
    }

    /**
     * Получение для пользователя целей из метрики отфильтрованных по goalIds, а также других целей
     * <p>
     * Цели состоят из доступных целей (отфильтрованы по goalIds), получаемых из метрики, и недоступных:
     * которые удалены в метрике, и получаются из METRIKA_GOALS в PPCDICT или из retargeting_conditions в PPC.
     * <p>
     * Одному рекламодателю может принадлежать не более 2000 условий подбора аудиторий.
     *
     * @param clientId id клиента
     * @return {@link List} целей
     */
    public List<Goal> getMetrikaGoalsForRetargetingByIds(ClientId clientId, Collection<Long> goalIds) {
        List<Goal> availableMetrikaGoals = getAvailableMetrikaGoalsForRetargeting(clientId, goalIds);
        return getMetrikaGoalsForRetargeting(clientId, availableMetrikaGoals);
    }

    public List<Goal> getMetrikaGoalsForRetargeting(ClientId clientId, List<Goal> availableMetrikaGoals) {
        Set<Long> metrikaGoalsIds = StreamEx.of(availableMetrikaGoals).map(Goal::getId).toSet();

        //Цели, используемые в ретаргетингах пользователя
        List<Goal> allMetrikaRetargetingConditionGoals = getAllMetrikaRetargetingConditionGoals(clientId);

        Map<Long, Goal> usedGoalsDeletedFromMetrikaById = StreamEx.of(allMetrikaRetargetingConditionGoals)
                .mapToEntry(RetargetingConditionGoal::getId, Function.identity())
                .removeKeys(metrikaGoalsIds::contains)
                .toMap();

        //Цели, удаленные из метрики, сохранившиеся в PPCDICT
        List<Goal> goalsFromPpcDict =
                retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(usedGoalsDeletedFromMetrikaById.keySet())
                        .stream()
                        .map(GoalUtilsService::changeEcommerceGoalName)
                        .collect(toList());
        Set<Long> goalsFromPpcDictIds =
                StreamEx.of(goalsFromPpcDict).map(Goal::getId).toSet();

        //Цели, удаленные из метрики, не сохранившиеся в PPCDICT
        List<Goal> goalsWithoutAccessNotSavedInPpcDict =
                EntryStream.of(usedGoalsDeletedFromMetrikaById)
                        .removeKeys(goalsFromPpcDictIds::contains)
                        .values()
                        .toList();

        List<Goal> inAppMobileGoals = mobileGoalsService.getAllAvailableInAppMobileGoals(clientId);

        List<Goal> goalsWithoutAccess = new ArrayList<>();
        goalsWithoutAccess.addAll(goalsFromPpcDict);
        goalsWithoutAccess.addAll(goalsWithoutAccessNotSavedInPpcDict);
        //Цели, удаленные из метрики, нельзя использовать: allowToUse=false
        goalsWithoutAccess.forEach(goal -> goal.setAllowToUse(false));

        var allGoals = StreamEx.of(availableMetrikaGoals).append(goalsWithoutAccess)
                .append(inAppMobileGoals).distinct(Goal::getId).toList();

        return StreamEx.of(allGoals).append(getLalSegmentForGoals(allGoals)).toList();
    }

    private List<Goal> getLalSegmentForGoals(List<Goal> parents) {
        Map<Long, Goal> goalById = listToMap(parents, Goal::getId);
        List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByParentIds(goalById.keySet());

        return StreamEx.of(lalSegments)
                .peek(segment -> {
                    var parent = goalById.get(segment.getParentId());
                    segment
                            .withName(parent.getName()) // Имя такое же как у родителя
                            .withAllowToUse(parent.getAllowToUse())
                            .withStatus(parent.getStatus())
                            .withLalParentType(parent.getType()); // нужно только для модели GdGoalTruncated
                })
                .toList();
    }

    public List<Goal> getAvailableMetrikaGoalsForRetargeting(ClientId clientId,
                                                             GoalType goalType) {
        Collection<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUids(clientId);
        var type = toMetrikaGoalType(goalType);
        if (type.isEmpty()) {
            throw new IllegalStateException("cannot find metrika type for goalType " + goalType);
        }

        return StreamEx.ofValues(metrikaClient.getGoals(clientRepresentativesUids, type.get()).getUidToConditions())
                .flatMap(Collection::stream)
                .map(GoalConverter::fromMetrikaRetargetingCondition)
                .toList();
    }

    // потенциально может вернуть 1кк целей.
    // используйте метод getAvailableMetrikaGoalsForRetargeting с фильтрацией по goalIds
    @Deprecated
    public List<Goal> getAvailableMetrikaGoalsForRetargeting(ClientId clientId) {
        Collection<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUids(clientId);
        return StreamEx.ofValues(metrikaClient.getGoalsByUids(clientRepresentativesUids))
                .flatMap(Collection::stream)
                .map(GoalConverter::fromMetrikaRetargetingCondition)
                .toList();
    }

    /**
     * В рамках RestrictedCampaignsAddOperation лучше использовать "кэширующую" версию метода
     * {@link ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter#getGoalsByUid}
     */
    public List<Goal> getAvailableMetrikaGoalsForRetargeting(ClientId clientId, Collection<Long> goalIds) {
        if (isEmpty(goalIds)) {
            return new ArrayList<>();
        }
        return StreamEx.of(goalUtilsService.getAvailableMetrikaGoals(clientId, goalIds))
                .map(GoalConverter::fromMetrikaRetargetingCondition)
                .toList();
    }

    public List<Long> getAccessibleGoalIdsForAllShards(IdModFilter filter) {
        List<Long> goals = new ArrayList<>();
        shardHelper.forEachShard(shard -> goals.addAll(retGoalsRepository.getAccessibleGoalIds(shard, filter)));
        return goals;
    }

    /**
     * Прогнозируем число посетителей с указанными условием ретаргетинга. Округляем результат до десятков вниз
     */
    public Result<Long> estimateRetargetingCondition(ClientId clientId, RetargetingCondition retCondition) {
        //не забыть поправить DaasFieldsValidatorHelper.validateCollectionInterests при изменениях тут
        Map<Long, Goal> allCryptaGoals = cryptaSegmentRepository.getAll();

        var parentGoalIds = StreamEx.of(retCondition.collectGoalsSafe())
                .map(GoalBase::getParentId)
                .nonNull()
                .toSet();
        var goalIds = StreamEx.of(retCondition.collectGoalsSafe())
                .map(GoalBase::getId)
                .append(parentGoalIds)
                .nonNull()
                .toSet();
        Set<Long> metrikaGoalIds = goalUtilsService.getAvailableMetrikaGoalIds(clientId, goalIds);

        List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByParentIds(metrikaGoalIds);
        Map<Long, Set<Long>> mutuallyExclusiveGoals = goalUtilsService.getMutuallyExclusiveGoalsMap();

        boolean skipGoalExistence = featureService.isEnabledForClientId(clientId,
                FeatureName.SKIP_GOAL_EXISTENCE_FOR_AGENCY);

        ValidationResult<RetargetingCondition, Defect> validation = RetargetingConditionsValidator
                .retConditionsIsValid(metrikaGoalIds, lalSegments, allCryptaGoals, mutuallyExclusiveGoals,
                        skipGoalExistence)
                .validateRetConditionForEstimate(retCondition);

        if (validation.hasAnyErrors()) {
            logger.debug("can not estimate retargeting condition: there are operation-level errors");
            return Result.broken(validation);
        }
        List<RetargetingGoalGroup> metrikaGoalGroups =
                mapList(retCondition.getRules(), RuleConverter::toMetrikaGoalGroup);

        long estimateUsers = metrikaClient.estimateUsersByCondition(metrikaGoalGroups);

        return Result.successful(zeroingLastDigit(estimateUsers), validation);
    }

    private List<Goal> getAllMetrikaRetargetingConditionGoals(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<RetargetingCondition> retargetingConditions = retConditionRepository.getFromRetargetingConditionsTable(
                shard,
                clientId,
                limited(MAX_RET_CONDITIONS_PER_CLIENT));

        return StreamEx.of(retargetingConditions)
                .flatCollection(RetargetingCondition::collectGoals)
                .filter(goal -> goal.getType().isMetrika())
                //берем цели уникальные по id, а не по всем сохраняемым в таблице полям(id, type, time),
                //так-как время не является, непосредственно, частью цели, а type вычисляется по id
                .distinct(Goal::getId)
                .toList();
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid,
                                List<RetargetingConditionBase> entities, Applicability applicability) {
        var retConditions = StreamEx.of(entities).select(RetargetingCondition.class).toList();
        return addRetargetingConditions(retConditions, clientId, applicability, false);
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<RetargetingConditionBase> entities, Applicability applicability) {
        var retConditions = StreamEx.of(entities).select(RetargetingCondition.class).toList();
        Objects.requireNonNull(retConditions, "retConditions");
        return addRetargetingConditionsWithMergeExisting(
                retConditions, copyContainer.getShardTo(), copyContainer.getClientIdTo(), copyContainer.getLocale(),
                applicability);
    }

    public MassResult<Long> addRetargetingConditions(
            List<RetargetingCondition> retConditionList, ClientId clientId) {
        return addRetargetingConditions(retConditionList, clientId, Applicability.PARTIAL, false);
    }

    /**
     * Добавление условий ретаргетинга
     *
     * @param applyValidationOnly - true если необходимо провести только валидацию без изменения в БД
     */
    public MassResult<Long> addRetargetingConditions(List<RetargetingCondition> retConditionList, ClientId clientId,
                                                     Applicability applicability, boolean applyValidationOnly) {
        Objects.requireNonNull(retConditionList, "retConditionList");

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return addRetargetingConditions(retConditionList, shard, clientId, applicability, applyValidationOnly);
    }

    /**
     * Добавление условий ретаргетинга
     *
     * @param applyValidationOnly - true если необходимо провести только валидацию без изменения в БД
     */
    private MassResult<Long> addRetargetingConditions(
            List<RetargetingCondition> retConditionList, int shard, ClientId clientId,
            Applicability applicability, boolean applyValidationOnly) {
        AddRetargetingConditionsOperation addRetConditionsOperation =
                new AddRetargetingConditionsOperation(applicability,
                        retConditionList,
                        retConditionRepository,
                        retConditionWithLalSegmentHelper,
                        lalSegmentRepository,
                        addValidationService,
                        clientId,
                        shard);

        return addRetargetingConditions(addRetConditionsOperation, applyValidationOnly);
    }

    /**
     * Добавление условий ретаргетинга со слиянием уже существующих ретаргетингов.
     */
    private MassResult<Long> addRetargetingConditionsWithMergeExisting(
            List<RetargetingCondition> retConditionList, int shard, ClientId clientId, Locale locale,
            Applicability applicability) {
        AddRetargetingConditionsOperation addRetConditionsOperation =
                new AddRetargetingConditionsOperation(applicability,
                        retConditionList,
                        retConditionRepository,
                        retConditionWithLalSegmentHelper,
                        lalSegmentRepository,
                        addValidationService,
                        renameProcessor,
                        locale,
                        clientId,
                        shard,
                        true,
                        true);

        return addRetargetingConditions(addRetConditionsOperation, false);
    }

    /**
     * Добавление условий ретаргетинга
     *
     * @param applyValidationOnly - true если необходимо провести только валидацию без изменения в БД
     */
    private MassResult<Long> addRetargetingConditions(
            AddRetargetingConditionsOperation addRetConditionsOperation, boolean applyValidationOnly) {

        Optional<MassResult<Long>> prepareResult = addRetConditionsOperation.prepare();
        if (prepareResult.isPresent()) {
            return prepareResult.get();
        }

        if (!applyValidationOnly) {
            return addRetConditionsOperation.apply();
        } else {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(emptyList()));
        }
    }

    /**
     * Поиск/создание условий ретаргетинга по id шорткатов
     *
     * @param shard            номер шарда
     * @param clientId         id клиента
     * @param shortcutIdsByCid Список дефолтных id шорткатов, разбитый по кампаниям
     * @return для каждой кампании из входных данных возвращается словарь:
     * дефолтный id шортката -> id существующего/созданного условия
     */
    public Map<Long, Map<Long, Long>> findOrCreateRetargetingConditionShortcuts(
            int shard, ClientId clientId, @Nullable Map<Long, List<Long>> shortcutIdsByCid) {
        if (shortcutIdsByCid == null || shortcutIdsByCid.isEmpty()) {
            return Map.of();
        }

        // поиск сохраненных шорткатов
        List<RetargetingCondition> allExistingShortcuts = getRetargetingConditions(clientId, null,
                Set.of(RetargetingConditionsRetargetingConditionsType.shortcuts), LimitOffset.maxLimited());
        var allExistingShortcutIds = mapList(allExistingShortcuts, RetargetingCondition::getId);
        Map<Long, List<Long>> existingShortcutIdsByCampaignId =
                retConditionRepository.getRetConditionIdsByCampaignId(shard, allExistingShortcutIds);
        var existingShortcutsById = listToMap(allExistingShortcuts, RetargetingCondition::getId);
        Map<Long, List<RetargetingCondition>> existingShortcutsByCampaignId =
                EntryStream.of(existingShortcutIdsByCampaignId)
                        .mapValues(ids -> mapList(ids, existingShortcutsById::get))
                        .toMap();

        // получаем цели стратегии или ключевые цели кампаний
        var campaignIds = shortcutIdsByCid.keySet();
        Map<Long, Set<Long>> campaignGoalIdsByCampaignId = metrikaCampaignRepository
                .getStrategyOrMeaningfulGoalIdsByCampaignId(shard, campaignIds);
        // не все цели, сохраненные на кампании, могут быть доступны для создания шорткатов
        var allCampaignGoalIds = flatMapToSet(campaignGoalIdsByCampaignId.values(), Function.identity());
        var allAvailableCampaignGoals = getAvailableMetrikaGoalsForRetargeting(clientId, allCampaignGoalIds);
        var allAvailableCampaignGoalIds = mapList(allAvailableCampaignGoals, GoalBase::getId);

        var campaignTypeById = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        // поиск или создание шорткатов для каждой из кампаний
        Map<Long, Map<Long, Long>> result = new HashMap<>(shortcutIdsByCid.size());
        shortcutIdsByCid.forEach((campaignId, shortcutIds) -> {
            var campaignType = campaignTypeById.get(campaignId);
            if (campaignType == null || !ALLOWED_CAMPAIGN_TYPES_FOR_SHORTCUTS.contains(campaignType)) {
                return;
            }

            var existingShortcuts = existingShortcutsByCampaignId.getOrDefault(campaignId, List.of());
            Map<String, Long> existingShortcutIdByName = StreamEx.of(existingShortcuts)
                    .mapToEntry(RetargetingCondition::getName, RetargetingCondition::getId)
                    // у кампании должно быть не больше одного сохраненного условия на каждый шорткат
                    .distinctKeys()
                    .toMap();
            var shortcutDefaultIdsNotToCreate = StreamEx.ofKeys(existingShortcutIdByName)
                    .map(SHORTCUT_DEFAULT_ID_BY_NAME::get)
                    .nonNull()
                    .toSet();
            var shortcutIdsToCreate = filterList(shortcutIds, id -> !shortcutDefaultIdsNotToCreate.contains(id));

            var campaignGoalIds = campaignGoalIdsByCampaignId.get(campaignId);
            var availableCampaignGoalIds = filterToSet(campaignGoalIds, allAvailableCampaignGoalIds::contains);
            // создание шорткатов для сохранения
            Map<Long, RetargetingCondition> shortcutByDefaultId = getRetargetingConditionShortcuts(clientId,
                    campaignId, availableCampaignGoalIds, shortcutIdsToCreate);

            var shortcutsToAdd = new ArrayList<>(shortcutByDefaultId.values());
            // отдельно сохраняем условия для каждой кампании, поскольку созданные условия можно различать только по
            // имени, а имена уникальны в рамках одной кампании
            retConditionRepository.add(shard, shortcutsToAdd);
            lalSegmentRepository.activateLalSegments(extractLalSegmentIds(shortcutsToAdd));

            // новые id уже добавлены у shortcutsToAdd
            var addedRetargetingConditionIdByName = listToMap(shortcutsToAdd, RetargetingCondition::getName,
                    RetargetingCondition::getId);

            var addedIdByDefaultId = listToMap(shortcutIds, defaultId -> defaultId, defaultId -> {
                var shortcutName = SHORTCUT_NAME_BY_DEFAULT_ID.get(defaultId);
                if (existingShortcutIdByName.containsKey(shortcutName)) {
                    return existingShortcutIdByName.get(shortcutName);
                } else {
                    // если шорткат должен был создаться, но не создался по какой-то причине, то возвращаем исходный id
                    return addedRetargetingConditionIdByName.getOrDefault(shortcutName, defaultId);
                }
            });
            result.put(campaignId, addedIdByDefaultId);
        });

        return result;
    }

    private Map<Long, RetargetingCondition> getRetargetingConditionShortcuts(ClientId clientId, Long campaignId,
                                                                             @Nullable Set<Long> campaignGoalIds,
                                                                             List<Long> shortcutIds) {
        Set<Long> uniqueShortcutIds = listToSet(shortcutIds); // для более быстрой contains проверки

        // получить счетчики кампании, если есть шорткаты, которым они требуются
        List<Long> counterIdsLong = null;
        List<Integer> counterIds = List.of();
        if (uniqueShortcutIds.contains(NOT_BOUNCE_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID)) {
            counterIdsLong = campMetrikaCountersService.getCounterByCampaignIds(clientId,
                    List.of(campaignId)).get(campaignId);
            // в counterIdsLong может вернуться null в случае, если счетчик на кампании "не найден"/удален
            counterIds = nvl(mapList(counterIdsLong, Long::intValue), List.of());
        }

        // если требуются "неотказные" шорткаты, то нужно достать существующие и создать недостающие сегменты
        List<Goal> notBounceSegments = null;
        List<Goal> notBounceLalSegments = null;
        if (uniqueShortcutIds.contains(NOT_BOUNCE_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID)) {
            Set<Long> notBounceSegmentIds;
            try {
                notBounceSegmentIds = metrikaSegmentService.getOrCreateNotBounceSegmentIds(clientId, counterIds);
            } catch (MetrikaClientException | InterruptedRuntimeException e) {
                logger.error("Got an exception when querying for metrika segments for client " + clientId
                        + " and counters " + counterIds, e);
                notBounceSegmentIds = Set.of();
            }

            if (uniqueShortcutIds.contains(NOT_BOUNCE_SHORTCUT_DEFAULT_ID)) {
                var time = retargetingConditionShortcutService.calculateGoalTime(NOT_BOUNCE_SHORTCUT_DEFAULT_ID);
                notBounceSegments = mapList(notBounceSegmentIds, id -> composeRetargetingGoal(id, time));
            }

            if (uniqueShortcutIds.contains(NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID)) {
                var time = retargetingConditionShortcutService.calculateGoalTime(NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID);
                notBounceLalSegments = mapList(findOrCreateLalRetargetingGoals(notBounceSegmentIds),
                        segment -> composeRetargetingGoal(segment.getId(), time));
            }
        }

        List<Goal> campaignGoals = null;
        List<Goal> campaignLalGoals = null;
        if (!isEmpty(campaignGoalIds)) {
            if (uniqueShortcutIds.contains(CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID)) {
                var time = retargetingConditionShortcutService.calculateGoalTime(CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID);
                campaignGoals = mapList(campaignGoalIds, id -> composeRetargetingGoal(id, time));
            }

            if (uniqueShortcutIds.contains(CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID)) {
                var time =
                        retargetingConditionShortcutService.calculateGoalTime(CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID);
                campaignLalGoals = mapList(findOrCreateLalRetargetingGoals(campaignGoalIds),
                        goal -> composeRetargetingGoal(goal.getId(), time));
            }
        }

        List<Goal> ecomPurchaseLalSegments = null;
        List<Goal> ecomAbandonedCartSegments = null;
        List<Goal> ecomViewedWithoutPurchaseSegments = null;
        if (uniqueShortcutIds.contains(ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID)
                || uniqueShortcutIds.contains(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID)) {

            List<Integer> ecomCounterIds = getEcomCounterIds(clientId, counterIdsLong);
            if (!isEmpty(ecomCounterIds)) {
                Set<MetrikaSegmentService.SegmentType> segmentTypesToGet = new HashSet<>();
                if (uniqueShortcutIds.contains(ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID)) {
                    segmentTypesToGet.add(MetrikaSegmentService.SegmentType.ECOM_PURCHASE);
                }
                if (uniqueShortcutIds.contains(ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID)) {
                    segmentTypesToGet.add(MetrikaSegmentService.SegmentType.ECOM_ABANDONED_CART);
                }
                if (uniqueShortcutIds.contains(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID)) {
                    segmentTypesToGet.add(MetrikaSegmentService.SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE);
                }

                // если требуются шорткаты по еком-сегментам, то нужно достать существующие и создать недостающие
                // (аналогично "неотказным" шорткатам)
                Map<MetrikaSegmentService.SegmentType, Set<Long>> ecomSegmentIdsBySegmentType;
                try {
                    ecomSegmentIdsBySegmentType = metrikaSegmentService
                            .getOrCreateSegmentIdsBySegmentType(clientId, ecomCounterIds, segmentTypesToGet);
                } catch (MetrikaClientException | InterruptedRuntimeException e) {
                    logger.error("Got an exception when querying for metrika segments for client " + clientId
                            + " and counters " + ecomCounterIds, e);
                    ecomSegmentIdsBySegmentType = Map.of();
                }

                if (ecomSegmentIdsBySegmentType.containsKey(MetrikaSegmentService.SegmentType.ECOM_PURCHASE)) {
                    var time = retargetingConditionShortcutService
                            .calculateGoalTime(ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID);
                    var segmentIds = ecomSegmentIdsBySegmentType.get(MetrikaSegmentService.SegmentType.ECOM_PURCHASE);
                    ecomPurchaseLalSegments = mapList(findOrCreateLalRetargetingGoals(segmentIds),
                            goal -> composeRetargetingGoal(goal.getId(), time));
                }

                if (ecomSegmentIdsBySegmentType.containsKey(MetrikaSegmentService.SegmentType.ECOM_ABANDONED_CART)) {
                    var time = retargetingConditionShortcutService
                            .calculateGoalTime(ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID);
                    var segmentIds = ecomSegmentIdsBySegmentType.get(
                            MetrikaSegmentService.SegmentType.ECOM_ABANDONED_CART);
                    ecomAbandonedCartSegments = mapList(segmentIds, id -> composeRetargetingGoal(id, time));
                }

                if (ecomSegmentIdsBySegmentType.containsKey(
                        MetrikaSegmentService.SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE)) {
                    var time = retargetingConditionShortcutService
                            .calculateGoalTime(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID);
                    var segmentIds = ecomSegmentIdsBySegmentType.get(
                            MetrikaSegmentService.SegmentType.ECOM_VIEWED_WITHOUT_PURCHASE);
                    ecomViewedWithoutPurchaseSegments = mapList(segmentIds, id -> composeRetargetingGoal(id, time));
                }
            }
        }

        return retargetingConditionShortcutService.getRetargetingConditionShortcuts(clientId,
                notBounceSegments, notBounceLalSegments,
                campaignGoals, campaignLalGoals,
                ecomPurchaseLalSegments, ecomAbandonedCartSegments, ecomViewedWithoutPurchaseSegments);
    }

    private List<Goal> findOrCreateLalRetargetingGoals(Set<Long> baseGoalIds) {
        if (baseGoalIds.isEmpty()) {
            return List.of();
        }

        List<Goal> existingLalSegments = lalSegmentRepository.getLalSegmentsByParentIds(baseGoalIds);
        Set<Long> goalIdsWithLalSegment = listToSet(existingLalSegments, Goal::getParentId);

        List<Long> goalIdsWithoutLalSegment = StreamEx.of(baseGoalIds)
                .remove(goalIdsWithLalSegment::contains)
                .toList();
        List<Goal> createdLalSegments = lalSegmentRepository.createLalSegments(goalIdsWithoutLalSegment);

        existingLalSegments.addAll(createdLalSegments);
        return existingLalSegments;
    }

    private static Goal composeRetargetingGoal(Long id, Integer time) {
        return (Goal) new Goal()
                .withId(id)
                .withTime(time);
    }

    /**
     * Работает не частично.
     * Создаем условия ретаргетингов для экспериментов.
     * Если подходящие условия уже есть отдаем существующие.
     *
     * @deprecated следует использовать метод с заранее полученными целями из метрики.
     * после выполнения тикета <a href="DIRECT-112091">DIRECT-112091</a>, необходимости в этом методе не будет.
     */
    @Deprecated
    public List<ExperimentRetargetingConditions> findOrCreateExperimentsRetargetingConditions(
            ClientId clientId,
            List<CampaignExperiment> campaignExperiments) {
        var uids = rbacService.getClientRepresentativesUids(clientId);

        var goalsByUids = metrikaClient.getGoals(uids,
                        ru.yandex.direct.metrika.client.model.request.GoalType.AB_SEGMENT)
                .getUidToConditions();

        return findOrCreateExperimentsRetargetingConditions(clientId, campaignExperiments, goalsByUids, null);
    }

    /**
     * Работает не частично.
     * Создаем условия ретаргетингов для экспериментов.
     * Если подходящие условия уже есть отдаем существующие.
     */
    public List<ExperimentRetargetingConditions> findOrCreateExperimentsRetargetingConditions(
            ClientId clientId,
            List<CampaignExperiment> campaignExperiments,
            Map<Long, List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids,
            @Nullable Map<Long, List<Long>> preparedExperimentsBySegments) {

        preparedExperimentsBySegments = nvl(preparedExperimentsBySegments, Collections.emptyMap());

        List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> clientAbSegments =
                getClientAbSegments(goalsByUids);

        var clientAbSegmentIdsBySectionId = StreamEx.of(clientAbSegments)
                .mapToEntry(ru.yandex.direct.metrika.client.model.response.RetargetingCondition::getSectionId,
                        ru.yandex.direct.metrika.client.model.response.RetargetingCondition::getId)
                .filterKeys(Objects::nonNull)
                .grouping();
        var allAbSegmentIdsBySectionId = EntryStream.of(clientAbSegmentIdsBySectionId)
                .append(preparedExperimentsBySegments)
                .toMap((a, b) -> StreamEx.of(a).append(b).distinct().toList());

        LocalDateTime now = LocalDateTime.now();

        List<RetargetingCondition> retargetingConditions = mapList(campaignExperiments,
                item -> createRetargetingConditionForAbSegment(now, clientId,
                        item.getAbSegmentGoalIds(), allAbSegmentIdsBySectionId));

        List<RetargetingCondition> statisticRetargetingConditions = mapList(campaignExperiments,
                item -> createRetargetingConditionForAbSegmentStatistic(now, clientId,
                        item.getSectionIds(), allAbSegmentIdsBySectionId));

        List<Long> statisticRetargetingConditionsIds = findOrCreateRetargetingConditionsForExperiment(clientId,
                statisticRetargetingConditions);

        List<Long> retargetingConditionsIds = findOrCreateRetargetingConditionsForExperiment(clientId,
                retargetingConditions);

        return EntryStream.of(campaignExperiments)
                .keys()
                .map(index -> new ExperimentRetargetingConditions()
                        .withRetargetingConditionId(retargetingConditionsIds.get(index))
                        .withStatisticRetargetingConditionId(statisticRetargetingConditionsIds.get(index))
                )
                .toList();
    }

    private List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> getClientAbSegments(Map<Long,
            List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> goalsByUids) {
        return EntryStream.of(goalsByUids)
                .values()
                .flatMap(Collection::stream)
                .filter(x -> x.getType() == AB_SEGMENT)
                .toList();
    }

    /**
     * Создание условий ретаргетингов для экспериментов, если уже существуют, возвращаем существующие
     */
    private List<Long> findOrCreateRetargetingConditionsForExperiment(
            ClientId clientId,
            List<RetargetingCondition> retargetingConditions) {
        List<RetargetingCondition> clientsRetargetingConditions =
                getRetargetingConditions(clientId, null,
                        Set.of(RetargetingConditionsRetargetingConditionsType.ab_segments), LimitOffset.maxLimited());

        Map<Integer, Long> existsRetargetingConditionIdByIndex = tryToGetExistsConditions(clientsRetargetingConditions,
                retargetingConditions);

        List<RetargetingCondition> retargetingConditionToAdd = EntryStream.of(retargetingConditions)
                .removeKeys(existsRetargetingConditionIdByIndex::containsKey)
                .values()
                .nonNull()
                .distinct()
                .toList();

        Map<RetargetingCondition, List<Integer>> retargetingConditionToIndex = EntryStream.of(retargetingConditions)
                .invert()
                .nonNullKeys()
                .grouping();

        int shard = shardHelper.getShardByClientId(clientId);

        Map<Integer, Integer> indexOfAddedConditionByInputIndex = EntryStream.of(retargetingConditionToAdd)
                .mapValues(retargetingConditionToIndex::get)
                .flatMapValues(Collection::stream)
                .invert()
                .toMap();

        List<Long> addedRetargetingConditionIds = retConditionRepository.add(shard, retargetingConditionToAdd);

        return EntryStream.of(retargetingConditions)
                .keys()
                .map(index -> getRetargetingConditionByIndex(existsRetargetingConditionIdByIndex,
                        indexOfAddedConditionByInputIndex, addedRetargetingConditionIds, index))
                .toList();
    }

    private Map<Integer, Long> tryToGetExistsConditions(
            List<RetargetingCondition> clientsRetargetingConditions,
            List<RetargetingCondition> retargetingConditions) {

        return EntryStream.of(retargetingConditions)
                .nonNullValues()
                .mapValues(retargetingCondition -> findRetargetingConditionInExisting(clientsRetargetingConditions,
                        retargetingCondition))
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();
    }

    private Optional<Long> findRetargetingConditionInExisting(List<RetargetingCondition> clientConditions,
                                                              RetargetingCondition condition) {
        Comparator<RetargetingCondition> retargetingConditionComparator =
                getRetargetingConditionComparator();
        return StreamEx.of(clientConditions)
                .filter(x -> retargetingConditionComparator.compare(x, condition) == 0)
                .findFirst()
                .map(RetargetingConditionBase::getId);
    }

    private Comparator<RetargetingCondition> getRetargetingConditionComparator() {
        return Comparator.comparing(RetargetingCondition::getClientId)
                .thenComparing(RetargetingCondition::getType)
                .thenComparing(retargetingCondition ->
                        RetargetingConditionMappings.rulesToJson(retargetingCondition.getRules()));
    }

    private Long getRetargetingConditionByIndex(Map<Integer, Long> existsRetargetingConditionIdByIndex, Map<Integer,
            Integer> indexOfAddedConditionByInputIndex, List<Long> addedRetargetingConditionIds, Integer index) {
        return existsRetargetingConditionIdByIndex.get(index) != null ?
                existsRetargetingConditionIdByIndex.get(index) :
                ifNotNull(indexOfAddedConditionByInputIndex.get(index),
                        addedRetargetingConditionIds::get);
    }

    public Map<Long, List<Long>> getCampaignIds(ClientId clientId, List<Long> retargetingConditionIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return retConditionRepository.getCampaignIds(shard, retargetingConditionIds);
    }

    /**
     * Обнуление последнего знака числа
     */
    private long zeroingLastDigit(long number) {
        return (number / 10) * 10;
    }
}
