package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.common.db.PpcPropertyNames.RETARGETING_CONDITION_CAMPAIGN_GOALS_LAL_SHORTCUT_GOAL_TIME_DAYS;
import static ru.yandex.direct.common.db.PpcPropertyNames.RETARGETING_CONDITION_CAMPAIGN_GOALS_SHORTCUT_GOAL_TIME_DAYS;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOALS_PER_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOAL_TIME;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_GOAL_TIME;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

@Service
public class RetargetingConditionShortcutService {
    // id для шорткатов взяты из gap-а в shard_inc_ret_cond_id, подробнее в DIRECT-150327.
    // Свободны ret_cond_id с 11746 по 41743.

    // Менять порядок выдачи шорткатов можно в методе getAvailableTruncatedRetargetingConditionShortcuts.
    // При смене имени шортката нужно добавить устаревшее имя в словарь OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME.

    public static final Set<CampaignType> ALLOWED_CAMPAIGN_TYPES_FOR_SHORTCUTS = Set.of(CampaignType.TEXT);

    // Шорткаты на сегмент Метрики "Неотказы"
    public static final Long NOT_BOUNCE_SHORTCUT_DEFAULT_ID = 12000L;
    public static final String NOT_BOUNCE_SHORTCUT_NAME = "Посетители";
    public static final String NOT_BOUNCE_SHORTCUT_DESCRIPTION = "Пользователи вашего сайта";

    public static final Long NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID = 12001L;
    public static final String NOT_BOUNCE_LAL_SHORTCUT_NAME = "Похожие на посетителей";
    public static final String NOT_BOUNCE_LAL_SHORTCUT_DESCRIPTION =
            "Пользователи, похожие на посетителей вашего сайта";

    // Шорткаты на цели кампании
    public static final Long CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID = 12002L;
    public static final String CAMPAIGN_GOALS_SHORTCUT_NAME_OBSOLETE = "Пользователи, достигнувшие целей кампании";
    public static final String CAMPAIGN_GOALS_SHORTCUT_NAME = "Достигли целей кампании";
    private static final String CAMPAIGN_GOALS_SHORTCUT_DESCRIPTION =
            "Пользователи сайта, достигавшие целей из настроек кампании";

    public static final Long CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID = 12003L;
    public static final String CAMPAIGN_GOALS_LAL_SHORTCUT_NAME_OBSOLETE =
            "Похожие на пользователей, достигнувших целей кампании";
    static final String CAMPAIGN_GOALS_LAL_SHORTCUT_NAME = "Похожие на достигнувших целей кампании";
    private static final String CAMPAIGN_GOALS_LAL_SHORTCUT_DESCRIPTION =
            "Пользователи, похожие на достигнувших целей из настроек кампании";

    // Шорткаты на цели электронной коммерции
    public static final Long ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID = 12004L;
    static final String ECOM_PURCHASE_LAL_SHORTCUT_NAME = "Похожие на покупателей";
    private static final String ECOM_PURCHASE_LAL_SHORTCUT_DESCRIPTION =
            "Пользователи, похожие на посетителей сайта, совершивших покупки";

    public static final Long ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID = 12005L;
    static final String ECOM_ABANDONED_CART_SHORTCUT_NAME = "Брошенные корзины";
    private static final String ECOM_ABANDONED_CART_SHORTCUT_DESCRIPTION =
            "Пользователи, добавившие товары в корзину, но не оформившие заказ";

    public static final Long ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID = 12006L;
    static final String ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_NAME = "Смотрели товары, но не купили";
    private static final String ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DESCRIPTION =
            "Пользователи, просматривавшие товары на вашем сайте, но не оформившие покупку";

    public static final List<Long> RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS =
            List.of(NOT_BOUNCE_SHORTCUT_DEFAULT_ID, NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID,
                    CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID, CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID,
                    ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID, ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID,
                    ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID);
    public static final Map<String, Long> SHORTCUT_DEFAULT_ID_BY_NAME = Map.of(
            NOT_BOUNCE_SHORTCUT_NAME, NOT_BOUNCE_SHORTCUT_DEFAULT_ID,
            NOT_BOUNCE_LAL_SHORTCUT_NAME, NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID,
            CAMPAIGN_GOALS_SHORTCUT_NAME, CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID,
            CAMPAIGN_GOALS_LAL_SHORTCUT_NAME, CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID,
            ECOM_PURCHASE_LAL_SHORTCUT_NAME, ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID,
            ECOM_ABANDONED_CART_SHORTCUT_NAME, ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID,
            ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_NAME, ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID
    );
    public static final Map<Long, String> SHORTCUT_NAME_BY_DEFAULT_ID = EntryStream.of(SHORTCUT_DEFAULT_ID_BY_NAME)
            .invert()
            .toMap();

    // Словарь сопоставлений старых и текущих названий одних и тех же шорткатов,
    // требуется для сравнения сохраненных и еще несохраненных шорткатов при возврате доступных условий ретаргетинга
    public static final Map<String, String> OBSOLETE_SHORTCUT_NAME_TO_CURRENT_NAME = Map.of(
            CAMPAIGN_GOALS_SHORTCUT_NAME_OBSOLETE, CAMPAIGN_GOALS_SHORTCUT_NAME,
            CAMPAIGN_GOALS_LAL_SHORTCUT_NAME_OBSOLETE, CAMPAIGN_GOALS_LAL_SHORTCUT_NAME
    );

    // У сегментов и lal-сегментов время всегда должно быть равно 540 - Goal.java:148
    private static final Map<Long, PpcPropertyName<Integer>> SHORTCUT_GOAL_TIME_PROPERTY_BY_DEFAULT_ID = Map.of(
            CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID, RETARGETING_CONDITION_CAMPAIGN_GOALS_SHORTCUT_GOAL_TIME_DAYS,
            CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID, RETARGETING_CONDITION_CAMPAIGN_GOALS_LAL_SHORTCUT_GOAL_TIME_DAYS
    );
    private static final Integer SHORTCUT_DEFAULT_GOAL_TIME = MAX_GOAL_TIME;

    private static final List<Rule> EMPTY_RULES = List.of();

    private final PpcPropertiesSupport ppcPropertiesSupport;

    public RetargetingConditionShortcutService(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    private RetargetingCondition getNotBounceShortcut(List<Goal> notBounceSegments) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(NOT_BOUNCE_SHORTCUT_DEFAULT_ID)
                .withName(NOT_BOUNCE_SHORTCUT_NAME)
                .withDescription(NOT_BOUNCE_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, notBounceSegments);

        return condition;
    }

    private RetargetingCondition getNotBounceLalShortcut(List<Goal> notBounceLalSegments) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(NOT_BOUNCE_LAL_SHORTCUT_DEFAULT_ID)
                .withName(NOT_BOUNCE_LAL_SHORTCUT_NAME)
                .withDescription(NOT_BOUNCE_LAL_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, notBounceLalSegments);

        return condition;
    }

    private RetargetingCondition getCampaignGoalsShortcut(List<Goal> campaignGoals) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(CAMPAIGN_GOALS_SHORTCUT_DEFAULT_ID)
                .withName(CAMPAIGN_GOALS_SHORTCUT_NAME)
                .withDescription(CAMPAIGN_GOALS_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, campaignGoals);

        return condition;
    }

    private RetargetingCondition getCampaignGoalsLalShortcut(List<Goal> campaignLalGoals) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(CAMPAIGN_GOALS_LAL_SHORTCUT_DEFAULT_ID)
                .withName(CAMPAIGN_GOALS_LAL_SHORTCUT_NAME)
                .withDescription(CAMPAIGN_GOALS_LAL_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, campaignLalGoals);

        return condition;
    }

    private RetargetingCondition getEcomPurchaseLalShortcut(List<Goal> ecomPurchaseLalSegments) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(ECOM_PURCHASE_LAL_SHORTCUT_DEFAULT_ID)
                .withName(ECOM_PURCHASE_LAL_SHORTCUT_NAME)
                .withDescription(ECOM_PURCHASE_LAL_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, ecomPurchaseLalSegments);

        return condition;
    }

    private RetargetingCondition getEcomAbandonedCartShortcut(List<Goal> ecomAbandonedCartSegments) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(ECOM_ABANDONED_CART_SHORTCUT_DEFAULT_ID)
                .withName(ECOM_ABANDONED_CART_SHORTCUT_NAME)
                .withDescription(ECOM_ABANDONED_CART_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, ecomAbandonedCartSegments);

        return condition;
    }

    private RetargetingCondition getEcomViewedWithoutPurchaseShortcut(List<Goal> ecomViewedWithoutPurchaseSegments) {
        var condition = (RetargetingCondition) new RetargetingCondition()
                .withId(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DEFAULT_ID)
                .withName(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_NAME)
                .withDescription(ECOM_VIEWED_WITHOUT_PURCHASE_SHORTCUT_DESCRIPTION)
                .withRules(EMPTY_RULES);
        fillInRules(condition, ecomViewedWithoutPurchaseSegments);

        return condition;
    }

    private void fillInRules(RetargetingCondition condition, List<Goal> goals) {
        if (!isEmpty(goals)) {
            if (goals.size() > MAX_GOALS_PER_RULE) {
                goals = goals.subList(0, MAX_GOALS_PER_RULE);
            }
            condition.withRules(List.of(new Rule()
                    .withType(RuleType.OR)
                    .withGoals(goals)));
        }
    }

    public List<RetargetingCondition> getAvailableTruncatedRetargetingConditionShortcuts(ClientId clientId,
                                                                                         boolean isNotBounceShortcutsAvailable,
                                                                                         boolean isCampaignGoalsAvailable,
                                                                                         boolean isEcomPurchaseSegmentsAvailable,
                                                                                         boolean isEcomAbandonedCartSegmentsAvailable,
                                                                                         boolean isEcomViewedWithoutPurchaseSegmentsAvailable) {
        var shortcuts = new ArrayList<RetargetingCondition>(RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS.size());
        // Порядок важен! В DIRECT-161989 решили первыми в списке доступных отдавать еком-шорткаты
        if (isEcomPurchaseSegmentsAvailable) {
            shortcuts.add(getEcomPurchaseLalShortcut(null));
        }

        if (isEcomAbandonedCartSegmentsAvailable) {
            shortcuts.add(getEcomAbandonedCartShortcut(null));
        }

        if (isEcomViewedWithoutPurchaseSegmentsAvailable) {
            shortcuts.add(getEcomViewedWithoutPurchaseShortcut(null));
        }

        if (isNotBounceShortcutsAvailable) {
            shortcuts.add(getNotBounceShortcut(null));
            shortcuts.add(getNotBounceLalShortcut(null));
        }

        if (isCampaignGoalsAvailable) {
            shortcuts.add(getCampaignGoalsShortcut(null));
            shortcuts.add(getCampaignGoalsLalShortcut(null));
        }

        return shortcuts.stream()
                .map(r -> fillInRetargetingConditionShortcut(r, clientId))
                .collect(Collectors.toList());
    }

    public Map<Long, RetargetingCondition> getRetargetingConditionShortcuts(ClientId clientId,
                                                                            @Nullable List<Goal> notBounceSegments,
                                                                            @Nullable List<Goal> notBounceLalSegments,
                                                                            @Nullable List<Goal> campaignGoals,
                                                                            @Nullable List<Goal> campaignLalGoals,
                                                                            @Nullable List<Goal> ecomPurchaseLalSegments,
                                                                            @Nullable List<Goal> ecomAbandonedCartSegments,
                                                                            @Nullable List<Goal> ecomViewedWithoutPurchaseSegments) {
        var shortcuts = new ArrayList<RetargetingCondition>(RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS.size());
        if (!isEmpty(notBounceSegments)) {
            shortcuts.add(getNotBounceShortcut(notBounceSegments));
        }
        if (!isEmpty(notBounceLalSegments)) {
            shortcuts.add(getNotBounceLalShortcut(notBounceLalSegments));
        }
        if (!isEmpty(campaignGoals)) {
            shortcuts.add(getCampaignGoalsShortcut(campaignGoals));
        }
        if (!isEmpty(campaignLalGoals)) {
            shortcuts.add(getCampaignGoalsLalShortcut(campaignLalGoals));
        }
        if (!isEmpty(ecomPurchaseLalSegments)) {
            shortcuts.add(getEcomPurchaseLalShortcut(ecomPurchaseLalSegments));
        }
        if (!isEmpty(ecomAbandonedCartSegments)) {
            shortcuts.add(getEcomAbandonedCartShortcut(ecomAbandonedCartSegments));
        }
        if (!isEmpty(ecomViewedWithoutPurchaseSegments)) {
            shortcuts.add(getEcomViewedWithoutPurchaseShortcut(ecomViewedWithoutPurchaseSegments));
        }

        return shortcuts.stream()
                .collect(Collectors.toMap(
                        RetargetingCondition::getId,
                        r -> (RetargetingCondition) fillInRetargetingConditionShortcut(r, clientId)
                                // удаляем дефолтный id; при создании пропишется новый
                                .withId(null)
                                .withLastChangeTime(LocalDateTime.now())));
    }

    private RetargetingCondition fillInRetargetingConditionShortcut(RetargetingCondition condition,
                                                                    ClientId clientId) {
        return (RetargetingCondition) condition
                .withClientId(clientId.asLong())
                .withType(ConditionType.shortcuts)
                .withAvailable(true)
                .withInterest(false)
                .withDeleted(false)
                .withAutoRetargeting(false);
    }

    public Integer calculateGoalTime(Long shortcutId) {
        PpcPropertyName<Integer> propertyName = SHORTCUT_GOAL_TIME_PROPERTY_BY_DEFAULT_ID.get(shortcutId);

        // TODO add cache
        Integer result = propertyName != null
                ? ppcPropertiesSupport.get(propertyName).getOrDefault(SHORTCUT_DEFAULT_GOAL_TIME)
                : SHORTCUT_DEFAULT_GOAL_TIME;

        return (result != null && result >= MIN_GOAL_TIME && result <= MAX_GOAL_TIME)
                ? result
                : SHORTCUT_DEFAULT_GOAL_TIME;
    }
}
