package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.helper.RetargetingConditionWithLalSegmentHelper;
import ru.yandex.direct.core.entity.retargeting.service.validation2.UpdateRetargetingConditionValidationService2;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.extractLalSegmentIds;
import static ru.yandex.direct.model.ModelChanges.isPropertyChanged;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновлений условий ретаргетинга
 */
public class RetargetingConditionsUpdateOperation extends SimpleAbstractUpdateOperation<RetargetingCondition, Long> {

    private final UpdateRetargetingConditionValidationService2 updateValidationService;
    private final RetargetingConditionRepository retConditionRepository;
    private final RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper;
    private final LalSegmentRepository lalSegmentRepository;
    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;
    private final boolean skipPixelValidation;
    private final boolean partOfComplexOperation;
    private final ClientId clientId;
    private final int shard;

    private Collection<AppliedChanges<RetargetingCondition>> appliedChanges;

    public RetargetingConditionsUpdateOperation(Applicability applicability,
                                                List<ModelChanges<RetargetingCondition>> modelChanges,
                                                UpdateRetargetingConditionValidationService2 updateValidationService,
                                                RetargetingConditionRepository retConditionRepository,
                                                RetargetingConditionWithLalSegmentHelper retConditionWithLalSegmentHelper,
                                                LalSegmentRepository lalSegmentRepository,
                                                RetargetingConditionOperationFactory retargetingConditionOperationFactory,
                                                boolean skipPixelValidation,
                                                boolean partOfComplexOperation,
                                                ClientId clientId,
                                                int shard) {
        super(applicability, modelChanges, id -> {
            RetargetingCondition retargetingCondition = new RetargetingCondition();
            retargetingCondition.setId(id);
            return retargetingCondition;
        });

        this.updateValidationService = updateValidationService;
        this.retConditionRepository = retConditionRepository;
        this.retConditionWithLalSegmentHelper = retConditionWithLalSegmentHelper;
        this.lalSegmentRepository = lalSegmentRepository;
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;

        this.skipPixelValidation = skipPixelValidation;
        this.partOfComplexOperation = partOfComplexOperation;
        this.clientId = clientId;
        this.shard = shard;
    }

    @Override
    protected ValidationResult<List<ModelChanges<RetargetingCondition>>, Defect> validateModelChanges(
            List<ModelChanges<RetargetingCondition>> modelChanges) {
        return updateValidationService.preValidate(modelChanges, clientId, shard);
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<RetargetingCondition> modelChangesValidatedStep) {
        prepareRulesForSaving(modelChangesValidatedStep);
    }

    /**
     * Подготовка правил условий ретаргетинга в объектах ModelChanges к сохранению.
     * Проводится после первого этапа валидации для объектов ModelChanges без ошибок.
     * Изменяет валидные ModelChanges.
     * 1. заполняет недостающее значение в поле unionWithId у lal-сегментов.
     */
    private void prepareRulesForSaving(ModelChangesValidatedStep<RetargetingCondition> modelChangesValidatedStep) {
        List<Rule> rules = modelChangesValidatedStep.getValidModelChanges().stream()
                .filter(isPropertyChanged(RetargetingCondition.RULES))
                .map(changes -> changes.getChangedProp(RetargetingCondition.RULES))
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
        retConditionWithLalSegmentHelper.updateLalSegmentsOnRules(rules);
    }

    @Override
    protected Collection<RetargetingCondition> getModels(Collection<Long> ids) {
        return retConditionRepository.getFromRetargetingConditionsTable(shard, clientId, ids);
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<RetargetingCondition> changesAppliedStep) {
        appliedChanges = changesAppliedStep.getAppliedChangesForValidModelChanges();
        appliedChanges.forEach(RetargetingConditionsUpdateOperation::updateRelatedProperties);
    }

    private static void updateRelatedProperties(AppliedChanges<RetargetingCondition> appliedChanges) {
        Set<ModelProperty<? super RetargetingCondition, ?>> actuallyChangedProps =
                appliedChanges.getActuallyChangedProps();
        if (!actuallyChangedProps.isEmpty()) {
            appliedChanges.modify(RetargetingCondition.LAST_CHANGE_TIME, LocalDateTime.now());
        }
    }

    @Override
    protected ValidationResult<List<RetargetingCondition>, Defect> validateAppliedChanges(
            ValidationResult<List<RetargetingCondition>, Defect> validationResult) {
        List<RetargetingCondition> retConditions = mapList(appliedChanges, AppliedChanges::getModel);

        boolean skipInterconnectionsWithAdsValidation = partOfComplexOperation;

        return updateValidationService
                .validateUpdateElements(validationResult, retConditions, appliedChanges, skipPixelValidation,
                        skipInterconnectionsWithAdsValidation, clientId, shard);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<RetargetingCondition>> applicableAppliedChanges) {
        retConditionRepository.update(shard, applicableAppliedChanges);
        var models = mapList(applicableAppliedChanges, AppliedChanges::getModel);
        lalSegmentRepository.activateLalSegments(extractLalSegmentIds(models));
        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<RetargetingCondition> executionStep) {
        Collection<AppliedChanges<RetargetingCondition>> appliedChanges =
                executionStep.getAppliedChangesForExecution();

        retargetingConditionOperationFactory.retConditionPostUpdate(appliedChanges, shard);
    }
}
