package ru.yandex.direct.core.entity.retargeting.service;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.retargeting.model.RetargetingCampaignInfo;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingDeleteRequest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

public class RetargetingDeleteOperation extends AbstractOperationWithId {

    private final RetargetingRepository retargetingRepository;
    private final RetargetingDeleteService retargetingDeleteService;
    private final DeleteRetargetingValidationService deleteRetargetingValidationService;

    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;

    private Map<Long, RetargetingCampaignInfo> retargetingsInfo;
    private Set<Long> validIds;

    public RetargetingDeleteOperation(Applicability applicability, List<Long> modelIds,
                                      RetargetingRepository retargetingRepository,
                                      RetargetingDeleteService retargetingDeleteService,
                                      DeleteRetargetingValidationService deleteRetargetingValidationService,
                                      long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(applicability, modelIds);

        this.retargetingRepository = retargetingRepository;
        this.retargetingDeleteService = retargetingDeleteService;
        this.deleteRetargetingValidationService = deleteRetargetingValidationService;

        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        retargetingsInfo = retargetingRepository.getRetargetingToCampaignMappingForDelete(shard, ids);

        ValidationResult<List<Long>, Defect> vr =
                deleteRetargetingValidationService.validate(ids, retargetingsInfo, operatorUid);
        if (!vr.hasErrors()) {
            validIds = new HashSet<>(getValidItems(vr));
        }
        return vr;
    }

    public Set<Long> getPotentiallyDeletedRetargetingIds() {
        checkState(isPrepared(), "operation must be prepared before calling this method");
        checkState(!isExecuted(), "must not be called after execution");
        checkState(getResult().isEmpty(), "must not be called on invalid input data");
        return validIds;
    }

    @Override
    protected void execute(List<Long> ids) {
        List<RetargetingDeleteRequest> deleteRequests = EntryStream.of(retargetingsInfo)
                .filterKeys(ids::contains)
                .values()
                .map(RetargetingUtils::convertToRetargetingDeleteRequest)
                .toList();
        retargetingDeleteService.delete(shard, UidAndClientId.of(clientUid, clientId), operatorUid, deleteRequests);
    }
}
