package ru.yandex.direct.core.entity.retargeting.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Sets;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingDeleteRequest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.dbutil.model.UidAndClientId;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent.deleteRetargetingEvent;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class RetargetingDeleteService {
    private final RetargetingRepository retargetingRepository;
    private final AdGroupRepository adGroupRepository;
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;

    public RetargetingDeleteService(RetargetingRepository retargetingRepository, AdGroupRepository adGroupRepository,
                                    LogPriceService logPriceService,
                                    MailNotificationEventService mailNotificationEventService) {
        this.retargetingRepository = retargetingRepository;
        this.adGroupRepository = adGroupRepository;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
    }

    /**
     * Удаление ретаргетингов. В метод нужно передавать валидные запросы на удаление, дополнительная валидация
     * не выполняется.
     */
    public void delete(int shard, UidAndClientId owner, long operatorUid,
                       List<RetargetingDeleteRequest> validDeleteRequests) {
        if (validDeleteRequests.isEmpty()) {
            return;
        }
        List<Long> idsToDelete = mapList(validDeleteRequests, RetargetingDeleteRequest::getRetargetingId);
        retargetingRepository.delete(shard, idsToDelete);

        Set<Long> affectedAdGroups = listToSet(validDeleteRequests, RetargetingDeleteRequest::getAdGroupId);
        adGroupRepository.actualizeAdGroupsOnChildModification(shard, affectedAdGroups);
        Set<Long> nonEmptyGroupIds = adGroupRepository.getAdGroupIdsWithConditions(shard, affectedAdGroups);
        Set<Long> emptyGroupIds = Sets.difference(affectedAdGroups, nonEmptyGroupIds);
        adGroupRepository.updateModerationStatusesAfterConditionsAreGone(shard, emptyGroupIds);
        logPriceChanges(validDeleteRequests, operatorUid);
        sendMailNotifications(shard, owner, operatorUid, validDeleteRequests);
    }

    /**
     * Логгирование изменения цен
     */
    private void logPriceChanges(Collection<RetargetingDeleteRequest> requests, long operatorUid) {
        List<LogPriceData> logPriceList = mapList(requests, RetargetingDeleteService::createLogPriceData);
        logPriceService.logPrice(logPriceList, operatorUid);
    }

    private static LogPriceData createLogPriceData(RetargetingDeleteRequest r) {
        return new LogPriceData(
                r.getCampaignId(),
                r.getAdGroupId(),
                r.getRetargetingId(),
                null,
                null,
                null,
                LogPriceData.OperationType.DELETE_2);
    }

    private void sendMailNotifications(int shard, UidAndClientId owner, long operatorUid,
                                       Collection<RetargetingDeleteRequest> retargetingDeleteRequests) {
        Set<Long> affectedAdGroupIds = listToSet(retargetingDeleteRequests, RetargetingDeleteRequest::getAdGroupId);
        Map<Long, AdGroupName> adGroupIdToAdGroup =
                adGroupRepository.getAdGroupNames(shard, owner.getClientId(), affectedAdGroupIds);
        checkState(affectedAdGroupIds.size() == adGroupIdToAdGroup.keySet().size(),
                "All ad groups of deleted retargetings must exists.");

        //noinspection rawtypes
        List<RetargetingEvent> events = new ArrayList<>();

        retargetingDeleteRequests.forEach(req -> {
            checkNotNull(req.getRetargetingConditionId(), "retargeting condition id must be not null");

            long adGroupId = req.getAdGroupId();
            long campaignId = adGroupIdToAdGroup.get(adGroupId).getCampaignId();
            String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();

            events.add(deleteRetargetingEvent(operatorUid, owner.getUid(), campaignId, adGroupId, adGroupName,
                    req.getRetargetingConditionId()));
        });

        mailNotificationEventService.queueEvents(operatorUid, owner.getClientId(), events);
    }

}
